/* --COPYRIGHT--,BSD_EX
 * Copyright (c) 2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *******************************************************************************
 *
 *                       MSP430 CODE EXAMPLE DISCLAIMER
 *
 * MSP430 code examples are self-contained low-level programs that typically
 * demonstrate a single peripheral function or device feature in a highly
 * concise manner. For this the code may rely on the device's power-on default
 * register values and settings such as the clock configuration and care must
 * be taken when combining code from several examples to avoid potential side
 * effects. Also see www.ti.com/grace for a GUI- and www.ti.com/msp430ware
 * for an API functional library-approach to peripheral configuration.
 *
 * --/COPYRIGHT--*/
//******************************************************************************
//  MSP430FR231x Demo: Low-side bidirectional current sensing circuit  + ADC10
//
//  Description: Configure SAC0 in general purpose mode to connect the
//  appropriate external circuitry, including the voltage divider and unity
//  feedback loop.
//  Configure TIA in general purpose mode in order to connect the
//  appropriate external shunt resistor circuitry. The output of TIA (TRIO) is
//  sampled by the on-board ADC (ADC10) and the status LED turns on if the
//  measured current is over 0A (1.65 V).
//  ACLK = n/a, MCLK = SMCLK = default DCODIV ~1MHz.
//
//                MSP430FR231x
//             -------------------
//         /|\|          P1.2/OA0O|--->Vref
//          | |          P1.3/OA0-|<---Vref
//          | |          P1.4/OA0+|<---Voltage Divider
//          --|RST                |
//            |          P1.5/TRIO|--->Vo
//            |          P1.6/TRI-|<---Vshunt-
//            |          P1.7/TRI+|<---Vshunt+
//            |                   |
//            |               P1.0|----> LED
//
//   ***PLEASE REFER TO THE REFERENCE DESIGN (SLAA917) FOR A DETAILED
//   ***SCHEMATIC SHOWING HOW TO CONNECT THE SHUNT RESISTOR AND OTHER
//   ***PASSIVE COMPONENTS TO THE MSP430 SMART ANALOG COMBOS
//
//   Mitch Ridgeway
//   Texas Instruments Inc.
//   November 2019
//   Built with IAR Embedded Workbench v7.12.1 & Code Composer Studio v9.0.1
//******************************************************************************
#include <msp430.h> 

unsigned int adcResult;

int main(void)
{
    WDTCTL = WDTPW | WDTHOLD;	// stop watchdog timer

    P1DIR |= BIT0;                            // Select P1.0 as output
    P1OUT &= ~BIT0;                           // Set P1.0 output low
    PM5CTL0 &= ~LOCKLPM5;                     // Disable the GPIO power-on default high-impedance mode
                                              // to activate previously configured port settings
	
    //Configure SAC-L1 for GP mode
    P1SEL0 |= BIT2 + BIT3 + BIT4;             // Select P1.2 P1.3 P1.4 OA function
    P1SEL1 |= BIT2 + BIT3 + BIT4;             // Select P1.2 P1.3 P1.4 OA function

    SAC0OA |= NMUXEN + PMUXEN;                // Enable negative and positive input
    SAC0OA |= OAPM;                           // Select low speed and low power mode
    SAC0OA |= SACEN + OAEN;                   // Enable SAC and OA


    //Configure TIA
    P1SEL0 |= BIT5 | BIT6 | BIT7;             // Select P1.5 P1.6 P1.7 TRI function
    P1SEL1 |= BIT5 | BIT6 | BIT7;

    TRI0CTL = TRIEN | TRIPM;                 // Enable TRI, Select low speed and low power mode

    // Configure ADC10
    ADCCTL0 &= ~ADCENC;                       // Disable ADC
    ADCCTL0 = ADCSHT_2 | ADCON;               // ADCON, S&H=16 ADC clks
    ADCCTL1 = ADCSHP;                         // ADCCLK = MODOSC; sampling timer
    ADCCTL2 = ADCRES;                         // 10-bit conversion results
    ADCIE = ADCIE0;                           // Enable ADC conv complete interrupt
    ADCMCTL0 = ADCINCH_5 | ADCSREF_0;         // A5 ADC input select = TRI output
                                              // Vref = DVCC

    while(1)
    {
        ADCCTL0 |= ADCENC | ADCSC;          // Sampling and conversion start
        __bis_SR_register(LPM0_bits | GIE); // Enter LPM0, ADC_ISR will force exit
        __no_operation();                   // For debug only

        if (adcResult > 512)                // OA output > 1/2 VCC
            P1OUT |= BIT0;                  // Set P1.0 LED on
        else
            P1OUT &= ~BIT0;                 // Clear P1.0 LED off
    }
}

// ADC interrupt service routine
#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector=ADC_VECTOR
__interrupt void ADC_ISR(void)
#elif defined(__GNUC__)
void __attribute__ ((interrupt(ADC_VECTOR))) ADC_ISR (void)
#else
#error Compiler not supported!
#endif
{
  switch(__even_in_range(ADCIV, ADCIV_ADCIFG))
  {
      case ADCIV_NONE:
          break;
      case ADCIV_ADCOVIFG:
          break;
      case ADCIV_ADCTOVIFG:
          break;
      case ADCIV_ADCHIIFG:
          break;
      case ADCIV_ADCLOIFG:
          break;
      case ADCIV_ADCINIFG:
          break;
      case ADCIV_ADCIFG:
          adcResult = ADCMEM0;            // Read ADC memory
          __bic_SR_register_on_exit(LPM0_bits);// Exit from LPM
          break;
      default:
          break;
  }
}
