/* --COPYRIGHT--,BSD_EX
 * Copyright (c) 2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *******************************************************************************
 *
 *                       MSP430 CODE EXAMPLE DISCLAIMER
 *
 * MSP430 code examples are self-contained low-level programs that typically
 * demonstrate a single peripheral function or device feature in a highly
 * concise manner. For this the code may rely on the device's power-on default
 * register values and settings such as the clock configuration and care must
 * be taken when combining code from several examples to avoid potential side
 * effects. Also see www.ti.com/grace for a GUI- and www.ti.com/msp430ware
 * for an API functional library-approach to peripheral configuration.
 *
 * --/COPYRIGHT--*/
//******************************************************************************
//  MSP430FR235x Demo: Low-side bidirectional current sensing circuit  + eCOMP0
//
//  Description: Configure SAC1 in DAC mode to provide the 1.65V reference
//  voltage. Configure SAC2 in general purpose mode in order to connect the
//  appropriate external circuitry.The output of SAC2 (OA2O) is sampled by the
//  on-board comparator (eCOMP0) and the status LED turns on if the measured
//  current is over 0A (1.65 V). The internal eCOMP DAC is used to compare
//  the output of SAC2 to 1.65V.
//  ACLK = n/a, MCLK = SMCLK = default DCODIV ~1MHz.
//
//                MSP430FR235x
//             -------------------
//         /|\|          P3.1/OA2O|--->Vo
//          | |          P3.2/OA2-|<---Vshunt-
//          | |          P3.3/OA2+|<---Vshunt+
//          --|RST                |
//            |          P1.5/OA1O|--->Vref
//            |          P1.6/OA1-|
//            |          P1.7/OA1+|
//            |                   |
//            |               P1.0|----> LED
//
//   ***PLEASE REFER TO THE REFERENCE DESIGN (SLAA917) FOR A DETAILED
//   ***SCHEMATIC SHOWING HOW TO CONNECT THE SHUNT RESISTOR AND OTHER
//   ***PASSIVE COMPONENTS TO THE MSP430 SMART ANALOG COMBOS
//
//   Mitch Ridgeway
//   Texas Instruments Inc.
//   November 2019
//   Built with IAR Embedded Workbench v7.12.1 & Code Composer Studio v9.0.1
//******************************************************************************
#include <msp430.h> 

int main(void)
{
    WDTCTL = WDTPW + WDTHOLD;                 // Stop watch dog timer

    P1DIR |= BIT0;                            // Select P1.0 as output
    P1OUT &= ~BIT0;                           // Set P1.0 output low

    PM5CTL0 &= ~LOCKLPM5;                     // Disable the GPIO power-on default
                                              // high-impedance mode
                                              // to activate previously configured
                                              // port settings

    // Configure reference module
    PMMCTL0_H = PMMPW_H;                      // Unlock the PMM registers
    PMMCTL2 = INTREFEN | REFVSEL_2;           // Enable internal 2.5V reference
    while(!(PMMCTL2 & REFGENRDY));            // Poll till internal reference settles


    //Configure SAC1 in DAC buffer mode to provide Vref (1.65V)
    P1SEL0 |= BIT5;                           // Select P1.5(OA1O) as OA function
    P1SEL1 |= BIT5;                           // Select P1.5(OA1O) as OA function
    SAC1DAT = 0xA8F;                          // Set SAC1 DAC output = 1.65V
    SAC1DAC |= DACSREF | DACEN;               // Select internal shared reference and
                                              // Enable SAC1 DAC
    SAC1OA |= NMUXEN + PMUXEN + NSEL_1 + PSEL_1; // Enable SAC1 negative and
                                                 // positive input in DAC mode
    SAC1OA |= OAPM;                           // Select low speed and low power mode
    SAC1OA |= SACEN + OAEN;                   // Enable SAC1 and OA


    //Configure SAC2 as general purpose op amp for shunt resistor voltage
    P3SEL0 |= BIT1 + BIT2 + BIT3;             // Select P3.1(OA2O), P3.2(OA2-),
                                              // and P3.3(OA2+) as OA function
    P3SEL1 |= BIT1 + BIT2 + BIT3;             // Select P3.1(OA2O), P3.2(OA2-),
                                              // and P3.3(OA2+) as OA function
    SAC2OA |= NMUXEN + PMUXEN;                // Enable SAC2 negative and positive
                                              // input
    SAC2OA |= OAPM;                           // Select low speed and low power mode
    SAC2OA |= SACEN + OAEN;                   // Enable SAC2 and OA

    //Setup eCOMP
    CP0CTL0 = CPPSEL_6 | CPNSEL_5;             // Select internal eCOMP DAC and
                                               // SAC2 output as eCOMP0 inputs
    CP0CTL0 |= CPPEN | CPNEN;                  // Enable eCOMP input
    CP0CTL1 |= CPIIE | CPIE | CPHSEL_3;        // Enable eCOMP hystersis and
                                               // dual edge interrupt
    CP0DACCTL |= CPDACREFS | CPDACEN;          // Select on-chip VREF and enable DAC
    CP0DACDATA |= 0x002A;                      // CPDACBUF1=On-chip VREF*42/64 ~ 1.65V
    CP0CTL1 |= CPEN | CPMSEL;                  // Turn on eCOMP, in low power mode

    __bis_SR_register(LPM3_bits | GIE);        // Enter LPM3
    __no_operation();                          // For debug only
}

// eCOMP interrupt service routine
#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector = ECOMP0_ECOMP1_VECTOR
__interrupt void ECOMP0_ISR(void)
#elif defined(__GNUC__)
void __attribute__ ((interrupt(ECOMP0_ECOMP1_VECTOR))) ECOMP0_ISR (void)
#else
#error Compiler not supported!
#endif
{
    switch(__even_in_range(CP0IV, CPIV__CPIIFG))
    {
        case CPIV__NONE:
            break;
        case CPIV__CPIFG:
            P1OUT &= ~BIT0;
            break;
        case CPIV__CPIIFG:
            P1OUT |= BIT0;
            break;
        default:
            break;
    }
}
