//*******************************************************************************
//  Ultra Low Power Thermostat Using MSP430F4794
//
//  Description: This code demonstrates the use of a 4794 in a Thermostat application.
//  The SD16 samples the battery voltage and thermistor once every 6 seconds using
//  the SD16 GROUP feature. Temperature set-point is always displayed and is
//  changed with UP/DOWN button presses. The time is displayed for 3 seconds followed
//  by the room temperature which is also displayed for 3 seconds. Battery voltage
//  can be seen by pressing the BATT_VOLTAGE button.
//
//  Built with IAR KiskStart Embedded Workbench for MSP430 V4.09A
//******************************************************************************

#include "include.h"

#define UP    BIT0
#define DOWN  BIT1
#define HOLD  BIT2
#define RUN   BIT3
#define PRGM  BIT4
#define TIME  BIT5

#define BATT_VOLTAGE  BIT1
#define CP_SET        BIT1

int k = 0;
char count = 0;
char backlight_on;
char mode;
char port_number;

volatile unsigned int ADCresult;
volatile unsigned int V_cc;

volatile unsigned int j;
volatile unsigned int dig = 0;
int set_point = 70;

float batt_voltage_dec;
int batt_voltage;
int cp_setting;


void main(void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop watchdog timer
  FLL_CTL0 |= XCAP14PF;                     // Configure load caps
  Init_Ports();
  Init_LCD();
  Init_ADC();

  // Init Basic Timer for SW RTC
  BTCTL = BT_fCLK2_DIV128 + BTDIV;          // Source from ACLK, 1 sec interrupt


  IE2 |= BTIE;                              // Enable Basic Timer interrupt

  //Init TimerA for LCD backlight timer
  TACCTL0 = CCIE;                           // TACCR0 interrupt enabled
  TACTL = TASSEL_1 + ID0 + ID1;             // Source from ACLK. Div by 8
  TACCR0 = 15000;                           // continuous mode


 // Main program loop

  while(1)
  {

    count++;                                // Main loop counter (Temp/Time)
    _BIS_SR(LPM3_bits+GIE);                 // Enter LPM3 w/ interrupts
    RTC_Tick(0);

    if (count == 3)                         // Display time for 3 seconds
    {
      mode = 0;
      P1IE |= UP+DOWN;                      //Enable UP/DOWN button interrupts
      refresh_time();
      update_set_point(set_point);
    }

    else if (count == 6)                    // Sample temp and display for 3 seconds
    {

     // Sample Temperature & Batt Voltage
     SD16CTL |= SD16REFON + SD16VMIDON;     // Turn on SD16 Ref and Ref buffer

     P2OUT |= 0x01;                         // Disable Snooze
     for (j = 0; j < 100; j++);             // Delay for 1.2V ref startup

     SD16CCTL1 |= SD16SC;                   // Set bit to start conversion on Ch1 (Batt V) and Ch0 (Temp)

     __bis_SR_register(LPM0_bits);          // Enter LPM0/get voltage

     SD16CTL &= ~(SD16REFON+SD16VMIDON);    // Turn off SD16 Ref and Ref buffer to save current
     P2OUT &= ~0x01;                        // Enable Snooze
     count = 0;                             // Reset main loop counter
     update_temp(ADCresult);                // Display result on LCD
     update_set_point(set_point);



    }
  }
}

void debounce ()
{
  volatile unsigned int i;                  // Debounce
  for (i=0;i<3000;i++);
}


#pragma vector=SD16A_VECTOR
__interrupt void SD16AISR(void)
{
  ADCresult = SD16MEM0;
  V_cc = SD16MEM1;
  __bic_SR_register_on_exit(CPUOFF);        // Exit LPM0
}


// Basic Timer Interrupt Service Routine
#pragma vector=BASICTIMER_VECTOR
__interrupt void basic_timer_ISR(void)
{
  _BIC_SR_IRQ(LPM3_bits);                   // Clear LPM3 bits from 0(SR)
}


// Timer A0 interrupt service routine
#pragma vector=TIMERA0_VECTOR
__interrupt void Timer_A (void)
{
  P3OUT &= ~BIT4;                           // Turn off Backlight
  TACTL &= ~MC_1;                           // Stop Backlight Timer
  TAR = 0;
  TACTL &= ~TAIFG;                          // Clear TimerA flag
}

// Port 1 Interrupt Service Routine
#pragma vector=PORT1_VECTOR
__interrupt void Port1_ISR(void)
{
  P1IE &= ~0xFF;                            //disable port interrupts
  debounce();
  P3OUT |= BIT4;                            // Turn on backlight
  TACTL |=MC_1;                             //Start Backlight Timer

// Determine which button was pressed. If displaying batt voltage don't care about UP/DOWN
  if(!mode)
  {
    if (~P1IN & UP)
    {
      if (set_point > 89)                     // Don't exceed 90 deg F
        set_point = 90;
      else
        set_point++;

      update_set_point(set_point);

    }

    if (~P1IN & DOWN)
    {
      if (set_point < 61)                    // Don't exceed 60 deg F
        set_point = 60;
      else
        set_point--;

      update_set_point(set_point);

    }
  }

  if (~P1IN & TIME)
  {
  }

  if (~P1IN & PRGM)
  {
    P1IE &= ~(UP+DOWN);
    mode = BATT_VOLTAGE;
    // Calculate batt voltage. Use 0x7C90 as 0V offset. Calibrated using a known voltage
    batt_voltage_dec = (V_cc - 0x7C90)* .0018 * 11;
    batt_voltage = batt_voltage_dec;
    update_batt_voltage(batt_voltage);
    count = 0;
  }
  if (~P1IN & RUN)
  {
    P1IE &= ~(UP+DOWN);
    LCDM13 &= ~0x10;                        // clear semicolon
    mode = CP_SET;
    cp_setting++;

    if (cp_setting == 0x0F+1)
    {
      cp_setting = 0;
      LCDAVCTL0 &= ~LCDCPEN;
    }

    if(cp_setting)
      LCDAVCTL0 |= LCDCPEN;

    LCDAVCTL1 = cp_setting<<1;
    display_cp_setting(cp_setting);

    count = 0;
  }





// Don't want to be able to modify batt voltage with up/down button presses
  if (mode)
    P1IE = 0xFC;
  else
    P1IE = 0xFF;

  P1IFG &= ~0xFF;                           // All P1IFG's Cleared



}
