#include "include.h"

#define TEMP_OFFSET 3

// LCD segment definitions.
#define MIN_LO_L LCDM15
#define MIN_LO_H LCDM16

#define MIN_HI_L LCDM17
#define MIN_HI_H LCDM18

#define HOUR_LO_L LCDM19
#define HOUR_LO_H LCDM20

#define HOUR_HI LCDM14

#define SET_HI_H LCDM11
#define SET_HI_L LCDM10
#define SET_LO_H LCDM9
#define SET_LO_L LCDM8


// Temperature table: Resistance values for 99 deg. F down to 32 deg. F
const int RES_TAB[69] = {5860,5995,6130,6265,6400,6535,6693,6864,7035,
                         7206,7376,7547,7718,7888,8059,8261,8478,8696,
                         8913,9130,9348,9565,9783,10000,10259,10538,
                         10817,11096,11374,11653,11932,12211,12490,
                         12824,13183,13543,13902,14262,14621,14981,
                         15340,15700,16134,16601,17068,17535,18002,
                         18469,18936,19403,19870,20439,21052,21664,
                         22277,22889,23502,24115,24727,25340,26091,
                         26900,27708,28517,29325,30134,30943,31751,
                         32560};

// S (a,b,c,g)
#define c 0x01
#define b 0x02
#define g 0x10
#define a 0x20

const char char_gen_L[] = {                 // definitions for digits
  a+b+c,                                    // Displays "0"
  b+c,                                      // Displays "1"
  a+b+g,                                    // Displays "2"
  a+b+c+g,                                  // Displays "3"
  b+c+g,                                    // Displays "4"
  a+c+g,                                    // Displays "5"
  a+c+g,                                    // Displays "6"
  a+b+c,                                    // Displays "7"
  a+b+c+g,                                  // Displays "8"
  a+b+c+g,                                  // Displays "9"
  a+g,                                      // Displays "F"
  a,                                        // Displays "C"
  a+b+g,                                        // Displays "P"
  0,                                        // Blank
  0,                                        // Blank
  0                                         // Blank
};

//S+1 (d,e,f)
#define f 0x01
#define d 0x10
#define e 0x20

const char char_gen_H[] = {                 // definitions for digits
  d+e+f,                                    // Displays "0"
  0,                                        // Displays "1"
  d+e,                                      // Displays "2"
  d,                                        // Displays "3"
  f,                                        // Displays "4"
  d+f,                                      // Displays "5"
  e+d+f,                                    // Displays "6"
  0,                                        // Displays "7"
  d+e+f,                                    // Displays "8"
  f,                                        // Displays "9"
  f+e,                                      // Displays "F"
  f+e+d,                                    // Displays "C"
  f+e,                                        // Displays "P"
  0,                                        // Blank
  0,                                        // Blank
  0                                         // Blank
};


// Vars for RTC
static unsigned char Secs = 0x00;           // RTC, default time 12:00:00
static unsigned char Mins = 0x00;
static unsigned char Hrs = 0x12;
char half_day_count = 1;
char day;
char AmPm = 0x10;                           // Initialize in "PM" state

// Vars for Temperatures
char DegF_H;
char DegF_L;
unsigned int DegF;
unsigned int set_point_bcd;
float Thermistor_V;
long int Thermistor_R;

// Vars for Resistance table search
unsigned int test;
unsigned int tens;
unsigned int ones;
int tempFOnes;
int tempFTens;
int tempF;
char Temp_Range;

int batt_voltage_bcd;
int cp_setting_bcd;

volatile unsigned int i;
//volatile unsigned int DegF;



// Separate hi byte and low byte from RTC hour, minute, and seconds register words
// Use them to pick out LCD-readable numbers from array. Only update minutes, hour,
// AM/PM, and day when necessary

void RTC_Tick(unsigned int TickMins)
{

  // Always update time every second.
  Secs = __bcd_add_short(Secs, 1);
  if ((Secs == 0x60) || TickMins)
  {
    Secs = 0x00;
    Mins = __bcd_add_short(Mins, 1);

    MIN_LO_H &= BIT1;
    MIN_LO_H |= char_gen_H[Mins & 0x0F];
    MIN_LO_L = char_gen_L[Mins & 0x0F];

    MIN_HI_H &= BIT1;
    MIN_HI_H |= char_gen_H[Mins>>4]|0x80;
    MIN_HI_L = char_gen_L[Mins>>4]|0x80;

                                         // update the hours
    HOUR_LO_H &= BIT1;
    HOUR_LO_H |= char_gen_H[Hrs & 0x0F];
    HOUR_LO_L = char_gen_L[Hrs & 0x0F];

    if (Hrs>>4 == 0)
    {
      HOUR_HI &= BIT1;
    }
    else
    {
      HOUR_HI &= BIT1;
      HOUR_HI |=0x20;
    }

    LCDM12 |= AmPm;                         // Restore AmPm

    LCDM13 &= BIT1;                         // Restore semicolon
    LCDM13 |= 0x10;




    if (Mins == 0x60)
    {
      Mins = 0x00;

      MIN_HI_H &= BIT1;
      MIN_HI_H |= char_gen_H[0];
      MIN_HI_L = char_gen_L[0];

      Hrs = __bcd_add_short(Hrs, 1);

      HOUR_LO_H &= BIT1;
      HOUR_LO_H |= char_gen_H[Hrs & 0x0F];
      HOUR_LO_L = char_gen_L[Hrs & 0x0F];

      if (Hrs>>4 == 0)
      {
        HOUR_HI &= BIT1;
      }
      else
      {
        HOUR_HI &= BIT1;
        HOUR_HI |=0x20;
      }

      if (Hrs == 0x12)                      // Handle AM/PM and Day increment
      {
        if (half_day_count == 1)
        {
          AmPm = 0x20;
          LCDM12 &= ~BIT4;                  //aM
          LCDM12 |= BIT5;
          day++;
          if (day == 7)
          {
            day = 0;
          }
          update_day(day);
        }
        if (half_day_count == 2)
        {
          AmPm = 0x10;
          LCDM12 &= ~BIT5;
          LCDM12 |= BIT4;                   //pM
          half_day_count = 0;
        }
        half_day_count++;
      }
      if (Hrs == 0x13)
      {
        Hrs = 0x01;
        HOUR_LO_H &= BIT1;
        HOUR_LO_H |= char_gen_H[Hrs & 0x0F];
        HOUR_LO_L = char_gen_L[Hrs & 0x0F];
        if (Hrs>>4 == 0)
          LCDM14 &= BIT1;
        else
        {
        LCDM14 &= BIT1;
        LCDM14 |=0x20;
        }
      }
    }
  }
}


// After displaying the temperature it is necessary to refresh the time on the
// LCD
void refresh_time (void)
{
    LCDM12 |= AmPm;   //restore Am/Pm after dispaying temp

    MIN_LO_H &= BIT1;                       // Clear but don't touch the day
    MIN_LO_H |= char_gen_H[Mins & 0x0F];
    MIN_LO_L = char_gen_L[Mins & 0x0F];

    MIN_HI_H &= BIT1;
    MIN_HI_H |= (char_gen_H[Mins>>4]|0x80);
    MIN_HI_L = char_gen_L[Mins>>4]|0x80;

    HOUR_LO_H &= BIT1;
    HOUR_LO_H |= char_gen_H[Hrs & 0x0F];
    HOUR_LO_L = char_gen_L[Hrs & 0x0F];

    if (Hrs>>4 == 0)
      HOUR_HI &= BIT1;
    else
    {
      HOUR_HI &= BIT1;
      HOUR_HI |= 0x20;
    }
      LCDM13 &= BIT1;
      LCDM13 |= 0x10;

}


void update_temp (unsigned int ADCresult)
{

  //AmPm = LCDM12;
  LCDM12 &=~AmPm;    //Save AmPm state

  // Calculate voltage across the Thermistor
  Thermistor_V = (ADCresult-0x8000)*.000018;

  //Calculate Thermistor Resistance using  series voltage divider (47k+R_therm)
  Thermistor_R = (-47000*Thermistor_V)/(Thermistor_V - 1.2);

  // Convert resistance values to deg. F by looking up in table
  test = 0;
  tens = 0;
  ones = 0;

  if ((Thermistor_R < RES_TAB[test]) || (Thermistor_R == RES_TAB[test]))
    Temp_Range = 2;                         // Over 99F
  while (Thermistor_R > RES_TAB[test]) {    // Obtaining thermistor temperature
    test++;                                 // reading from table...
    if (test > 68)
    {
      Temp_Range = 1;                       // Under 32F
      break;
    }
    Temp_Range = 0;                         // Temp in range
    if (ones == 0)
    {
      ones = 9;
      tens = tens - 1;
      if (tens > 9)
        tens = 9;
    }
    else
    {
      ones = ones - 1;
    }
  }

  tempFOnes = ones;
  tempFTens = tens;

  // Thermistor temperature in deg F
  tempF = (10*tempFTens)+tempFOnes - TEMP_OFFSET;

  // Take integer tempF and convert to packaged BCD (DegF) display on LCD
  for (i = 16, DegF = 0; i; i--)
  {
    DegF = __bcd_add_short(DegF, DegF);
    if (tempF & 0x8000)
      DegF = __bcd_add_short(DegF, 1);
    tempF <<= 1;
  }

  // Display Temperature on LCD
  DegF_H = (DegF&0xF0) >> 4;
  DegF_L = DegF & 0x0F;

  MIN_LO_H &= BIT1;
  MIN_LO_H |= char_gen_H[DegF_L];
  MIN_LO_L = char_gen_L[DegF_L];

  MIN_HI_H &= BIT1;
  MIN_HI_H |= char_gen_H[DegF_H]|0x80;
  MIN_HI_L = char_gen_L[DegF_H]|0x80;

  HOUR_LO_H &= BIT1;
  HOUR_LO_L = 0x00;
  HOUR_HI &=BIT1;
  LCDM13 &= ~0x10;
}

void update_batt_voltage (batt_voltage)
{
  //AmPm = LCDM12;
  LCDM12 &=~AmPm;                           //Clear AM/PM

  //Take batt_voltage and convert to bcd (batt_voltage_bcd) for display on LCD
  for (i = 16, batt_voltage_bcd = 0; i; i--)
  {
    batt_voltage_bcd = __bcd_add_short(batt_voltage_bcd, batt_voltage_bcd);
    if (batt_voltage & 0x8000)
     batt_voltage_bcd = __bcd_add_short(batt_voltage_bcd, 1);
    batt_voltage <<= 1;
  }

  MIN_HI_H &= BIT1;
  MIN_HI_L = 0;

  HOUR_LO_H &= BIT1;
  HOUR_LO_L = 0;

  HOUR_HI &= BIT1;


  LCDM13 &= ~0x10; // clear semicolon

  SET_LO_H &= BIT1;
  SET_LO_H |= char_gen_H[batt_voltage_bcd & 0x0F];
  SET_LO_L = char_gen_L[batt_voltage_bcd & 0x0F];

  SET_HI_H &= BIT1;
  SET_HI_H |= char_gen_H[(batt_voltage_bcd&0x00F0)>>4];
  SET_HI_L = char_gen_L[(batt_voltage_bcd&0x00F0)>>4];

  MIN_LO_H &= BIT1;
  MIN_LO_H |= char_gen_H[(batt_voltage_bcd&0x0F00)>>8];
  MIN_LO_L = char_gen_L[(batt_voltage_bcd&0x0F00)>>8];


}

void update_set_point (int set_point)
{
  //Take setpoint and convert to bcd (set_point_bcd) for display on LCD
  for (i = 16, set_point_bcd = 0; i; i--)
  {
    set_point_bcd = __bcd_add_short(set_point_bcd, set_point_bcd);
    if (set_point & 0x8000)
      set_point_bcd = __bcd_add_short(set_point_bcd, 1);
    set_point <<= 1;
  }



  SET_HI_H &= BIT1;
  SET_HI_H |= char_gen_H[set_point_bcd>>4];
  SET_HI_L = char_gen_L[set_point_bcd>>4];

  SET_LO_H &= BIT1;
  SET_LO_H |= char_gen_H[set_point_bcd & 0x0F];
  SET_LO_L = char_gen_L[set_point_bcd & 0x0F];
}

void display_cp_setting (int cp_setting)
{
  LCDM12 &=~AmPm;                           //Clear AM/PM

  HOUR_LO_H &= BIT1;                        //"O"
  HOUR_LO_H = 0;
  HOUR_LO_L = 0;

  HOUR_HI &= BIT1;


  if (cp_setting == 0)
  {

    SET_HI_H &= BIT1;
    SET_HI_L = 0;
    SET_LO_H &= BIT1;
    SET_LO_L = 0;
    //Display "OFF"

    HOUR_LO_H &= BIT1;                      //"O"
    HOUR_LO_H |= char_gen_H[0];
    HOUR_LO_L = char_gen_L[0];

    MIN_HI_H &= BIT1;                       //"F"
    MIN_HI_H |= char_gen_H[10];
    MIN_HI_L = char_gen_L[10];

    MIN_LO_H &= BIT1;                       //"F"
    MIN_LO_H |= char_gen_H[10];
    MIN_LO_L = char_gen_L[10];

  }
  else
  {
  //Take cp_setting and convert to bcd (cp_setting_bcd) for display on LCD

  for (i = 16, cp_setting_bcd = 0; i; i--)
  {
    cp_setting_bcd = __bcd_add_short(cp_setting_bcd, cp_setting_bcd);
    if (cp_setting & 0x8000)
     cp_setting_bcd = __bcd_add_short(cp_setting_bcd, 1);
    cp_setting <<= 1;
  }

  LCDM13 &= ~0x10; // clear semicolon

  // Display CP setting
  SET_LO_H &= BIT1;
  SET_LO_H |= char_gen_H[cp_setting_bcd & 0x0F];
  SET_LO_L = char_gen_L[cp_setting_bcd & 0x0F];

  SET_HI_H &= BIT1;
  SET_HI_H |= char_gen_H[(cp_setting_bcd&0x00F0)>>4];
  SET_HI_L = char_gen_L[(cp_setting_bcd&0x00F0)>>4];

  // Display "CP"
  MIN_HI_H &= BIT1;                         // "C"
  MIN_HI_H |= char_gen_H[11];
  MIN_HI_L = char_gen_L[11];

  MIN_LO_H &= BIT1;                         // "P"
  MIN_LO_H |= char_gen_H[12];
  MIN_LO_L = char_gen_L[12];
  }



}




void Init_Ports (void)
{
  P1OUT = 0;                                // All P1.x reset
  P1IE |= BIT0+BIT1+BIT3+BIT4+BIT5;         // P1.0 Interrupts enabled BIT0 - BIT5
  P1IES|= BIT0+BIT1+BIT3+BIT4+BIT5;         // P1.0, P1.1 hi/low edge BIT0- BIT5
  P1DIR = 0xC0;                             // P1.0/1 = input (switches)

  P2OUT = 0;                                // All P2.x reset
  P2DIR = 0xFF;                             // All P2.x outputs
  P3OUT = 0;                                // All P3.x reset (3.5 = buzzer)
  P3DIR = 0xFF;                             // I2C pins = inp
  P4OUT = 0;                                // All P4.x reset
  P4DIR = 0xFF;                             // All P4.x outputs

  P5DIR = 0xFF;                             // All P5.x outputs
  P5SEL = BIT2;                             // Enable LCDA COM1
  P5OUT = 0x00;                             // All P5.x reset

  P7OUT = 0;                                // All P7.x reset
  P7DIR = 0xFF;                             // All P7.x outputs
  P8OUT = 0;                                // All P8.x reset
  P8DIR = 0xFF;                             // All P8.x outputs
  P9OUT = 0;                                // All P9.x reset
  P9DIR = 0xFF;                             // All P9.x outputs
  P10OUT = 0;                               // All P10.x reset
  P10DIR = 0xFF;                            // All P10.x outputs
}


void Init_ADC (void)
{
  SD16CTL = SD16SSEL0;                      // SMCLK
  SD16CCTL0 |= SD16SNGL+SD16GRP;            // Single conv, group with ch1 enable interrupt
  SD16INCTL0 |= SD16INTDLY0;                // Interrupt on 3rd sample

  SD16CCTL1 |= SD16SNGL+SD16IE;             // Single conv, enable interrupt
  SD16INCTL1 |= SD16INCH_5 + SD16INTDLY0;   // Set Ch3 to measure Batt voltage
}


void Init_LCD (void)
{
  // Turn on LCD f/256, 2 mux, all segments on
  LCDACTL = LCDON + LCDSON + LCDMX0 + LCDFREQ2 + LCDFREQ0;

  // Set all multiplexed segment pins to LCD functions. Select all internal
  // voltage supply and reference scheme
  LCDAPCTL0 = LCDS12+LCDS16+LCDS20+LCDS24+LCDS28;
  LCDAPCTL1 = LCDS32+LCDS36;
  LCDAVCTL0 =0;

  //Clear LCD Display
  LCDM8 = 0x00;
  LCDM9 = 0x00;
  LCDM10 = 0x00;
  LCDM11 = 0x00;
  LCDM12 = 0x00;
  LCDM13 = 0x00;
  LCDM14 = 0x00;
  LCDM15 = 0x00;
  LCDM16 = 0x00;
  LCDM17 = 0x00;
  LCDM18 = 0x00;
  LCDM19 = 0x00;
  LCDM20 = 0x00;

  // Initialize LCD to Display 12:00 pm
  MIN_LO_L = char_gen_L[0];
  MIN_LO_H = char_gen_H[0];
  MIN_HI_L = char_gen_L[0];
  MIN_HI_H = char_gen_H[0];
  HOUR_LO_L = char_gen_L[2];
  HOUR_LO_H = char_gen_H[2];
  HOUR_HI = 0x20;
  LCDM13 = 0x10;                            // colon on
  LCDM12 = AmPm;                            //PM

  update_day(day);

}

void update_day (char day)
{
  if (day == 0)
  {
    LCDM11 &= ~BIT1;                        // clear Sat. Display sunday
    LCDM9 |= BIT1;
  }
  if (day == 1)
  {
    LCDM9 &= ~BIT1;                         //Clear Sun. Display Mon
    LCDM14 |= BIT1;
  }
  if (day == 2)
  {
    LCDM14 &= ~BIT1;
    LCDM20 |= BIT1;
  }
  if (day == 3)
  {
    LCDM20 &= ~BIT1;
    LCDM18 |= BIT1;
  }
  if (day == 4)
  {
    LCDM18 &= ~BIT1;
    LCDM16 |= BIT1;
  }
  if (day == 5)
  {
    LCDM16 &= ~BIT1;
    LCDM13 |= BIT1;
  }
  if (day == 6)
  {
    LCDM13 &= ~BIT1;
    LCDM11 |= BIT1;
  }
}

//------------------------------------------------------------------------------
// Decrement RTC counter variables
//
// TickMins == 0 --> decrement seconds
// TickMins == 1 --> decrement minutes
//------------------------------------------------------------------------------
void RTC_Dec(unsigned int TickMins)
{
  Secs = __bcd_add_short(Secs, 0x9999);
  if ((Secs == 0x99) || TickMins)
  {
    Secs = TickMins ? 0x00 : 0x59;
    Mins = __bcd_add_short(Mins, 0x9999);
    if (Mins == 0x99)
    {
      Mins = 0x59;
      Hrs = __bcd_add_short(Hrs, 0x9999);
      if (Hrs == 0x00)
        Hrs = 0x12;
    }
  }
}
