/* --COPYRIGHT--,BSD
 * Copyright (c) 2015, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
/*******************************************************************************
 *
 * IPE_FR59xx.c
 *
 * Demonstrates a framework for using IP Encapsulation (IPE) on MSP430FR5xx/6xx
 * devices. Shows how to place code in the IPE segments so that IPE setup can
 * be performed by the tool built-in to CCS. It also demonstrates some IPE
 * best practices, like keeping all variables used by the IPE code also inside
 * the IPE area, and clearing all hardware registers used by the IPE code before
 * exiting the IPE area. Several different types of data and code are stored to
 * demonstrate different potential uses (variables, constants, functions).
 *
 * This code is meant to run on the MSP-EXP430FR5969 launchpad, but could be
 * adapted to run on other boards or devices that have IPE capabilities.
 *
 * The main code simply toggles a green LED roughly once per second (once for
 * each iteration of the while(1) loop, which includes a delay of 500k cycles).
 * The main while(1) loop also calls IPE_encapsulatedBlink() from the IPE area
 * once per iteration. The IPE_encapsulatedBlink() function toggles the red LED
 * quickly, a number of times corresponding to the current value of
 * IPE_encapsulatedCount stored in the IPE area. IPE_encapsulatedCount is
 * incremented after each call to IPE_encapsulatedBlink, so there is one
 * additional blink each time, until there are 5 red LED blinks at which point
 * it resets to 1.
 *
 * Make sure to use the IPE_FR59xx_load Debug configuration for loading new
 * software. This is needed to erase the IPE area. To test the encapsulation
 * preventing readout, either use the "Hard reset" option while in the debug
 * mode, or power-cycle the device and then use the IPE_FR59xx_test Debug
 * configuration to debug without re-loading the code. All variables and
 * addresses inside the IPE area should read 0x3FFF when encapsulation has
 * taken effect.
 *
 * NOTE: The first time loading a new project after having this code present
 * on the device with the IPE enabled, in the new project Debug settings select
 * "On connect, erase main, information, and IP protected area" - otherwise a
 * verification error will occur because the project will not be able to write
 * to the previously protected area of memory.
 *
 * The IPE area contains:
 * ----------------------
 * variables: IPE_i, IPE_encapsulatedCount
 * constants: IPE_encapsulatedKeys()
 * functions: IPE_encapsulatedInit(), IPE_encapsulatedBlink()
 * IPE structure is also included, but this is done automatically by the IPE
 * tool built into CCS.
 *
 * K. Pier
 * Texas Instruments Inc.
 * November 2015
 * Built with CCS v6.1.1 and IAR v6.40.1
 *
 ******************************************************************************/
#include <msp430.h> 
#include <stdint.h>

/*
 * IPE_encapsulatedCount is a variable inside the IPE area.
 * It is used by the IPE_encapsulatedBlink() function to track the number of
 * times to toggle the red LED
 */
#pragma DATA_SECTION(IPE_encapsulatedCount, ".ipe_vars")
uint8_t IPE_encapsulatedCount;

/*
 * IPE_i is a variable inside the IPE area.
 * It is used by the IPE_encapsulatedBlink() function as a simple loop counter.
 * This is used instead of a local variable, to keep it from being in RAM
 * where it would be accessible
 */
#pragma DATA_SECTION(IPE_i, ".ipe_vars")
uint8_t IPE_i;

/*
 * IPE_encapsulatedKeys is a constant inside the IPE area.
 * It demonstrates how to store keys and other consts inside the IPE area.
 */
#pragma DATA_SECTION(IPE_encapsulatedKeys, ".ipe_const")
const uint16_t IPE_encapsulatedKeys[] = {0x0123, 0x4567, 0x89AB, 0xCDEF,
										 0xAAAA, 0xBBBB, 0xCCCC, 0xDDDD};

/*
 * otherFramVar demonstrates how to store a variable in FRAM but outside the IPE
 * area.
 * It is used simply to represent any other FRAM variables in a real application
 */
#pragma PERSISTENT(otherFramVar)
uint8_t otherFramVar = 0;

/*
 * myArray is an array stored in RAM.
 * It is used simply to represent other RAM variables in a real application
 */
volatile uint8_t myArray[10] = {1, 2, 3, 4, 5, 6, 7, 8, 9, 10};

void IPE_encapsulatedInit(void);
void IPE_encapsulatedBlink(void);

/*
 * main.c
 */
void main(void) {
    WDTCTL = WDTPW | WDTHOLD;   //Stop watchdog timer

    /*
     * Initialize the variables inside the encapsulated area
     */
    IPE_encapsulatedInit();

    /*
     * Initialize other variables used by the application that are stored FRAM
     */
    otherFramVar = 0;

    /*
     * Initialize GPIOs.
     * Set P1.0 up for the green LED output.
     * Unlock GPIOs for settings to take effect.
     */
    P1DIR |= BIT0;
    PM5CTL0 &= ~LOCKLPM5;

    while(1)
    {
    	/*
    	 * Code to represent operations happening on FRAM variables outside IPE
    	 */
        otherFramVar++;

        /*
         * Code to represent operations happening on RAM variables
         */
        myArray[0]++;

        /*
         * Toggle the green LED to show one iteration through the main while(1)
         */
        P1OUT ^= BIT0;

        /*
         * Delay to make the green LED toggle visible
         */
        __delay_cycles(500000);

        /*
         * Call IPE_encapsulatedBlink(), a function stored in the IPE area.
         */
        IPE_encapsulatedBlink();

        /*
         * Code to represent operations happening on FRAM variables outside IPE
         */
        if(otherFramVar > 10)
        	otherFramVar = 0;
    }
}

/*
 * IPE_encapsulatedInit
 *
 * Function stored in the IPE area.
 * This function initializes any variables stored in the IPE area, because they
 * cannot be initialized by normal C-startup code because that is outside the
 * IPE area.
 */
#pragma CODE_SECTION(IPE_encapsulatedInit, ".ipe")
void IPE_encapsulatedInit(void)
{
	/*
	 * IPE_encapsulatedCount initialized to 1 for 1 blink on first execution of
	 * IPE_encapsulatedBlink()
	 */
	IPE_encapsulatedCount = 1;
}

/*
 * IPE_encapsulatedBlink
 *
 * Function stored in the IPE area
 * This function represents the actual code that would be the
 * proprietary code IP to be hidden in the encapsulated area.
 *
 * In this case, it simply:
 * 1. Initializes the red LED
 * 2. Initializes a timer module for the delay between LED toggles
 * 3. Toggles the LED for a number of times set by the IPE_encapsulatedCount
 *    IPE variable.
 * 4. Clears all module registers used (Timer, GPIO) after the code finishes.
 *
 * It also demonstrates the use of a constant in the IPE area to load AES keys
 * into the AES module, and clearing the AES module key at the end again so that
 * the keys are not already loaded for use by any code outside IPE.
 */
#pragma CODE_SECTION(IPE_encapsulatedBlink, ".ipe")
void IPE_encapsulatedBlink(void)
{
	/*
	 * Initialize GPIOs used by IPE code.
	 * Set P4.6 up for the red LED output. Starts out low.
	 */
    P4OUT &= ~BIT6;
    P4DIR |= BIT6;

    /*
     * Initialize the timer for setting the delay between red LED toggles
     * 50000/~1000000 = 50ms delay (20 times per second)
     */
    TA0CCTL0 = CCIE;
    TA0CCR0 = 50000;

    /*
     * Loading AES keys from IPE const section
     */
    for(IPE_i = 0; IPE_i < 8; IPE_i++)
        AESAKEY = IPE_encapsulatedKeys[IPE_i];

    /*
     * Red LED toggles for the number of times stored in the
     * IPE_encapsulatedCount variable
     */
    for(IPE_i = 0; IPE_i < IPE_encapsulatedCount; IPE_i++)
    {
        P4OUT |= BIT6;                          //Turn red LED on
        TA0CTL = TASSEL__SMCLK| MC__UP | TACLR; //Start timer
        __bis_SR_register(LPM0_bits + GIE);     //Sleep until timer int (50ms)
        __no_operation();
        P4OUT &= ~BIT6;                         //Turn red LED off
        __bis_SR_register(LPM0_bits + GIE);     //Sleep until timer int (50ms)
        __no_operation();
        TA0CTL = 0;                             //Stop timer
    }

    /*
     * Increment IP_encapsulatedCount to toggle 1 more time on next call of
     * IPE_encapsulatedBlink(), up to 5 times max
     */
    if(IPE_encapsulatedCount < 5)
    	IPE_encapsulatedCount++;
    else
    	IPE_encapsulatedCount = 1;

    /*
     * Reset AES module so that keys and settings will be cleared, so it can't
     * be used by any code outside of the IPE area
     */
    AESACTL0 = AESSWRST;

    /*
     * Clear all registers used (IPE best practice to conceal what
     * IPE code does)
     * Clear the timer registers and port registers in this case
     */
    TA0CCTL0 = 0;
    TA0CCR0 = 0;
    TA0CCR1 = 0;
    TA0CCR2 = 0;
    TA0CTL = 0;
    TA0R = 0;
    P4DIR &= ~BIT6;
    P4OUT &= ~BIT6;

    /*
     * Clear General purpose CPU registers R4-R15
     *
     * Note: if passing parameters back to code outside IPE, you may need to
     * preserve some CPU registers used for this. See www.ti.com/lit/pdf/slau132
     * MSP430 C/C++ compiler guide section on How a Function Makes a Call and
     * How a Called Function Responds for more information about the registers
     * that the compiler uses for these parameters, R12-R15. Note that if too
     * many arguments are passed, the stack will be used as well.
     *
     * In this case the function is declared void(void), so there is no issue.
     */
    __asm(" mov.w #0, R4");
    __asm(" mov.w #0, R5");
    __asm(" mov.w #0, R6");
    __asm(" mov.w #0, R7");
    __asm(" mov.w #0, R8");
    __asm(" mov.w #0, R9");
    __asm(" mov.w #0, R10");
    __asm(" mov.w #0, R11");
    __asm(" mov.w #0, R12");
    __asm(" mov.w #0, R13");
    __asm(" mov.w #0, R14");
    __asm(" mov.w #0, R15");

    /*
     * At this point, it may also be desired to clear any RAM that was allocated
     * in the course of the function, because RAM is not cleared on
     * de-allocation. However, this is completely application-dependent and care
     * would need to be taken not to corrupt the stack that is needed by the
     * application for proper functioning. It may be better in cases like this
     * to either:
     * 1. Use no local variables, and therefore only use variables in
     *    FRAM IPE area, to have less concern (though there could still be some
     *    stack usage leaving things in RAM after execution).
     *    This is what is done in this example (see how even the loop counter
     *    IPE_i is placed in IPE rather than a local variable).
     * 2. If there is real concern about what could be placed in RAM, write the
     * 	  IPE routines in assembly so that the user can completely control RAM
     * 	  and stack usage.
     */
}

/*
 * TIMER0_A0_ISR
 *
 * Demonstrates placing an ISR in the IPE area. This ISR is used for the timer
 * delay for the LED toggling in the IPE area, and simply wakes the device.
 */
#pragma CODE_SECTION(TIMER0_A0_ISR, ".ipe:_isr")
#pragma vector=TIMER0_A0_VECTOR
__interrupt
void TIMER0_A0_ISR(void)
{
    __bic_SR_register_on_exit(LPM0_bits); //Wake the device on ISR exit
    __no_operation();
}
