/* --COPYRIGHT--,BSD
 * Copyright (c) 2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//============================================================================
// curved_slider.scad
//
// This OpenScad script creates a capacitive touch curved slider sensor.
// The sensor design is exported to DXF and can be imported into any PCB layout.
// User controls sensor design by editing various parameters, such as size,
// number of elements, spacing, etc. OpenSCAD is unit less so the values you use
// to represent your dimensions can be mm, mils, etc.  Just remember
// when importing into PCB CAD tool, specify same units used in the design.
// For example, if parameters are in mm, then specify mm when importing.
//
// NOTE:  The accuracy of the dimensions created by this script cannot be
// guaranteed.  A reasonable effort has been made verify the accuracy,
// however, it is up to the user to verify.
//
// Texas Instruments
// MSP430 Applications
// version 1.0
// March 26, 2019

//====================================================================
// USER DEFINED INPUTS:
//====================================================================
// STEP #1
// USER DEFINED NUMBER OF ELEMENTS IN THE WHEEL (MIN IS 3, TYPICAL IS 4)
total_elements = 4;

// STEP #2
// USER DEFINED NUMBER OF FINGERS (TINES) (TYPICAL = 5)
tines = 5;

// STEP #3
// START ANGLE (DEGREES)
start_angle = 45;
stop_angle = 135;

// STEP #4
// USER DEFINED OUTER RADIUS (mils IN THIS EXAMPLE)
r_outside = 2000;

// STEP #5
// USER DEFINED INNER RADIUS (mils IN THIS EXAMPLE)
r_inside = 1750;

// STEP #6
// USER DEFINED LEFT AND RIGHT END PADDING SIZE (mils IN THIS EXAMPLE)
padding = 50;

// STEP #7
// USER DEFINED CLEARANCES AND TIP WIDTH (mils IN THIS EXAMPLE)
clearance = 10;
tip = 10;

// STEP #7 (OPTIONAL)
// USER DEFINED RESOLUTION
// DEFAULT SHOULD BE BETWEEN 100 AND 200
// SMALLER VALUES ARE LESS ACCURATE
// LARGER VALUE TAKES LONGER TO RENDER
$fn = 400;
number_of_steps = $fn;

//====================================================================
// CALCULATIONS:
//====================================================================
slider_theta = stop_angle - start_angle;
slider_width = r_outside - r_inside;
middle_radius = r_inside + (slider_width/2);
pitch = (slider_width)/(tines - 1);
base = (pitch * 2) - (2 * clearance + tip);
radii = [for(i = [0:tines-1]) r_outside - (i * pitch)];
CCW = 1;
CW = -1;
LEFT_SIDE = -1;
RIGHT_SIDE = 1;

// LEFT/RIGHT PADDING
padding_theta = calculate_arc_theta(padding, middle_radius);

// CLEARANCE THETA ANGLE
// Take clearance as arc length and calculate corresponding theta
clearance_theta = calculate_arc_theta(clearance, middle_radius);

// ELEMENTS
// Calculate the angle (theta) of an element
sector_theta = (slider_theta - (2 * padding_theta)) / total_elements;

//====================================================================
// COMMAND LINE BUILD CONSTANT
//====================================================================
element=99;

//====================================================================
// FUNCTIONS:
//====================================================================
// FUNCTION THETA TO ARC LENGTH
function calculate_arc_theta(arc_length, radius) = (arc_length * 360)/(2 * PI * radius);

// FUNCTION ARC LENGTH TO THETA
function calculate_arc_length(arc_theta, radius) = (arc_theta/360) * (2 * PI * radius);

//====================================================================
// MODULES:
//====================================================================

//====================================================================
// CREATE_LEFT_ELEMENT BUILDS THE LEFT HAND SIDE HALF-ELEMENT
module create_left_element()
{
    for(i=[0:2:tines-1])
        create_tine(radii[i], CW);  
    
}

//====================================================================
// CREATE_RIGHT_ELEMENT BUILDS THE RIGHT HAND SIDE HALF-ELEMENT
module create_right_element()
{
    for(i=[1:2:tines-1])
        create_tine(radii[i], CCW);  
}

//===========================================================
// CREATE_PAD BUILDS THE PADDING THAT IS ATTACHED TO END OF LEFT AND RIGHT ELEMENTS
module create_pad(radius, direction)
{
    step_size = padding_theta / number_of_steps;
    
 for(i=[0:number_of_steps-1]){
    
  hull(){
    rotate([0,0,start_angle + (direction)* (i * step_size)])
        translate([radius,0,0])
            square(size=[slider_width ,.01], center = true);
 
    rotate([0,0,start_angle + (direction)* ((i+1) * step_size)])
        translate([radius,0,0])
            square(size=[slider_width ,.01], center = true);  
  };
}   
}//end module

//===========================================================
// CREATE_TINE CREATES THE TINES OR FINGERS ON EACH ELEMENT
module create_tine(radius, direction)
{
    width_decrement = ((base - tip) / number_of_steps);
    theta_step_size = ((sector_theta - (clearance_theta)) / number_of_steps);
   
    for(i=[0:number_of_steps-1]){
   
        hull()
        {
            rotate([0,0,start_angle + (direction * i * theta_step_size)])
                translate([radius,0,0])
                    square(size=[base  - (i * width_decrement),.01], center = true);
                    
            rotate([0,0,start_angle + (direction* (i+1) * theta_step_size)])
                translate([radius,0,0])
                    square(size=[base - ((i+1) * width_decrement),.01], center = true);  
        };
    }   
}//end module

//===========================================================
// CREATE_ELEMENT CREATES ALL FULL ELEMENTS
module create_element()
{
    for(i=[0:tines-1])
    {
        if(i%2 == 0)
            create_tine(radii[i], CW);  
        else
            create_tine(radii[i], CCW);  
    }
    
}//end module

//===========================================================
// SLIDER MODULE BUILDS AND ASSEMBLES THE ELEMENTS INTO A SLIDER SENSOR
module curved_slider()
{
    if(element==99)
    {
        echo("Building element 0");
        create_pad(middle_radius, RIGHT_SIDE);
        rotate([0,0, padding_theta])create_right_element();
       
        for(i=[1:total_elements-1]){
            echo("Building element ", i);
            rotate([0,0,(i * sector_theta) + padding_theta]) create_element();
        }
        echo("Building element ", total_elements);
        rotate([0,0,(total_elements * sector_theta) + padding_theta]) create_left_element();
        rotate([0,0, slider_theta]) create_pad(middle_radius,LEFT_SIDE);

    }    
    else
    {  
       echo("Building element ",element); 
       if(element == 0)
       {
            create_pad(middle_radius, RIGHT_SIDE);
            rotate([0,0, padding_theta]) create_right_element();   
            
            rotate([0,0,((total_elements) * sector_theta) + padding_theta]) create_left_element();
            rotate([0,0, slider_theta]) create_pad(middle_radius,LEFT_SIDE);
       }
       else if (element < total_elements)
       {
            rotate([0,0, (element * sector_theta) + padding_theta])  create_element();
       }
       else if(element == total_elements)
       {
         //  rotate([0,0,(element * sector_theta) + padding_theta]) create_left_element();
         //  rotate([0,0, slider_theta]) create_pad(middle_radius,LEFT_SIDE);
       }
       else
           echo("ERROR: Wrong number of elements");
   }
    
   
}//end module

//====================================================================
// MAIN CODE:

// TOP LEVEL - THIS IS WERE THE SLIDER IS RENDERED AND TRIMMED

// PERFORM FINAL TRANSLATE TO CURVED SLIDER IS NEAR ORIGIN
//translate([0, -r_inside,0])
{
// DIFFERENCE TRIMS INNER RADIUS
difference(){
    
    //INTERSECTION TRIMS OUTER RADIUS
    intersection()
    {
        circle(d = 2 * r_outside);
        curved_slider();
    }
    
    circle(d = 2 * r_inside);
    
}
}
echo("Done");
