/* --COPYRIGHT--,BSD
 * Copyright (c) 2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//============================================================================
// slider.scad
//
// This OpenScad script creates a capacitive touch slider sensor.
// The sensor design is exported to DXF and can be imported into any PCB layout.
// User controls sensor design by editing various parameters, such as size,
// number of elements, spacing, etc. OpenSCAD is unit less so the values you use
// to represent your dimensions can be mm, mils, etc.  Just remember
// when importing into PCB CAD tool, specify same units used in the design.
// For example, if parameters are in mm, then specify mm when importing.
//
// NOTE:  The accuracy of the dimensions created by this script cannot be
// guaranteed.  A reasonable effort has been made verify the accuracy,
// however, it is up to the user to verify.
//
// Texas Instruments
// MSP430 Applications
// version 1.0
// March 26, 2019
//
// ================= CONSTANTS ======================
// DON'T CHANGE THESE   
$fn = 400;
LEFT = 1;
RIGHT = -1;

//====================================================================
// USER DEFINED INPUTS:
//====================================================================
// STEP #1
// USER DEFINED NUMBER OF ELEMENTS IN THE SLIDER (MIN IS 3, TYPICAL IS 4)
total_elements = 4;

// STEP #2
// USER DEFINED NUMBER OF FINGERS (TINES) (TYPICAL = 5)
tines = 5;

// STEP #3
// USER DEFINED LENGTH (mm IN THIS EXAMPLE)
slider_length = 150;

// STEP #4
// USER DEFINED WIDTH (mm IN THIS EXAMPLE)
slider_width = 15;

// STEP #5
// USER DEFINED LEFT AND RIGHT END PADDING SIZE (mm IN THIS EXAMPLE)
//(mm IN THIS EXAMPLE)
padding = 2;

// STEP #6
// USER DEFINED CLEARANCES AND TIP WIDTH (mm IN THIS EXAMPLE)
//(mm IN THIS EXAMPLE)
clearance = 0.5; 
tip_width = 0.25;


// NOTE: The accuracy of an object's dimensions created by these script
// cannot be guaranteed accurate for all possible combinations of user inputs.
// A reasonable effort has been made to verify the accuracy of several
// rendered designs using realistic parameters, however, it is up to the
// user to verify the accuracy of any design created from these scripts.

//====================================================================
// CALCULATIONS:
//====================================================================
tine_length = (slider_length - (2 * padding))/total_elements;
pitch = (slider_width)/(tines-1);
base = (pitch * 2) - (2 * clearance + tip_width);
offsets = [for(i=[0:tines-1]) (i * pitch)];


//====================================================================
// COMMAND LINE BUILD CONSTANT
//====================================================================
element=99;

//====================================================================
// MODULES:
//====================================================================

// NOTE: The accuracy of an object's dimensions created by these script
// cannot be guaranteed accurate for all possible combinations of user inputs.
// A reasonable effort has been made to verify the accuracy of several
// rendered designs using realistic parameters, however, it is up to the
// user to verify the accuracy of any design created from these scripts.

//===========================================================
// CREATE_TINE CREATES THE TINES OR FINGERS ON EACH ELEMENT
module create_tine(offset, direction)
{
    hull(){
    translate([0,offset,0])
        square(size=[0.01, base],center = true);
    translate([direction * (tine_length - clearance),offset,0])
        square(size=[0.01, tip_width],center = true);
    }
}

//===========================================================
// CREATE_MIDDLE_ELEMENTS CREATES ALL FULL ELEMENTS
module create_middle_elements()
{
    for(i=[0:tines-1])
    {
        if(i%2 == 0)
            create_tine(offsets[i], 1);  
        else
            create_tine(offsets[i], -1);  
    }
}

//===========================================================
// CREATE_PAD BUILDS THE PADDING THAT IS ATTACHED TO END OF LEFT AND RIGHT ELEMENTS
module create_pad()
{
    square(size=[padding, slider_width],center = false);
}//end module

//===========================================================
// CREATE_LEFT_ELEMENT BUILDS THE LEFT HAND SIDE HALF-ELEMENT
module create_left_element()
{
  create_pad();
  translate([padding,0,0])
  {  
      for(i=[0:2:tines-1])
      {
        create_tine(i * pitch,1);
      }
  }
}//end module

//===========================================================
// CREATE_RIGHT_ELEMENT BUILDS THE RIGHT HAND SIDE HALF-ELEMENT
module create_right_element()
{
    translate([slider_length - padding,0,0])
    {
        create_pad();
        
        for(i=[1:2:tines-1])
        {
            create_tine(i * pitch,-1);
        }
    }
}//end module

//===========================================================
// SLIDER MODULE BUILDS AND ASSEMBLES THE ELEMENTS INTO A SLIDER SENSOR
module slider()
{
    if(element==99)
    {
        echo("Building element 0");       
        create_left_element();
        create_right_element();
        
        translate([padding,0,0])
        {  
            for(i=[1:total_elements-1]) translate([i * tine_length,0,0])
            {
                create_middle_elements();
                echo("Building element ",i);
            }
        }
    }
    else if(element==0)
    {
        echo("Building element 0");       
        create_left_element();
        create_right_element();
    }
    else if(element < total_elements)
    {
        translate([padding,0,0])
        {  
            translate([element * tine_length,0,0])
            {
                create_middle_elements();
                echo("Building element ",element);
            }
        }
    }
    else
        echo("ERROR: Wrong number of elements");
  
}//end module

//====================================================================
// MAIN CODE:
//====================================================================

// TOP LEVEL - THIS IS WERE THE SLIDER IS RENDERED AND TRIMMED

// INTERSECTION TRIMS OUTSIDE OF SLIDER
intersection()
{
    square(size=[slider_length , slider_width], center=false);

    slider();
}
echo("Done");