/* --COPYRIGHT--,BSD
 * Copyright (c) 2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//============================================================================
// wheel.scad
//
// This OpenScad script creates a capacitive touch wheel sensor.
// The sensor design is exported to DXF and can be imported into any PCB layout.
// User controls sensor design by editing various parameters, such as size,
// number of elements, spacing, etc. OpenSCAD is unit less so the values you use
// to represent your dimensions can be mm, mils, etc.  Just remember
// when importing into PCB CAD tool, specify same units used in the design.
// For example, if parameters are in mm, then specify mm when importing.
//
// NOTE:  The accuracy of the dimensions created by this script cannot be
// guaranteed.  A reasonable effort has been made verify the accuracy,
// however, it is up to the user to verify.
// 
// Texas Instruments
// MSP430 Applications
// version 1.0
// March 26, 2019
//
// ================= CONSTANTS ======================
// DON'T CHANGE THESE 
$fn = 400; 

//====================================================================
// USER DEFINED INPUTS:
//====================================================================

// STEP #1
// USER DEFINED NUMBER OF ELEMENTS IN THE WHEEL (MIN IS 3, TYPICAL IS 4)
total_elements = 4;

// STEP #2
// USER DEFINED NUMBER OF FINGERS (TINES) (TYPICAL = 5)
tines = 5;

// STEP #3
// USER DEFINED START ANGLE (DEGREES)
start_angle = 90;

// STEP #4
// USER DEFINED OUTER RADIUS (mm IN THIS EXAMPLE)
r_outside = 30;

// STEP #5
// USER DEFINED INNER RADIUS (mm IN THIS EXAMPLE)
r_inside = 18;

// STEP #6
// USER DEFINED CLEARANCES AND TIP WIDTH (mm IN THIS EXAMPLE)
clearance = 0.5;
tip = .25;

// NOTE: The accuracy of an object's dimensions created by these script
// cannot be guaranteed accurate for all possible combinations of user inputs.
// A reasonable effort has been made to verify the accuracy of several
// rendered designs using realistic parameters, however, it is up to the
// user to verify the accuracy of any design created from these scripts.

//====================================================================
// CALCULATIONS:
//====================================================================
slider_theta = 360;
slider_width = r_outside - r_inside;
middle_radius = r_inside + (slider_width/2);
pitch = (slider_width)/(tines - 1);
base = (pitch * 2) - (2 * clearance + tip);
radii = [for(i = [0:tines-1]) r_outside - (i * pitch)];
number_of_steps = $fn;

// CLEARANCE THETA ANGLE
// Take clearance as arc length and calculate corresponding theta
clearance_theta = calculate_arc_theta(clearance, middle_radius);

// ELEMENTS
// Calculate the angle (theta) of an element
sector_theta = (slider_theta) / total_elements;

//====================================================================
// CONSTANTS:
//==================================================================== 
// This determines the direction (CW or CCW) the wheel is constructed.
rotation_direction = -1;


//====================================================================
// COMMAND LINE BUILD CONSTANT
//====================================================================
element=99;

//====================================================================
// FUNCTIONS:
//====================================================================
// FUNCTION THETA TO ARC LENGTH
function calculate_arc_theta(arc_length, radius) = (arc_length * 360)/(2 * PI * radius);

// FUNCTION ARC LENGTH TO THETA
function calculate_arc_length(arc_theta, radius) = (arc_theta/360) * (2 * PI * radius);

//====================================================================
// MODULES:
//====================================================================

//===========================================================
// CREATE_TINE CREATES THE TINES OR FINGERS ON EACH ELEMENT
module create_tine(radius, direction)
{
    width_decrement = ((base - tip) / number_of_steps);
    theta_step_size = ((sector_theta - (clearance_theta)) / number_of_steps);
   
    for(i=[0:number_of_steps-1]){
        hull()
        {
            rotate([0,0,start_angle + (direction * i * theta_step_size)])
                translate([radius,0,0])
                    square(size=[base  - (i * width_decrement),.01], center = true);
            rotate([0,0,start_angle + (direction* (i+1) * theta_step_size)])
                translate([radius,0,0])
                    square(size=[base - ((i+1) * width_decrement),.01], center = true);  
         };
    }   
}//end module

//===========================================================
// CREATE_ELEMENT CREATES THE ELEMENTS
module create_element()
{
    for(i=[0:tines-1])
    {
        if(i%2 == 0)
            create_tine(radii[i], 1);  
        else
            create_tine(radii[i], -1);  
    }
    
}//end module

//===========================================================
// WHEEL MODULE BUILDS AND ASSEMBLES THE ELEMENTS INTO A WHEEL SENSOR
module wheel()
{
    if(element == 99)
    {
       for(i=[0:total_elements - 1])
       {
           echo("building element: ", i);
           rotate([0,0,(i * sector_theta * rotation_direction)]) create_element();
       }
    }    
    else
    {  
       echo("building element: ",element);
       rotate([0,0,(element * sector_theta * rotation_direction)]) create_element();
   }
}//end module

//====================================================================
// MAIN CODE:
//====================================================================
// TOP LEVEL - THIS IS WERE THE WHEEL IS RENDERED AND TRIMMED
// DIFFERENCE TRIMS INNER RADIUS
difference(){
    
    //INTERSECTION TRIMS OUTER RADIUS
    intersection()
    {
        circle(d = 2 * r_outside);
        wheel();
    }
    
    circle(d = 2 * r_inside);
    
}
echo("done");