/*
 * PPG.c
 *
 *  Created on: Apr 8, 2020
 *      Author: a0223954
 */
#include "msp430.h"
#include "PPG.h"
#include "log.h"
#include "algorithm.h"


#define PERIOD              (6000000/SAMPLE_RATE-1)  // TB3 CCR0
#define INTERVAL            (6*1000-1)               // 1000us
#define DAC_ON_TIME         (6*250-1)                // 250us
#define OA2_GAIN            32
#define VCC_HALF            2048
#define DAC_VOS             0
#define VREF                2000                    // 200mV
#define IDAC_VAL(x)         (x*20.0/VREF*4095)      // x - mA@20ohm feedback resistor

#define EN_FILTER           0

#define PPG_IR              0
#define PPG_VS              1
#define PPG_AB              2

typedef struct {
    uint8_t id;
    int16_t sample[2];
    int16_t led_level;
    int16_t dc_1st;
    int32_t dc_register_1st;
    int16_t offset_2nd;
    int32_t dc_2nd;
    int32_t dc_register_2nd;
    int32_t signal;
    FILTER filter;
}PPG_Signal;

PPG_Signal vs_ppg;
PPG_Signal ir_ppg;
PPG_Signal ab_ppg;

PPG_Info ppgInfoVS;
PPG_Info ppgInfoIR;

uint32_t count;
int16_t adc_buf[2][16];

uint8_t log_count;
uint8_t log_signal;
uint16_t log_saturation_count;

void PPG_turnOffLight(PPG_Signal *ppg);
void PPG_setLightAndPGAOffset(PPG_Signal *ppg);
void PPG_performMeasurement(PPG_Signal *ppg);
void PPG_initDCTracker(PPG_Signal *ppg);
void PPG_adjustPGAOffset(PPG_Signal *ppg);
void PPG_recoverFullSignal(PPG_Signal *ir_ppg, PPG_Signal *vs_ppg, PPG_Signal *ab_ppg);

// post algorithm
int8_t PPG_signalQualityCheck(PPG_Signal *ppg, PPG_Info *ppgInfo);
void PPG_searchOnsetAndPeak(PPG_Signal *ppg, PPG_Info *ppgInfo);


// Functions
void PPG_init(void)
{
    /** Vref init  **/
    // Configure reference module
    PMMCTL0_H = PMMPW_H;                      // Unlock the PMM registers
    PMMCTL2 = INTREFEN | REFVSEL_1;           // Enable internal 2.0V reference
    while(!(PMMCTL2 & REFGENRDY));            // Poll till internal reference settles
    PMMCTL0_H = 0xFF;                         // lock the PMM registers

    /**  DAC init  **/
    // Configure SAC1 pins as IDAC for RED LED driving
    // P1.5/OA1O, P1.6/OA1-
    P1SEL0 |= BIT5 + BIT6;
    P1SEL1 |= BIT5 + BIT6;

    SAC1DAC = DACSREF_1 + DACLSEL_0;             // Select internal Vref as DAC reference,DAC latch loads when DACDAT written
    SAC1DAC |= DACEN;                            // Enable SAC1 DAC
    SAC1OA = NMUXEN + PMUXEN + PSEL_1 + NSEL_0;  // Select positive and negative input: positive-DAC, negative-P1.6
    SAC1OA |= SACEN;                             // Enable SAC1
    SAC1DAT = 0;                                 // mA@20ohm, VRED = 1.674V@10mA

    // Configure SAC3 pins as IDAC for IR LED driving
    // P3.5/OA3O, P3.6/OA3-
    P3SEL0 |= BIT5 + BIT6;
    P3SEL1 |= BIT5 + BIT6;

    SAC3DAC = DACSREF_1 + DACLSEL_0;              // Select internal Vref as DAC reference, DAC latch loads when DACDAT written
    SAC3DAC |= DACEN;                             // Enable SAC3 DAC
    SAC3OA = NMUXEN + PMUXEN + PSEL_1 + NSEL_0;   // Select positive and negative input: positive-DAC, negative-P3.6
    SAC3OA |= SACEN;                              // Enable SAC3
    SAC3DAT = 0;                                  // mA@20ohm, VIR = 1.464V@10mA

    /** SAC init **/
    // Configure SAC2 as TIA (I-V conversion)
    P3SEL0 |= BIT1 + BIT2;
    P3SEL1 |= BIT1 + BIT2;

    SAC2DAC = DACSREF_0 + DACLSEL_0;                // Select internal Vref as DAC reference, DAC latch loads when DACDAT written
    SAC2DAC |= DACEN;                               // Enable SAC2 DAC
    SAC2OA = NMUXEN + PMUXEN + PSEL_1 + NSEL_0;     // Select positive and negative input: positive-DAC, negative-P3.2
    SAC2OA |= SACEN + OAEN;                         // Enable SAC2 and OA2
    SAC2DAT = 600;                                  // Set SAC2 DAC output = 600/4095*2000mV = 293mV

    // Configure SAC0 as PGA inverting amplifier mode
    SAC0DAC = DACSREF_0 + DACLSEL_0;                // Select internal Vref as DAC reference,DAC latch loads when DACDAT written
    SAC0DAC |= DACEN;                               // Enable SAC0 DAC
    SAC0DAT = 1500;                                 // Set SAC0 DAC output = 0.75V
    SAC0OA = NMUXEN + PMUXEN + PSEL_1 + NSEL_1;     // Select positive and negative input: positive-DAC, negative-paired OA output
    SAC0PGA = MSEL_3;                               // Set negative input as paired OA output - SAC2
    SAC0PGA |= GAIN0 + GAIN1 + GAIN2;               // set PGA gain to 32
    //SAC0PGA |= GAIN0 + GAIN2;                     // set PGA gain to 16
    //SAC0PGA |= GAIN2;                             // set PGA gain to 8
    //SAC0PGA |= GAIN0 + GAIN1;                     // set PGA gain to 4
    //SAC0PGA |= GAIN1;                             // set PGA gain to 2
    //SAC0PGA |= GAIN0;                             // set PGA gain to 1
    SAC0OA |= SACEN + OAEN;                         // Enable SAC1 and OA1

    /** ADC_init(void) **/
    // A0/A1
    P1SEL0 |= BIT0 + BIT1;
    P1SEL1 |= BIT0 + BIT1;

    // Configure ADC to sequence of channel mode
    ADCCTL0 |= ADCSHT_3 | ADCMSC | ADCON;                    // 32 ADCclks, MSC, ADC ON
    ADCCTL1 |= ADCSHS_0 | ADCSHP | ADCCONSEQ_1 | ADCSSEL_3;  // ADC clock SMCLK, sampling timer, sw trig,single sequence
    ADCCTL2 &= ~ADCRES;                                      // clear ADCRES in ADCCTL
    ADCCTL2 |= ADCRES_2;                                     // 12-bit conversion results
    ADCMCTL0 |= ADCINCH_1 | ADCSREF_0;                       // A0~1(EoS); Vref = Internal Vref


    /** TB3 init **/
    // Configure TB3 for period control
    TB3CCR0 = PERIOD;                                        // CCR0@400Hz for measurement period control
    TB3CCR1 = PERIOD-3*INTERVAL;                             // CCR1 for IR signal measurement, remain 100us for signal stable
    TB3CCR2 = PERIOD-2*INTERVAL;                             // CCR2 for VS signal measurement
    TB3CCR3 = PERIOD-1*INTERVAL;                             // CCR3 for AMB signal measurement
    TB3CCTL0 |= CCIE;                                        // CCR0 interrupt enabled
    TB3CCTL1 |= CCIE;                                        // CCR1 interrupt enabled
    TB3CCTL2 |= CCIE;                                        // CCR2 interrupt enabled
    TB3CCTL3 |= CCIE;                                        // CCR2 interrupt enabled
    TB3CTL = TBSSEL__SMCLK | MC__STOP;                       // SMCLK, UP mode

    __delay_cycles(24000);                                   // Delay 1ms for setting stable
}

void PPG_appStart(void)
{
    // Initialize parameter and start measurement loop
    count = 0;

    //Set initial values for the LED brightnesses
    ir_ppg.id    = PPG_IR;
    vs_ppg.id    = PPG_VS;
    ab_ppg.id    = PPG_AB;
    ir_ppg.led_level = IDAC_VAL(8);     // 10mA@20ohm
    vs_ppg.led_level = IDAC_VAL(10);     // 10mA@20ohm
    ab_ppg.led_level = 0;

    ir_ppg.filter.offset = 0;
    vs_ppg.filter.offset = 0;

    TB3CTL |= TBCLR | MC__UP;           // TB3 Start, 400Hz measurement loop

    __enable_interrupt();               // Enableinterrupts
    __no_operation();
}

void PPG_setLightAndPGAOffset(PPG_Signal *ppg)
{
    if(ppg->id == PPG_IR)
    {
        SAC3OA |= OAEN;                 // Enable OA
        SAC3DAT = ppg->led_level;       // Set DAC value for IR LED driving
    }
    if(ppg->id == PPG_VS)
    {
        SAC1OA |= OAEN;                 // Enable OA
        SAC1DAT = ppg->led_level;       // Set DAC value for IR LED driving
    }

    SAC0DAT = ppg->offset_2nd;          // Set 2nd stage PGA DC offset
}

void PPG_turnOffLight(PPG_Signal *ppg)
{
    if(ppg->id == PPG_IR)
    {
        SAC3DAT = 0;                    // Disable OA
        SAC3OA &= ~OAEN;
    }
    if(ppg->id == PPG_VS)
    {
        SAC1DAT = 0;                    // Disable OA
        SAC1OA &= ~OAEN;
    }
}

void PPG_performMeasurement(PPG_Signal *ppg)
{
    uint8_t i;

    // 16x oversampling
    for(i = 0;i < 16; i++)
    {
        ADCCTL0 |= ADCENC | ADCSC; // trigger ADC conversion
        while((ADCIFG&ADCIFG0) == 0); // ADC conversion completed
        adc_buf[1][i] = ADCMEM0;
        while((ADCIFG&ADCIFG0) == 0); // ADC conversion completed
        adc_buf[0][i] = ADCMEM0;
    }
    ppg->sample[0] = average(&adc_buf[0][0], 16); // A0 - TIA output
    ppg->sample[1] = average(&adc_buf[1][0], 16); // A1 - PGA output
}

void PPG_initDCTracker(PPG_Signal *ppg)
{
    if((ppg->sample[1]>4000)||(ppg->sample[1]<100)) // PGA output saturation
    {
        ppg->dc_1st = ppg->sample[0];
        ppg->dc_register_1st = ((int32_t)ppg->dc_1st<<K);
    }
}

void PPG_adjustPGAOffset(PPG_Signal *ppg)
{
    // dc estimator
    ppg->dc_1st = dc_estimator(&(ppg->dc_register_1st), ppg->sample[0]);

    // Second stage offset adjust
    ppg->offset_2nd = ((int32_t)ppg->dc_1st*OA2_GAIN + VCC_HALF)/(OA2_GAIN + 1) - DAC_VOS;;
}

// Recover full PPG signal with AC + DC: g*VO1 = ir_2nd_offset*(g+1) - ir_sample[1]
void PPG_recoverFullSignal(PPG_Signal *ir_ppg, PPG_Signal *vs_ppg, PPG_Signal *ab_ppg)
{
    // Recover full PPG signal with AC + DC + Ambient + Offset
    ir_ppg->signal = (int32_t)ir_ppg->offset_2nd*OA2_GAIN + (ir_ppg->offset_2nd - ir_ppg->sample[1]);
    vs_ppg->signal = (int32_t)vs_ppg->offset_2nd*OA2_GAIN + (vs_ppg->offset_2nd - vs_ppg->sample[1]);
    ab_ppg->signal = (int32_t)ab_ppg->offset_2nd*OA2_GAIN + (ab_ppg->offset_2nd - ab_ppg->sample[1]);

    // Remove ambient + Offset
    ir_ppg->signal = ir_ppg->signal - ab_ppg->signal;
    vs_ppg->signal = vs_ppg->signal - ab_ppg->signal;

    // Filter out noise
    ir_ppg->signal = fir_filter(ir_ppg->signal, &(ir_ppg->filter));
    vs_ppg->signal = fir_filter(vs_ppg->signal, &(vs_ppg->filter));
}


int8_t PPG_signalQualityCheck(PPG_Signal *ppg, PPG_Info *ppgInfo)
{
    // Check the saturation of signal
    if(ppg->sample[0] > 4000)
    {
        ppgInfo->flag |= PPG_SIGNAL_SATURATION;   // signal saturation
        ppgInfo->log_sat_count++;
        return -1;
    }
    else
    {
        ppgInfo->flag &= ~PPG_SIGNAL_SATURATION;   // signal saturation
        ppgInfo->log_sat_count = 0;
    }

    return 0;
}

void PPG_searchOnsetAndPeak(PPG_Signal *ppg, PPG_Info *ppgInfo)
{
    int16_t diff0,diff1;
    uint16_t intval;

    // Filter out noise of signal


    if((ppg->sample[1]>4000)||(ppg->sample[1]<100)) // PGA output saturation
    {
        ppgInfo->i = 0;
        return;
    }

    ppgInfo->i++;

    // Log x[i-2],x[i-1],x[i]
    ppgInfo->x[0] = ppgInfo->x[1];
    ppgInfo->x[1] = ppgInfo->x[2];
    ppgInfo->x[2] = ppg->signal;     // x[i-1] = x[i]

    // Get 0.6*diffMin as the thresh
    if(ppgInfo->i == 1)
    {
        ppgInfo->x[0] = ppg->signal;
        ppgInfo->x[1] = ppg->signal;
        ppgInfo->edge = 0;
        ppgInfo->beats = 0;
        ppgInfo->diffMin = 0;
    }
    else if(ppgInfo->i < 200)
    {
        diff0 = ppgInfo->x[2] - ppgInfo->x[1];
        if(ppgInfo->diffMin > diff0)
            ppgInfo->diffMin = diff0;
    }
    else if(ppgInfo->i == 200)
    {
        ppgInfo->thresh = (6*ppgInfo->diffMin)/10;
    }
    else if(ppgInfo->i <= 800)
    {
        // Search peak by diff
        //search_peak(ir_ppg->signal,ir_ppg->dc_2nd, &ppgInfoIR);
        diff0 = ppgInfo->x[1] - ppgInfo->x[0];
        diff1 = ppgInfo->x[2] - ppgInfo->x[1];
        if((diff0 < ppgInfo->thresh)&&(diff1 >= ppgInfo->thresh))
        {
            intval = ppgInfo->i - ppgInfo->edge; // for edge debouncer

            if((!ppgInfo->beats)||(intval > 20))
            {
                ppgInfo->flag |= PPG_FALLING_EDGE;
                ppgInfo->beats++;
                ppgInfo->edge = ppgInfo->i;
            }
        }
        if(ppgInfo->flag&PPG_FALLING_EDGE)
        {
            ppgInfo->flag &= ~PPG_FALLING_EDGE;
            // store search result
            if(ppgInfo->beats > 1)
            {
                // store onset and peak
                if(ppgInfo->beats < (HEART_BEATS+2))
                {
                    ppgInfo->peaks[ppgInfo->beats-2][0] = ppgInfo->peak[0];
                    ppgInfo->peaks[ppgInfo->beats-2][1] = ppgInfo->peak[1];
                    ppgInfo->onsets[ppgInfo->beats-2][0] = ppgInfo->onset[0];
                    ppgInfo->onsets[ppgInfo->beats-2][1] = ppgInfo->onset[1];
                }
            }
            // reset initial value
            ppgInfo->onset[1] = 0;
            ppgInfo->peak[1] = 0xFFFF;
        }
        // search onset and peak
        if(ppgInfo->onset[1] < ppgInfo->x[1])
        {
            if((ppgInfo->x[1] > ppgInfo->x[0])&&(ppgInfo->x[1] >= ppgInfo->x[2]))
            {
                ppgInfo->onset[0] = ppgInfo->i;
                ppgInfo->onset[1] = ppgInfo->x[1];
            }
        }
        if(ppgInfo->peak[1] > ppgInfo->x[1])
        {
            if((ppgInfo->x[1] < ppgInfo->x[0])&&(ppgInfo->x[1] <= ppgInfo->x[2]))
            {
                ppgInfo->peak[0] = ppgInfo->i;
                ppgInfo->peak[1] = ppgInfo->x[1];
            }
        }
    }
    else
    {
        ppgInfo->i = 0;
        if((ppgInfo->beats < (HEART_BEATS+2))&&(ppgInfo->beats > 1))
        {
            ppgInfo->log_beats = ppgInfo->beats - 1;
            memcpy(ppgInfo->log_peaks,ppgInfo->peaks,sizeof(uint16_t)*2*ppgInfo->log_beats);
            memcpy(ppgInfo->log_onsets,ppgInfo->onsets,sizeof(uint16_t)*2*ppgInfo->log_beats);
            ppgInfo->flag |= PPG_DATA_READY;
        }
    }
}

void PPG_appReInit(PPG_Results *ppgResults)
{
    count = 0;
    log_count = 0;
    ppgInfoIR.i = 0;
    ppgInfoVS.i = 0;
    ppgInfoIR.log_sat_count = 0;
    ppgInfoVS.log_sat_count = 0;
    ppgResults->new_state = 0;
    ppgResults->state = PPG_MEASURING;
}

void PPG_appHandler(PPG_Results *ppgResults)
{
    if((ppgInfoIR.flag&PPG_SIGNAL_SATURATION)||(ppgInfoVS.flag&PPG_SIGNAL_SATURATION))
    {
        if(ppgResults->state != PPG_FINGER_OUT)
            ppgResults->new_state = 1;
        ppgResults->state = PPG_FINGER_OUT;
        // mcu enter sleep mode if 500*10ms time out
        if((ppgInfoIR.log_sat_count > 500)||(ppgInfoVS.log_sat_count > 500))
            ppgResults->state = PPG_TIME_OUT;
    }
    else
    {
        if(ppgResults->state == PPG_FINGER_OUT)
        {
            ppgResults->new_state = 1;
            ppgResults->state = PPG_INITIAL;
        }
    }

    if((ppgInfoIR.flag&PPG_DATA_READY)&&(ppgInfoVS.flag&PPG_DATA_READY))
    {
        ppgInfoIR.flag &= ~PPG_DATA_READY;
        ppgInfoVS.flag &= ~PPG_DATA_READY;


        ppgResults->ir_pi = calculate_pi(&ppgInfoIR);
        ppgResults->vs_pi = calculate_pi(&ppgInfoVS);

        ppgResults->bpm = calculate_bpm(&ppgInfoIR);
        ppgResults->R   = ppgResults->vs_pi/ppgResults->ir_pi;
        ppgResults->SpO2 = calculate_spo2(ppgResults->R);

        if((ppgResults->SpO2<100)&&(ppgResults->bpm>=25)&&(ppgResults->bpm<=250))
            ppgResults->flag = 1;
    }
}

void PPG_appSleep(void)
{
    // Enter LPM0 and wait for measurement complete
    __bis_SR_register(LPM0_bits | GIE);                 // Enter LPM0 w/ interrupts
    __no_operation();
}

void PPG_appStop(void)
{
    TB3CTL &= ~MC__UP; // TB3 Stop
}

// Timer3 B0 interrupt service routine
#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector = TIMER3_B0_VECTOR
__interrupt void TIMER3_B0_ISR (void)
#elif defined(__GNUC__)
void __attribute__ ((interrupt(TIMER3_B0_VECTOR))) TIMER3_B0_ISR (void)
#else
#error Compiler not supported!
#endif
{
    //Step-1: PPG signal measurement and process main loop @ fiexd rate
    count++;

    //Step-1: Recover full AC+DC signal
    PPG_recoverFullSignal(&ir_ppg,&vs_ppg,&ab_ppg);

    //Step-2: Track the DC of TIA for PGA DC offset compensation
    PPG_initDCTracker(&ir_ppg);
    PPG_initDCTracker(&vs_ppg);
    PPG_initDCTracker(&ab_ppg);

    //Step-3: Second stage offset adjust
    PPG_adjustPGAOffset(&ir_ppg);
    PPG_adjustPGAOffset(&vs_ppg);
    PPG_adjustPGAOffset(&ab_ppg);

    //Step-4: Log signal for LCD wave display
    if(count%5 == 0)   // log per 50ms
    {
        log_count++;
        log_count = log_count%160;
        log_signal = ir_ppg.sample[1]/170;  // convert 0-4095 to 0-24
    }

    //Step-5: Post algorithm to get %SpO2 and bpm
    if(count >= 25)
    {
        if(PPG_signalQualityCheck(&ir_ppg, &ppgInfoIR) == 0)
            PPG_searchOnsetAndPeak(&ir_ppg, &ppgInfoIR);

        if(PPG_signalQualityCheck(&vs_ppg, &ppgInfoVS) == 0)
            PPG_searchOnsetAndPeak(&vs_ppg, &ppgInfoVS);
    }

#if  ENABLE_DATA_LOG        // Data Log through UART
    if(count > 25)
    {
        switch(scope_type)
        {
            case 0:
                LOG_print(ir_ppg.sample[0]);    // Raw TIA output
                LOG_print(vs_ppg.sample[0]);
                LOG_print(ab_ppg.sample[0]);
                break;
            case 1:
                LOG_print(ir_ppg.signal);       // Full PPG signal
                LOG_print(vs_ppg.signal);
                LOG_print(ab_ppg.signal);
                break;
        }
    }
    if(count >= (25+total_samples))
        __bic_SR_register_on_exit(LPM0_bits);
#endif
}

// Timer3_B7 Interrupt Vector (TBIV) handler
#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector=TIMER3_B1_VECTOR
__interrupt void TIMER3_B1_ISR(void)
#elif defined(__GNUC__)
void __attribute__ ((interrupt(TIMER3_B1_VECTOR))) TIMER3_B1_ISR (void)
#else
#error Compiler not supported!
#endif
{
  switch(__even_in_range(TB3IV,TBIV__TBIFG))
  {
      case TBIV__NONE:                  // No interrupt
          break;
      case TBIV__TBCCR1:                    // CCR1
          // Turn on IR LED, then get ADC samples
          PPG_setLightAndPGAOffset(&ir_ppg);
          __delay_cycles(24*250);           // delay 250us
          PPG_performMeasurement(&ir_ppg);  // 280us+70us
          PPG_turnOffLight(&ir_ppg);
          break;
      case TBIV__TBCCR2:                    // CCR2
          // Turn on RED LED, then get ADC samples
          PPG_setLightAndPGAOffset(&vs_ppg);
          __delay_cycles(24*250);           // delay 250us
          PPG_performMeasurement(&vs_ppg);  // 280us+70us
          PPG_turnOffLight(&vs_ppg);
          break;
      case TBIV__TBCCR3:                    // CCR3
          // Turn off all LED for ambient measurement
          PPG_setLightAndPGAOffset(&ab_ppg);
          __delay_cycles(24*250);           // delay 250us
          PPG_performMeasurement(&ab_ppg);  // 280us+70us
          break;
      case TBIV__TBIFG:                     // overflow
          break;
      default:
          break;
  }
}
