/*
 * algorithm.c
 *
 *  Created on: Aug 4, 2020
 *      Author: a0223954
 */
#include "msp430.h"
#include "algorithm.h"
#include <math.h>

typedef union {
    struct {
        uint16_t low;
        uint16_t high;
    }word;
    int32_t c;
}hw_res32;

//FIR filter coefficient for different cutoff frequency for
//removing 50/60Hz and 100/120Hz
//from the signals



// 6Hz cutoff@100Hz sample rate,23-ord FIR low pass filter coeffs
static const int16_t coeffs[12] =
{
    -67,  //_Q15 type
    -64,
    -46,
    41,
    264,
    670,
    1266,
    2005,
    2788,
    3487,
    3970,
    4143
};


// perfusion ration of ir is higher, use ir signal for bpm calculation
uint8_t calculate_bpm(PPG_Info *ppgInfo)
{
    float bpm;
    uint8_t n;
    uint16_t total_samples;

    n = ppgInfo->log_beats;
    total_samples = ppgInfo->log_onsets[n-1][0] - ppgInfo->log_onsets[0][0];
    bpm = 100.0*60*(n-1)/total_samples;

    return round(bpm);
}

float calculate_pi(PPG_Info *ppgInfo)
{
    uint8_t i,j,k,n;
    int32_t x0,y0,x1,y1,x;
    float interp, mean, std, pi;

    n = ppgInfo->log_beats;
    for(i = 0; i < n-1; i++)
    {
        x0 = ppgInfo->log_onsets[i][0];
        x1 = ppgInfo->log_onsets[i+1][0];
        y0 = ppgInfo->log_onsets[i][1];
        y1 = ppgInfo->log_onsets[i+1][1];

        for(j = i; j < n; j++)
        {
            x = ppgInfo->log_peaks[j][0];
            if((x > x0)&&(x < x1))
            {
                interp = y0 + 1.0*(y1-y0)/(x1-x0)*(x-x0);  // linear interplote
                pi = (interp - ppgInfo->log_peaks[j][1])/ppgInfo->log_onsets[j][1]; // AC/DC = (onset-peak)/onset
                ppgInfo->PI[i] = pi + 0.5*pi*pi;    // ln((I-dI)/I) = -dI/I - 1/2*(dI/I)^2
                break;
            }
        }
    }

    // filter out abnormal pi
    // calculate average
    mean = 0.0f;
    for(i = 0; i < n-1; i++)
        mean += ppgInfo->PI[i];
    mean = mean/(n-1);

    // calculate standar deviation
    std = 0.0f;
    for(i = 0; i < n-1; i++)
        std += (ppgInfo->PI[i]-mean)*(ppgInfo->PI[i]-mean);
    std = sqrt(std/(n-1));

    // pi should be in range (mean-2*std, mean+2*std)
    k = 0;
    pi = 0.0f;
    for(i = 0; i < n-1; i++)
    {
        if((ppgInfo->PI[i]>mean-2*std)&&(ppgInfo->PI[i]<mean+2*std))
        {
            k++;
            pi += ppgInfo->PI[i];
        }
    }

    return pi/k;
}

uint8_t calculate_spo2(float R)
{
    float SpO2;

    // Only for reference
    // accurate result is based on medical clinical data
    SpO2 = -8.6827*R*R - 19.685*R + 112.62;

    return round(SpO2);
}

//Symmetric FIR filter
int32_t fir_filter(int32_t input, FILTER *param)
{
    int i;
    int32_t z;
    int32_t *buf = param->buf;

    buf[param->offset] = input;
    z = mul32(coeffs[11], buf[(param->offset - 11) & 0x1F]);
    for (i = 0;  i < 11;  i++)
        z += mul32(coeffs[i], buf[(param->offset - i) & 0x1F] + buf[(param->offset - 22 + i) & 0x1F]);
    param->offset = (param->offset + 1) & 0x1F;

    return z>>15;
}

// IIR low pass filter to tracking DC level
// y(n) = (x(n) - y(n-1))/(2^K) + y(n-1)
// To ensure there is sufficient accuracy,
// the input is left shift by K
int32_t dc_estimator(int32_t *p, uint16_t x)
{
    *p += ((((int32_t) x << K) - *p) >> K);
    return (*p >> K);
}

int16_t average(int16_t *input, uint16_t size)
{
    uint16_t i;
    int32_t sum;

    sum = 0;
    for(i=0;i<size;i++)
    {
        sum += input[i];
    }

    return sum/size;
}

int32_t mul16(int16_t a, int16_t b)
{
    hw_res32 res;

    MPYS = a;  // Load operand 1 Low into multiplier
    OP2 = b;    // Load operand 2 Low into multiplier

    res.word.low = RES0; // Ready low 16-bits for return
    res.word.high = RES1; // Ready high 16-bits for return

    return res.c;
}

int32_t mul32(int32_t a, int32_t b)
{
    hw_res32 res;

    MPYS32L = a&0xFFFF;  // Load operand 1 Low into multiplier
    MPYS32H = a>>16;     // Load operand 1 High into multiplier
    OP2L = b&0xFFFF;    // Load operand 2 Low into multiplier
    OP2H = b>>16;       // Load operand 2 High, trigger MPY

    res.word.low = RES0; // Ready low 16-bits for return
    res.word.high = RES1; // Ready high 16-bits for return

    return res.c;
}
