/*
 * LCD.c
 *
 *  Created on: Jun 16, 2020
 *      Author: a0223954
 */
#include "lcd.h"
#include "msp430.h"

#define NULL                ((void*)0)

#define LCD_RES_LOW()       P4OUT &= ~BIT6
#define LCD_RES_HIGH()      P4OUT |= BIT6
#define LCD_DCX_LOW()       P4OUT &= ~BIT2
#define LCD_DCX_HIGH()      P4OUT |= BIT2
#define LCD_CSX_LOW()       P4OUT &= ~BIT0
#define LCD_CSX_HIGH()      P4OUT |= BIT0
#define LCD_BLK_ON()        P4OUT |= BIT7
#define LCD_BLK_OFF()       P4OUT &= ~BIT7

// LCD Driver Command Definition
#define LCD_CMD_NOP         0x00
#define LCD_CMD_SWRESET     0x01
#define LCD_CMD_PTLON       0x12
#define LCD_CMD_NORON       0x13
#define LCD_CMD_CASET       0x2A
#define LCD_CMD_RASET       0x2B
#define LCD_CMD_RAMWR       0x2C
#define LCD_CMD_MADCTL      0x36
#define LCD_CMD_COLMOD      0x3A

#define LCD_COLOR_12        0x03
#define LCD_COLOR_16        0x05
#define LCD_COLOR_18        0x06    // default value after power on


// define active area, offset
#define LCD_COLUMN_NUMBER   160
#define LCD_LINE_NUMBER     80
#define LCD_COLUMN_OFFSET   1
#define LCD_LINE_OFFSET     26

#define LCD_SET_COLUMN(x)   (x+LCD_COLUMN_OFFSET)
#define LCD_SET_LINE(y)     (y+LCD_LINE_OFFSET)

#define LCD_XS              LCD_COLUMN_OFFSET
#define LCD_XE              LCD_COLUMN_NUMBER+LCD_COLUMN_OFFSET-1
#define LCD_YS              LCD_LINE_OFFSET
#define LCD_YE              LCD_LINE_NUMBER+LCD_LINE_OFFSET-1


// 8*16
const uint8_t SpO2[][16] =
{
 {0x00,0x00,0x00,0x66,0x6F,0x3F,0x3F,0x1F,0x7E,0xF8,0xFC,0xFC,0xF6,0x66,0x00,0x00},/*"%",0*/

 {0x00,0x00,0x00,0xFC,0xC6,0xC6,0x06,0x0C,0x38,0x60,0xC0,0xC6,0xC6,0x7E,0x00,0x00},/*"S",1*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0x6E,0xC6,0xC6,0xC6,0x6E,0x3E,0x06,0x0F},/*"p",2*/

 {0x00,0x00,0x00,0x3C,0x66,0xC3,0xC3,0xC3,0xC3,0xC3,0xC3,0xC3,0x66,0x3C,0x00,0x00},/*"O",3*/

 {0x00,0x00,0x00,0x7C,0xC6,0xC6,0xC6,0xC0,0x60,0x30,0x18,0x0C,0xC6,0xFE,0x00,0x00},/*"2",4*/
};

const uint8_t PRbpm[][16] =
{
 {0x00,0x00,0x00,0x7F,0xC6,0xC6,0xC6,0xC6,0x7E,0x06,0x06,0x06,0x06,0x0F,0x00,0x00},/*"P",0*/

 {0x00,0x00,0x00,0x7F,0xC6,0xC6,0xC6,0x7E,0x36,0x36,0x66,0x66,0xC6,0xCF,0x00,0x00},/*"R",1*/

 {0x00,0x00,0x00,0x00,0x07,0x06,0x06,0x3E,0x6E,0xC6,0xC6,0xC6,0x6E,0x3E,0x00,0x00},/*"b",2*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3F,0x6E,0xC6,0xC6,0xC6,0x6E,0x3E,0x06,0x0F},/*"p",3*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xB6,0xB6,0xB6,0xB6,0xB6,0xFF,0x00,0x00},/*"m",4*/
};

// 16*32
const uint8_t digit[][64] =
{
 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xE0,0x0F,0xF0,0x3F,
  0xF8,0x3C,0x7C,0x7C,0x7C,0x78,0x3C,0xF8,0x3E,0xF8,0x3E,0xF8,0x3E,0xF8,0x3E,0xF0,
  0x3E,0xF0,0x3E,0xF0,0x3E,0xF0,0x3E,0xF8,0x3E,0xF8,0x3E,0xF8,0x3C,0xF8,0x7C,0x78,
  0x7C,0x7C,0xF8,0x3C,0xF0,0x1F,0xE0,0x0F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},/*"0",0*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x80,0x03,
  0xF8,0x03,0x80,0x03,0x80,0x03,0x80,0x03,0x80,0x03,0x80,0x03,0x80,0x03,0x80,0x03,
  0x80,0x03,0x80,0x03,0x80,0x03,0x80,0x03,0x80,0x03,0x80,0x03,0x80,0x03,0x80,0x03,
  0x80,0x03,0x80,0x03,0xC0,0x07,0xF8,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},/*"1",1*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xE0,0x0F,0x30,0x3C,
  0x18,0x38,0x0C,0x70,0x0C,0x70,0x1C,0x70,0x1C,0x70,0x00,0x70,0x00,0x38,0x00,0x38,
  0x00,0x1C,0x00,0x0E,0x00,0x07,0x80,0x03,0xC0,0x01,0xC0,0x00,0x60,0x60,0x30,0x60,
  0x18,0x60,0x0C,0x70,0xFC,0x3F,0xFC,0x3F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},/*"2",2*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xE0,0x07,0x38,0x0E,
  0x1C,0x1C,0x1C,0x38,0x1C,0x38,0x1C,0x38,0x00,0x38,0x00,0x38,0x00,0x1C,0x00,0x0E,
  0xC0,0x07,0x00,0x1E,0x00,0x38,0x00,0x30,0x00,0x70,0x00,0x70,0x1C,0x70,0x1C,0x70,
  0x1C,0x30,0x1C,0x38,0x38,0x1C,0xE0,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},/*"3",3*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x0E,0x00,0x0E,
  0x00,0x0F,0x00,0x0F,0x80,0x0F,0x80,0x0F,0xC0,0x0E,0x60,0x0E,0x60,0x0E,0x30,0x0E,
  0x30,0x0E,0x18,0x0E,0x1C,0x0E,0x0C,0x0E,0x06,0x0E,0xFE,0x7F,0x00,0x0E,0x00,0x0E,
  0x00,0x0E,0x00,0x0E,0x00,0x0E,0xC0,0x7F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},/*"4",4*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF0,0x7F,0xF0,0x7F,
  0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0x18,0x00,0xD8,0x0F,0x78,0x1C,
  0x38,0x38,0x18,0x30,0x00,0x70,0x00,0x70,0x00,0x70,0x00,0x70,0x1C,0x70,0x1C,0x70,
  0x0C,0x38,0x0C,0x38,0x38,0x1C,0xE0,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},/*"5",5*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x80,0x0F,0xE0,0x38,
  0x70,0x38,0x30,0x38,0x38,0x00,0x18,0x00,0x18,0x00,0x1C,0x00,0xDC,0x0F,0xFC,0x1C,
  0x3C,0x38,0x3C,0x30,0x1C,0x70,0x1C,0x70,0x1C,0x70,0x1C,0x70,0x1C,0x70,0x38,0x70,
  0x38,0x30,0x70,0x38,0xF0,0x1C,0xC0,0x0F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},/*"6",6*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0x7F,0xF8,0x7F,
  0x18,0x30,0x1C,0x18,0x0C,0x18,0x0C,0x0C,0x00,0x0C,0x00,0x06,0x00,0x06,0x00,0x06,
  0x00,0x03,0x00,0x03,0x80,0x01,0x80,0x01,0x80,0x01,0x80,0x01,0xC0,0x01,0xC0,0x01,
  0xC0,0x01,0xC0,0x01,0xC0,0x01,0xC0,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},/*"7",7*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xE0,0x0F,0x70,0x1C,
  0x38,0x38,0x1C,0x70,0x1C,0x70,0x1C,0x70,0x3C,0x70,0x3C,0x30,0xF8,0x38,0xF0,0x0D,
  0xE0,0x07,0x38,0x1F,0x1C,0x3E,0x1C,0x3C,0x0E,0x78,0x0E,0x70,0x0E,0x70,0x0E,0x70,
  0x0E,0x70,0x1C,0x38,0x38,0x1C,0xE0,0x07,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},/*"8",8*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xE0,0x07,0x38,0x0C,
  0x1C,0x18,0x1C,0x38,0x0E,0x30,0x0E,0x70,0x0E,0x70,0x0E,0x70,0x0E,0x70,0x0E,0x70,
  0x1E,0x78,0x1C,0x7C,0x38,0x7E,0xF0,0x73,0x00,0x70,0x00,0x38,0x00,0x38,0x00,0x18,
  0x1C,0x1C,0x1C,0x0E,0x1C,0x07,0xF0,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},/*"9",9*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0xFE,0x7F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00},/*"-",10*/
};

// 8*32 : F(0) i(1) n(2) g(3) e(4) r(5) O(6) u(7) t(8)
const uint8_t FingerOut[][64] =
{
 {0x00,0x00,0x00,0x00,0x00,0x00,0xFF,0xC6,0xC6,0x86,0x86,0x06,0x06,0x06,0x66,0x66,
  0x7E,0x66,0x66,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0x0E,0x0F,0x00,0x00,0x00,0x00},/*"F",0*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x38,0x38,0x00,0x00,0x00,0x00,0x1C,0x18,0x18,
  0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x18,0x38,0x7C,0x00,0x00,0x00,0x00},/*"i",1*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x36,0x7E,0x6E,
  0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xEF,0x00,0x00,0x00,0x00},/*"n",2*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xF8,0xEC,0x6C,
  0x66,0x66,0x66,0x66,0x6C,0x7C,0x1E,0x06,0x1E,0x7C,0xE6,0xC6,0xC6,0xC6,0xC6,0x7C},/*"g",3*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0x6C,
  0xC6,0xC6,0xC6,0xC6,0xFE,0x06,0x06,0x06,0xC6,0xCC,0x7C,0x38,0x00,0x00,0x00,0x00},/*"e",4*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xEE,0xFC,0xFC,
  0x1C,0x1C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x0C,0x3F,0x00,0x00,0x00,0x00},/*"r",5*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x6C,0x6C,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,
  0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0x6C,0x6C,0x38,0x00,0x00,0x00,0x00},/*"O",6*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xE7,0xC6,0xC6,
  0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xC6,0xE6,0xEC,0xFC,0x18,0x00,0x00,0x00,0x00},/*"u",7*/

 {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x18,0x18,0x7E,0x18,0x18,
  0x18,0x18,0x18,0x18,0x18,0x18,0x18,0xD8,0xD8,0xD8,0x78,0x30,0x00,0x00,0x00,0x00},/*"t",8*/
};

//
void LCD_configure(void);
void LCD_spiTransmit(uint8_t byte);
void LCD_sendCommand(uint8_t cmd);
void LCD_sendData(uint8_t data);
void LCD_softwareReset(void);
void LCD_setAddress(uint16_t XS, uint16_t XE, uint16_t YS, uint16_t YE);
void LCD_setColorFormat(uint8_t format);

void LCD_clear(uint8_t x, uint8_t y, Graphics *g);
void LCD_drawRect(uint8_t x, uint8_t y, Graphics *g);
void LCD_drawText(uint8_t x, uint8_t y, Graphics *g, const uint8_t *text);
void LCD_drawLine(uint8_t x0, uint8_t y0, uint8_t y1, Graphics *g);

void LCD_init(void)
{
    // P4.0-LCD_CS,P4.2-LCD_DC
    // P4.6-LCD_RES,P4.7-LCD_BLK

    P4OUT = 0x00;
    P4DIR |= BIT0+BIT2+BIT6+BIT7;

    // SPI, P4.1-LCD_SCL,P4.3-LCD_SDA
    P4SEL0 |= BIT1 | BIT3;                    // set 3-SPI pin as second function
    UCA1CTLW0 |= UCSWRST;                     // **Put state machine in reset**
    UCA1CTLW0 |= UCMST|UCSYNC|UCCKPH|UCMSB;   // 3-pin, 8-bit SPI master
                                              // Clock polarity high, MSB
    UCA1CTLW0 |= UCSSEL__SMCLK;               // Select SMCLK
    UCA1BR0 = 0x01;                           // BRCLK = SMCLK/6
    UCA1BR1 = 0;                              //
    UCA1MCTLW = 0;                            // No modulation
    UCA1CTLW0 &= ~UCSWRST;                    // **Initialize USCI state machine**

    LCD_configure();
}

void LCD_configure(void)
{
    // LCD driver initialize flow
    LCD_CSX_HIGH();

    LCD_RES_LOW();
    __delay_cycles(480);         // longer than 10us
    LCD_RES_HIGH();
    __delay_cycles(29000000);   // longer than 5ms, 120ms for sleep out

    LCD_sendCommand(0x11);       //Sleep Out
    __delay_cycles(2900000);               //DELAY120ms

    LCD_sendCommand(0x21);             //Display inversion on
    //--------------------------------ST7735S Frame rate setting----------------------------------//
    LCD_sendCommand(0xB1);     //Panel Function set
    LCD_sendData(0x05);
    LCD_sendData(0x3A);
    LCD_sendData(0x3A);

    LCD_sendCommand(0xB2);
    LCD_sendData(0x05);
    LCD_sendData(0x3A);
    LCD_sendData(0x3A);

    LCD_sendCommand(0xB3);
    LCD_sendData(0x05);
    LCD_sendData(0x3A);
    LCD_sendData(0x3A);
    LCD_sendData(0x05);
    LCD_sendData(0x3A);
    LCD_sendData(0x3A);

    LCD_sendCommand(0xB4);             //Column inversion
    LCD_sendData(0x03);
    //---------------------------------ST7735S Power setting--------------------------------------//
    LCD_sendCommand(0xC0);         //LCM Control
    LCD_sendData(0x62);
    LCD_sendData(0x02);
    LCD_sendData(0x04);

    LCD_sendCommand(0xC1);             //Power Control Setting
    LCD_sendData(0xC0);

    LCD_sendCommand(0xC2);             //In Normal Mode (Full Colors)
    LCD_sendData(0x0D);
    LCD_sendData(0x00);

    LCD_sendCommand(0xC3);             //In Idle Mode (8-colors)
    LCD_sendData(0x8D);
    LCD_sendData(0x6A);

    LCD_sendCommand(0xC4);             //In Partial Mode + Full colors
    LCD_sendData(0x8D);
    LCD_sendData(0xEE);

    LCD_sendCommand(0xC5);  /*VCOM*/   //VCOM Control 1
    LCD_sendData(0x0E);
    //--------------------------------ST7735S gamma setting---------------------------------------//
    LCD_sendCommand(0xE0);
    LCD_sendData(0x10);
    LCD_sendData(0x0E);
    LCD_sendData(0x02);
    LCD_sendData(0x03);
    LCD_sendData(0x0E);
    LCD_sendData(0x07);
    LCD_sendData(0x02);
    LCD_sendData(0x07);
    LCD_sendData(0x0A);
    LCD_sendData(0x12);
    LCD_sendData(0x27);
    LCD_sendData(0x37);
    LCD_sendData(0x00);
    LCD_sendData(0x0D);
    LCD_sendData(0x0E);
    LCD_sendData(0x10);

    LCD_sendCommand(0xE1);         //Negative Voltage Gamma Contro
    LCD_sendData(0x10);
    LCD_sendData(0x0E);
    LCD_sendData(0x03);
    LCD_sendData(0x03);
    LCD_sendData(0x0F);
    LCD_sendData(0x06);
    LCD_sendData(0x02);
    LCD_sendData(0x08);
    LCD_sendData(0x0A);
    LCD_sendData(0x13);
    LCD_sendData(0x26);
    LCD_sendData(0x36);
    LCD_sendData(0x00);
    LCD_sendData(0x0D);
    LCD_sendData(0x0E);
    LCD_sendData(0x10);

    LCD_sendCommand(0x3A);          //Interface Pixel Format
    LCD_sendData(0x05);             //16BIT mode

    LCD_sendCommand(0x36);          //MX, MY, RGB mode
    LCD_sendData(0x68);             //ʾʾ0XC8

    LCD_sendCommand(0x29);          //Display On
}

void LCD_spiTransmit(uint8_t byte)
{
    while (!(UCA1IFG&UCTXIFG));
    UCA1TXBUF = byte;
}

void LCD_sendCommand(uint8_t cmd)
{
    LCD_DCX_LOW();      // DCX low for command

    LCD_CSX_LOW();
    LCD_spiTransmit(cmd);
    __delay_cycles(24);
    LCD_CSX_HIGH();
}

void LCD_sendData(uint8_t data)
{
    LCD_DCX_HIGH();      // DCX high for data
    LCD_CSX_LOW();
    LCD_spiTransmit(data);
    __delay_cycles(24);
    LCD_CSX_HIGH();
}

void LCD_softwareReset(void)
{
    LCD_sendCommand(LCD_CMD_SWRESET);
}

void LCD_setAddress(uint16_t XS, uint16_t XE, uint16_t YS, uint16_t YE)
{
    // set colum address
    LCD_sendCommand(LCD_CMD_CASET);
    LCD_sendData(XS>>8);
    LCD_sendData(XS);
    LCD_sendData(XE>>8);
    LCD_sendData(XE);
    // set row address
    LCD_sendCommand(LCD_CMD_RASET);
    LCD_sendData(YS>>8);
    LCD_sendData(YS);
    LCD_sendData(YE>>8);
    LCD_sendData(YE);
}

void LCD_setColorFormat(uint8_t format)
{
    LCD_sendCommand(LCD_CMD_COLMOD);
    LCD_sendData(format);
}

void LCD_turnOnBackLight(void)
{
    LCD_BLK_ON();
}

void LCD_turnOffBackLight(void)
{
    LCD_BLK_OFF();
}

void LCD_clearAll(uint16_t color)
{
    Graphics g;

    g.color = color;
    g.w     = 160;
    g.h     = 80;

    LCD_clear(0,0,&g);
}

void LCD_showSpO2(uint8_t SpO2)
{
    Graphics g;

    g.color = CYAN;
    g.w     = 16;
    g.h     = 32;
    // draw num
    LCD_drawText(20,20,&g,digit[SpO2/10]);
    LCD_drawText(36,20,&g,digit[SpO2%10]);
}

void LCD_showPRbpm(uint8_t PRbpm)
{
    uint8_t num;

    Graphics g;

    g.w     = 16;
    g.h     = 32;
    // draw num
    num = PRbpm/100;
    if(num>0)
    {
        g.color = CYAN;
        LCD_drawText(80,20,&g,digit[num]);
    }
    else
    {
        g.color = BLACK;
        LCD_clear(80,20,&g);
    }
    g.color = CYAN;
    num = (PRbpm%100)/10;
    LCD_drawText(96,20,&g,digit[num]);
    num = (PRbpm%100)%10;
    LCD_drawText(112,20,&g,digit[num]);
}

void LCD_showDefault(void)
{
    Graphics g;

    // clear area
    g.color = BLACK;
    g.w     = 108;
    g.h     = 32;
    LCD_drawRect(20,20,&g);

    // draw text
    g.color = CYAN;
    g.w     = 16;
    g.h     = 32;
    // SpO2%:__, PRbpm:__
    LCD_drawText(20,20,&g,digit[10]);
    LCD_drawText(36,20,&g,digit[10]);

    LCD_drawText(96,20,&g,digit[10]);
    LCD_drawText(112,20,&g,digit[10]);
}

void LCD_showFingerOut(void)
{
    Graphics g;

    // draw text
    g.color = CYAN;
    g.w     = 8;
    g.h     = 32;

    LCD_drawText(20,20,&g,FingerOut[0]);
    LCD_drawText(20+8,20,&g,FingerOut[1]);
    LCD_drawText(20+2*8,20,&g,FingerOut[2]);
    LCD_drawText(20+3*8,20,&g,FingerOut[3]);
    LCD_drawText(20+4*8,20,&g,FingerOut[4]);
    LCD_drawText(20+5*8,20,&g,FingerOut[5]);

    LCD_drawText(80,20,&g,FingerOut[6]);
    LCD_drawText(80+8,20,&g,FingerOut[7]);
    LCD_drawText(80+2*8,20,&g,FingerOut[8]);

    g.color = BLACK;
    g.w  = 24;
    LCD_clear(104,20,&g);
}

void LCD_showSignalBar(uint8_t y)
{
    Graphics g;

    g.color = GREEN;
    g.w     = 8;
    g.h     = y;
    // draw num
    LCD_drawRect(5,52-g.h,&g);
    g.color = BLACK;
    g.h     = 24-g.h;
    LCD_drawRect(5,28,&g);
}

void LCD_showSignal(uint8_t x,uint8_t y)
{
    static uint8_t prev_y;
    Graphics g;

    if(x == 0)
    {
        prev_y = y;
        return;
    }

    g.color = CYAN;
    LCD_drawLine(x-1,79-prev_y,79-y,&g);

    g.w = 2;
    g.h = 25;
    g.color = BLACK;
    if(x==159)
        LCD_clear(0,55,&g);
    else
        LCD_clear(x+1,55,&g);

    prev_y = y;
}

void LCD_showBackground(void)
{
    Graphics g;

    // display RED
    LCD_clearAll(BLACK);

    g.color = YELLOW;
    g.w     = 8;
    g.h     = 16;
    // draw %SpO2
    LCD_drawText(20,0,&g,SpO2[0]);
    LCD_drawText(28,0,&g,SpO2[1]);
    LCD_drawText(36,0,&g,SpO2[2]);
    LCD_drawText(44,0,&g,SpO2[3]);
    LCD_drawText(52,0,&g,SpO2[4]);

    // draw --
    LCD_drawText(20,20,&g,digit[10]);
    LCD_drawText(36,20,&g,digit[10]);

    // draw PRbpm
    LCD_drawText(80,0,&g,PRbpm[0]);
    LCD_drawText(80+8,0,&g,PRbpm[1]);
    LCD_drawText(80+16,0,&g,PRbpm[2]);
    LCD_drawText(80+24,0,&g,PRbpm[3]);
    LCD_drawText(80+32,0,&g,PRbpm[4]);

    // draw ---
    LCD_drawText(80,20,&g,digit[10]);
    LCD_drawText(96,20,&g,digit[10]);
    LCD_drawText(112,20,&g,digit[10]);
}

void LCD_clear(uint8_t x, uint8_t y, Graphics *g)
{
    uint8_t i,j;
    uint8_t xs,xe,ys,ye;

    xs = LCD_SET_COLUMN(x);
    xe = xs + g->w - 1;
    ys = LCD_SET_LINE(y);
    ye = ys + g->h - 1;

    LCD_setAddress(xs,xe,ys,ye);
    LCD_sendCommand(LCD_CMD_RAMWR);    // Memory write

    for(i = 0; i < g->w; i++)
    {
        for(j = 0; j < g->h; j++)
        {
            LCD_sendData(g->color>>8);
            LCD_sendData(g->color);
        }
    }
}

void LCD_drawText(uint8_t x, uint8_t y, Graphics *g, const uint8_t *text)
{
    uint8_t i,j;
    uint8_t temp;
    uint8_t xs,xe,ys,ye;

    xs = LCD_SET_COLUMN(x);
    xe = xs + g->w - 1;
    ys = LCD_SET_LINE(y);
    ye = ys + g->h - 1;

    LCD_setAddress(xs,xe,ys,ye);
    LCD_sendCommand(LCD_CMD_RAMWR);    // Memory write

    for(i = 0; i < (g->w*g->h)/8; i++)
    {
        temp=*text;
        for(j = 0; j < 8; j++)
        {
            if(temp&0x01)
            {

                LCD_sendData(g->color>>8);
                LCD_sendData(g->color);
            }
            else
            {
                LCD_sendData(BLACK);
                LCD_sendData(BLACK);
            }
            temp>>=1;
        }
        text++;
    }
}

void LCD_drawRect(uint8_t x, uint8_t y, Graphics *g)
{
    uint8_t i,j;
    uint8_t xs,xe,ys,ye;

    xs = LCD_SET_COLUMN(x);
    xe = xs + g->w - 1;
    ys = LCD_SET_LINE(y);
    ye = ys + g->h - 1;

    LCD_setAddress(xs,xe,ys,ye);
    LCD_sendCommand(LCD_CMD_RAMWR);    // Memory write

    for(i = 0; i < g->w; i++)
    {
        for(j = 0; j < g->h; j++)
        {
            LCD_sendData(g->color>>8);
            LCD_sendData(g->color);
        }
    }
}

void LCD_drawLine(uint8_t x0, uint8_t y0, uint8_t y1, Graphics *g)
{
    uint8_t i,st,sp,mid;
    uint16_t color1,color2;
    uint8_t xs,xe,ys,ye;

    mid = (y0+y1)/2;
    if(y1 >= y0)
    {
        st = y0;
        sp = y1;
        color1 = g->color;
        color2 = BLACK;
    }
    else
    {
        st = y1;
        sp = y0;
        color1 = BLACK;
        color2 = g->color;
    }

    xs = LCD_SET_COLUMN(x0);
    xe = LCD_SET_COLUMN(x0+1);
    ys = LCD_SET_LINE(st);
    ye = LCD_SET_LINE(sp);

    LCD_setAddress(xs,xe,ys,ye);
    LCD_sendCommand(LCD_CMD_RAMWR);    // Memory write

    for(i = st; i <= sp; i++)
    {
        if(i <= mid)
        {
            LCD_sendData(color1>>8);
            LCD_sendData(color1);
            LCD_sendData(color2>>8);
            LCD_sendData(color2);
        }
        else
        {
            LCD_sendData(color2>>8);
            LCD_sendData(color2);
            LCD_sendData(color1>>8);
            LCD_sendData(color1);
        }
    }
}

void LCD_drawPicture(void)
{

}
