/*
 * uart_print.c
 *
 *  Created on: Jul 1, 2020
 *      Author: a0223954
 */
#include "msp430.h"
#include "log.h"

//#define UART_PRINTF

#ifdef UART_PRINTF
#include <stdio.h>
#include <string.h>

int fputc(int _c, register FILE *_fp);
int fputs(const char *_ptr, register FILE *_fp);
#endif

uint8_t RxBuf[3];
uint8_t scope_type;
uint16_t total_samples;

void LOG_init(void)
{
    // Configure UART pins
    P4SEL0 |= BIT2;             // RXD
    P4SEL0 |= BIT3;             // TXD
    // Configure UART
    UCA1CTLW0 |= UCSWRST;
    UCA1CTLW0 |= UCSSEL__SMCLK;

    // Baud Rate calculation
    // 6000000/(16*115200) = 3.255
    // Fractional portion = 0.255
    // User's Guide Table 17-4: UCBRSx = 0x44
    // UCBRFx = int ( (3.255-3)*16) = 4
    UCA1BR0 = 3;                              // 6000000/16/115200
    UCA1BR1 = 0x00;
    UCA1MCTLW = 0x4400 | UCOS16 | UCBRF_4;
    UCA1CTLW0 &= ~UCSWRST;                    // Initialize eUSCI

    UCA1IE |= UCRXIE;                         // Enable USCI_A1 RX interrupt
}

// UART print
void LOG_print(uint32_t data)
{
    while(!(UCA1IFG&UCTXIFG));
    UCA1TXBUF = data&0xFF;

    while(!(UCA1IFG&UCTXIFG));
    UCA1TXBUF = (data&0xFF00)>>8;

    while(!(UCA1IFG&UCTXIFG));
    UCA1TXBUF = (data&0xFF0000)>>16;
}

void LOG_waitCommand(void)
{
    // wait for UART command
    UCA1IE |= UCRXIE;                    // Enable USCI_A1 RX interrupt
    __bis_SR_register(LPM0_bits | GIE);  // Enter LPM0 w/ interrupts
    __no_operation();
}

// UART RX Interrupt
#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector=USCI_A1_VECTOR
__interrupt void USCI_A1_ISR(void)
#elif defined(__GNUC__)
void __attribute__ ((interrupt(USCI_A1_VECTOR))) USCI_A1_ISR (void)
#else
#error Compiler not supported!
#endif
{
    uint8_t temp;
    static uint8_t i;

    switch(__even_in_range(UCA1IV,USCI_UART_UCTXCPTIFG))
    {
        case USCI_NONE: break;
        case USCI_UART_UCRXIFG:
            temp = UCA1RXBUF;
            if((i==0)&&(temp!=0x55))
                break;
            RxBuf[i] = temp;
            i++;
            if(i==3)
            {
                i = 0;
                if(RxBuf[0]==0x55)
                {
                    UCA1IE &= ~UCRXIE;
                    scope_type = RxBuf[1];
                    total_samples = RxBuf[2]*100;
                    __bic_SR_register_on_exit(LPM0_bits);
                }
            }
            break;
        case USCI_UART_UCTXIFG: break;
        case USCI_UART_UCSTTIFG: break;
        case USCI_UART_UCTXCPTIFG: break;
        default: break;
    }
}

#ifdef UART_PRINTF
int fputc(int _c, register FILE *_fp)
{
    while(!(UCA1IFG&UCTXIFG));
    UCA1TXBUF = (unsigned char) _c;

    return((unsigned char)_c);
}

int fputs(const char *_ptr, register FILE *_fp)
{
    unsigned int i, len;

    len = strlen(_ptr);

    for(i=0 ; i<len ; i++)
    {
        while(!(UCA1IFG&UCTXIFG));
        UCA1TXBUF = (unsigned char) _ptr[i];
    }

    return len;
}
#endif
