#include "msp430.h"
#include "hal.h"
#include "log.h"
#include "lcd.h"
#include "PPG.h"


void enter_sleep_mode(void);

PPG_Results ppgResults;

/**
 * main.c
 */
int main(void)
{
    WDTCTL = WDTPW | WDTHOLD;               // Stop WDT

    // Disable the GPIO power-on default high-impedance mode
    // to activate previously configured port settings
    PM5CTL0 &= ~LOCKLPM5;

#if !ENABLE_DATA_LOG
    // Determine whether we are coming out of an LPMx.5 or a regular RESET.
    if (SYSRSTIV != SYSRSTIV_LPM5WU)
    {
        // Device powered up from a cold start.
        // It configures the device and puts the device into LPM4.5
        enter_sleep_mode();
    }
#endif

    // Hardware init
    hal_init();
    PPG_init();


#if ENABLE_DATA_LOG
    LOG_init();

    for(;;)
    {
        LOG_waitCommand();

        PPG_appStart();

        PPG_appSleep();

        PPG_appStop();
    }

#else
    // display RED
    LCD_init();
    LCD_turnOnBackLight();
    LCD_showBackground();

    // Start PPG measurement
    PPG_appStart();

    for(;;)
    {
        PPG_appHandler(&ppgResults);

        switch(ppgResults.state)
        {
            case PPG_INITIAL:
                PPG_appReInit(&ppgResults);
                LCD_showDefault();
                // re-initialize
                break;
            case PPG_FINGER_OUT:
                // Finger Out and enter sleep mode
                if(ppgResults.new_state == 1)
                {
                    ppgResults.new_state = 0;
                    LCD_clearAll(BLACK);
                    LCD_showBackground();
                    LCD_showFingerOut();
                }
                break;
            case PPG_MEASURING:
                LCD_showSignalBar(log_signal);
                LCD_showSignal(log_count,log_signal);
                if(ppgResults.flag == 1)
                {
                    ppgResults.flag = 0;
                    // update SpO2 and bpm
                    LCD_showSpO2(ppgResults.SpO2);
                    LCD_showPRbpm(ppgResults.bpm);
                }
                break;
            case PPG_TIME_OUT:
                PPG_appStop();
                LCD_turnOffBackLight();
                //enter sleep mode
                enter_sleep_mode();
                break;
        }
    }
#endif
}

void enter_sleep_mode(void)
{
    // Configure GPIO to save power
    P1DIR = 0xFF; P2DIR = 0xFF; P3DIR = 0xFF; P4DIR = 0xFF;
    P1OUT = 0x00; P2OUT = 0x00; P3OUT = 0x00; P4OUT = 0x00;
    P1IFG = 0x00; P2IFG = 0x00; P3IFG = 0x00; P4IFG = 0x00;

    // Configure P2.0 for wake-up from LPM4.5
    P2DIR &= ~BIT0;                     // Configure P2.0 as input direction pin
    P2OUT |= BIT0;                      // Configure P2.0 as pulled-up
    P2REN |= BIT0;                      // P2.0 pull-up register enable
    P2IES |= BIT0;                      // P2.0 Hi/Low edge
    P2IFG = 0;                          // Clear all P2 interrupt flags
    P2IE |= BIT0;                       // P2.0 interrupt enabled

    // Explicitly clear RTC control registers to disable it
    // just incase if the RTC was previously enabled
    RTCCTL = 0;

    PMMCTL0_H = PMMPW_H;                // Open PMM Registers for write
    PMMCTL0_L &= ~(SVSHE);              // Disable high-side SVS
    PMMCTL0_L |= PMMREGOFF;             // and set PMMREGOFF
    PMMCTL0_H = 0;                      // Lock PMM Registers

    // Enter LPM4 Note that this operation does not return. The LPM4.5
    // will exit through a RESET event, resulting in a re-start
    // of the code.
    __bis_SR_register(LPM4_bits | GIE);
}

