//
// Copyright (c) 2012-2021 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
//
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
//
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
//
//
//*****************************************************************************
/*
 * fee.c
 *
 *--------------------------------------------------------------
 * Revision History
 *--------------------------------------------------------------
 * Version Date       Author    Change ID     Description
 *--------------------------------------------------------------
 * 00.01.00 23Mar2021 RM Crosby 0000000000000 Initial Version
 */

#include <stdint.h>
#include <stdbool.h>
#include <stdlib.h>
#include <stddef.h>
#include <feeConfig.h>
#include "fee.h"
#include "inc/hw_eeprom.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "driverlib/sysctl.h"
#include "driverlib/eeprom.h"
#include "driverlib/crc.h"

static uint32_t readEE(uint16_t eepromLocation);
static void writeEE(uint16_t eepromLocation, uint32_t word);
static int32_t sortListCompareFunc(const void *pvFirst, const void *pvSecond);
static uint32_t calculateRAMCRC(const uint32_t *bufferAddress, uint8_t count);
static uint32_t calculateEepromCRC(uint16_t eepromAddress, uint8_t count);
static uint32_t calculateBlankCRC(uint8_t count);
static uint16_t imageAddress(uint8_t ui8Dataset, uint8_t ui8ImageNumber);
static uint32_t imageCount3(uint8_t ui8Dataset, uint8_t ui8ImageNumber);
static void createSortedImageList(uint8_t ui8Dataset,
                                  tFeeImageSortList *psList);
static uint32_t offsetFromAddress(uint16_t ui16Address);
static uint32_t blockFromAddress(uint16_t ui16Address);
static void executeStateMachine(void);
static void retryLastOperation(void);

const tFeeVersion sFeeVersion =
{
     1u, /* Major Version */
     0u  /* Minor Version */
};

const uint16_t pui16FirstSectorDatasetStart[NUMBER_OF_DATASETS] =
{
    (EE_STARTING_BLOCK * EE_BYTES_PER_BLOCK)
#if(NUMBER_OF_DATASETS > 1u)
        ,
    ((EE_STARTING_BLOCK + 16u) * EE_BYTES_PER_BLOCK)
#endif
#if(NUMBER_OF_DATASETS > 2u)
        ,
    ((EE_STARTING_BLOCK + 32u) * EE_BYTES_PER_BLOCK)
#endif
#if(NUMBER_OF_DATASETS > 3u)
        ,
    ((EE_STARTING_BLOCK + 48u) * EE_BYTES_PER_BLOCK)
#endif
#if(NUMBER_OF_DATASETS > 4u)
        ,
    ((EE_STARTING_BLOCK + 64u) * EE_BYTES_PER_BLOCK)
#endif
#if(NUMBER_OF_DATASETS > 5u)
        ,
    ((EE_STARTING_BLOCK + 80u) * EE_BYTES_PER_BLOCK)
#endif
};

const tFeeDataset psDatasets[NUMBER_OF_DATASETS] =
{
    { SIZE_DATASET0, IMAGES_PER_SECTOR0 }
#if(NUMBER_OF_DATASETS > 1u)
    ,
    {SIZE_DATASET1, IMAGES_PER_SECTOR1}
#endif
#if(NUMBER_OF_DATASETS > 2u)
    ,
    {SIZE_DATASET2, IMAGES_PER_SECTOR2}
#endif
#if(NUMBER_OF_DATASETS > 3u)
    ,
    {SIZE_DATASET3, IMAGES_PER_SECTOR3}
#endif
#if(NUMBER_OF_DATASETS > 4u)
    ,
    {SIZE_DATASET4, IMAGES_PER_SECTOR4}
#endif
#if(NUMBER_OF_DATASETS > 5u)
    ,
    {SIZE_DATASET5, IMAGES_PER_SECTOR5}
#endif
};

/* Static values that are used between calls to FEE functions */
static uint32_t g_ui32CurrentImageCount;
static uint32_t g_ui32CurrentCRC;
static uint32_t *g_pui32WriteBuffer;
static uint16_t g_ui16CurrentImageAddress;
static uint16_t g_ui16WriteBufferIndex;
static uint8_t g_ui8WriteBufferSize;
static uint8_t g_pui8CurrentImageNumber[NUMBER_OF_DATASETS];
static uint8_t g_ui8CurrentImage;
static uint8_t g_ui8CurrentDataset;
static uint8_t g_ui8RetryCount = MAX_RETRIES;
static tFeeStatus g_eFeeStatus = FEE_UNINIT;
static tFeeJobResult g_eFeeResult = FEE_JOB_OK;
static tEEState g_eFeeState = EEStateIdle;

/* FeeInit
 * This function initializes the high endurance EEPROM functions. It calls
 * EEPROMInit() which resets and initializes the EEPROM module. It identifies
 * the most recent image for each of the datasets. If any dataset does not have
 * a valid image, it returns FEE_NOT_OK. If all datasets have a valid image, but
 * one or more of the datasets had to choose and older image because the image
 * with the highest COUNT3 value was corrupt, then the function returns FEE_OLD
 * and sets the module state to FEE_IDLE. Otherwise it returns FEE_OK and sets
 * the module state from FEE_UINIT to FEE_IDLE. If the function returns
 * FEE_NOT_OK, the program should check all of the datasets by calling
 * FeeCheckDatasetValid() to identify datasets without any valid data. Calling
 * FeeFormat() for that dataset will reformat the dataset, but the old data will
 * be lost.
 */
tFeeReturn FeeInit(void)
{
    uint8_t ui8Dataset;
    tFeeReturn eFeeReturn;
    tFeeReturn eDatasetValidReturn;

    // Enable the EEPROM module.
    SysCtlPeripheralEnable(SYSCTL_PERIPH_EEPROM0);
    // Wait for the EEPROM module to be ready.
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_EEPROM0))
    {
    }
    // Do not check if the EEPROM Initialization returned an error
    // Let the FEE driver try to recover
    EEPROMInit();
    // Initialize the CRC (while EEPROM does its thing)
    SysCtlPeripheralEnable(SYSCTL_PERIPH_CCM0);
    eFeeReturn = FEE_OK;
    g_eFeeStatus = FEE_IDLE;
    for(ui8Dataset = 0u; ui8Dataset < NUMBER_OF_DATASETS; ui8Dataset++)
    {
        eDatasetValidReturn = FeeCheckDatasetValid(ui8Dataset);
        if((eDatasetValidReturn == FEE_OLD) && (eFeeReturn == FEE_OK))
        {
            eFeeReturn = FEE_OLD;
        }
        else if(eDatasetValidReturn == FEE_NOT_OK)
        {
            eFeeReturn = FEE_NOT_OK;
            g_eFeeStatus = FEE_UNINIT;
        }
    }
    return(eFeeReturn);
}

/* FeeFormat
 * This function forces an initialization of one dataset of the FEE. Any data in
 * that dataset is lost. The dataset will have one valid image with the data all
 * set to 0xFFFFFFFF. It sets the module state to FEE_BUSY and the initializa-
 * tion is finished by calls to FeeMainFunction(). Once reformatted, the
 * counters will be reset. At the end of formatting the dataset,
 * FeeMainFunction() will set the FEE status from FEE_UNINIT to FEE_IDLE only if
 * all datasets are now valid.
 */
void FeeFormat(uint8_t ui8Dataset)
{
    uint16_t ui16Address;

    if(EEStateIdle == g_eFeeState)
    {
        g_eFeeState = EEStateInitializeDataset;
        g_eFeeStatus = FEE_BUSY;
        g_eFeeResult = FEE_JOB_PENDING;
        g_ui8CurrentDataset = ui8Dataset;
        g_ui8CurrentImage = 0u;
        ui16Address = imageAddress(ui8Dataset, g_ui8CurrentImage);
        writeEE((ui16Address + COUNT1_OFFSET), (uint32_t)g_ui8CurrentImage);
    }
}

/* FeeGetJobResult
 * This function returns the state of the last job request.
 * - FEE_JOB_OK: The last job has finished successfully.
 * - FEE_JOB_PENDING: The last job is waiting for execution or is currently
 *   being executed.
 * - FEE_JOB_FAILED: The last job failed.
 */
tFeeJobResult FeeGetJobResult(void)
{
    return g_eFeeResult;
}

/* FeeGetStatus
 * This function returns the status of the software FEE the state-machine.
 * - FEE_UNINIT: The FEE driver has not been successfully initialized.
 * - FEE_IDLE: The FEE driver is currently idle.
 * - FEE_BUSY: The FEE driver is currently busy.
 */
tFeeStatus FeeGetStatus(void)
{
    return g_eFeeStatus;
}

/* FeeRead
 * This function will read the most current version of the data. The buffer must
 * be large enough to hold the dataset. This function requires the state-machine
 * to be idle. A call to FeeRead() when the state-machine is not idle will
 * return FEE_NOT_OK. Reading a dataset whose most recent image was corrupted
 * will return FEE_OLD. Reads are performed synchronously and do not require
 * calls to FeeMainFunction() to complete.
 */
tFeeReturn FeeRead(uint8_t ui8Dataset, uint32_t pui32DataBuffer[])
{
    uint32_t ui32RamCrc, ui32EeCrc;
    uint16_t ui16EeAddress;
    uint16_t ui16DataAddress;
    uint8_t ui8ImageSize;
    uint8_t ui8Index;
    uint8_t ui8ImageNumber;
    tFeeReturn eReturnValue;

    ui16EeAddress = imageAddress(ui8Dataset, g_pui8CurrentImageNumber[ui8Dataset]);
    ui8ImageSize = psDatasets[ui8Dataset].ui8Words;
    if(g_eFeeStatus != FEE_IDLE)
    {
        eReturnValue = FEE_NOT_OK;
    }
    else
    {
        g_eFeeStatus = FEE_BUSY;
        /* point to first word of the data */
        ui16DataAddress = ui16EeAddress + DATA_OFFSET;
        for(ui8Index = 0u; ui8Index < ui8ImageSize; ui8Index++)
        {
            pui32DataBuffer[ui8Index] = readEE(ui16DataAddress);
            ui16DataAddress += 4u;
        }
        ui32RamCrc = calculateRAMCRC(pui32DataBuffer, ui8ImageSize);
        ui32EeCrc = readEE(ui16EeAddress + CRC_OFFSET);
        if(ui32RamCrc == ui32EeCrc)
        {
            eReturnValue = FEE_OK;
            g_eFeeResult = FEE_JOB_OK;
        }
        else
        {
            eReturnValue = FEE_NOT_OK; /* will change if we find a good image */
            g_eFeeResult = FEE_JOB_FAILED;
            /* try to find a good image decrementing through the images */
            ui8ImageNumber = g_pui8CurrentImageNumber[ui8Dataset];
            do
            {
                if(0u == ui8ImageNumber)
                {
                    ui8ImageNumber = (psDatasets[ui8Dataset].ui8Images * 2u) - 1u;
                }
                else
                {
                    ui8ImageNumber--;
                }
                ui16EeAddress = imageAddress(ui8Dataset, ui8ImageNumber);
                ui32EeCrc = readEE(ui16EeAddress + CRC_OFFSET);
                ui16EeAddress += 16u; /* point to first word of the data */
                for(ui8Index = 0u; ui8Index < ui8ImageSize; ui8Index++)
                {
                    pui32DataBuffer[ui8Index] = readEE(ui16EeAddress);
                    ui16EeAddress += 4u;
                }
                ui32RamCrc = calculateRAMCRC(pui32DataBuffer, ui8ImageSize);
                if(ui32RamCrc == ui32EeCrc)
                {
                    eReturnValue = FEE_OLD;
                    g_eFeeResult = FEE_JOB_OK;
                    /* set this image to the current image for future reads,
                     * this will end the while loop
                     */
                    g_pui8CurrentImageNumber[ui8Dataset] = ui8ImageNumber;
                }
            }while(ui8ImageNumber != g_pui8CurrentImageNumber[ui8Dataset]);
        }
        g_eFeeStatus = FEE_IDLE;
    }
    return eReturnValue;
}

/* FeeWrite
 * This function starts the process of writing new data into the EEPROM. The
 * buffer must be held constant until the write is complete. This is determined
 * by calls to FeeGetStatus() with a return of FEE_IDLE. The function
 * FeeMainFunction() is used to complete the process of writing the new data and
 * updating the control words.
 */
tFeeReturn FeeWrite(uint8_t ui8Dataset, uint32_t  pui32DataBuffer[])
{
    tFeeReturn eReturnValue;
    uint8_t ui8TotalImages;
    uint16_t ui16Address;

    g_ui16CurrentImageAddress = imageAddress(ui8Dataset,
                                       g_pui8CurrentImageNumber[ui8Dataset]);
    if(g_eFeeStatus != FEE_IDLE)
    {
        eReturnValue = FEE_NOT_OK;
    }
    else
    {
        /* The new image will be the next available image number which will put
         * it in the other sector
         */
        g_eFeeStatus = FEE_BUSY;
        g_eFeeResult = FEE_JOB_PENDING;
        /* images in 2 sectors */
        ui8TotalImages = psDatasets[ui8Dataset].ui8Images << 1u;
        g_ui8CurrentImage = g_pui8CurrentImageNumber[ui8Dataset] + 1u;
        if(g_ui8CurrentImage == ui8TotalImages)
        {
            g_ui8CurrentImage = 0u;
        }
        g_ui32CurrentImageCount =
                readEE(g_ui16CurrentImageAddress + COUNT3_OFFSET) + 1u;
        ui16Address = imageAddress(ui8Dataset, g_ui8CurrentImage);
        /* alternate which to write first, count1 or count2 */
        if(((g_ui32CurrentImageCount / ((uint32_t)ui8TotalImages * 6u)) & 1u)
                == 1u)
        {
            writeEE((ui16Address +  COUNT2_OFFSET), g_ui32CurrentImageCount);
        }
        else
        {
            writeEE((ui16Address + COUNT1_OFFSET), g_ui32CurrentImageCount);
        }
        g_ui8CurrentDataset = ui8Dataset;
        g_ui16CurrentImageAddress = ui16Address;
        g_pui32WriteBuffer = pui32DataBuffer;
        g_ui8WriteBufferSize = psDatasets[g_ui8CurrentDataset].ui8Words;
        g_ui32CurrentCRC = calculateRAMCRC(pui32DataBuffer, g_ui8WriteBufferSize);
        g_eFeeState = EEStateProgrammingFirstCounter;
        eReturnValue = FEE_OK;

    }
    return eReturnValue;
}

/* FeeGetDatasetCounter
 * This function returns the current counter valid of the selected dataset.
 * This function is to be called only after a successful call to FeeInit(). The
 * counter value can be used as an indication of the number of times that a
 * dataset has been written. The counter is reset if the dataset is reformatted.
 */
uint32_t FeeGetDatasetCounter(uint8_t ui8Dataset)
{
    uint16_t ui16Address;

    ui16Address = imageAddress(ui8Dataset, g_pui8CurrentImageNumber[ui8Dataset]);
    return readEE(ui16Address + COUNT3_OFFSET);
}

/* FeeGetVersionInformation
 * This function returns a two-byte structure with a major and a minor version
 * number.
 */
tFeeVersion FeeGetVersionInformation(void)
{
    return sFeeVersion;
}

/* FeeMainFunction
 * This function must be called repeatedly to complete a FeeWrite() or
 * FeeFormat() operation. Each call to FeeMainFunction() will only program one
 * word, four bytes. An additional four words must be programmed for the control
 * data and the CRC value. Calls to FeeMainFunction() while the state is
 * FEE_BUSY will return immediately. This function can be repeatedly called by
 * the user application code or by the flash interrupt service routine.
 */
void FeeMainFunction(void)
{
#if defined(EEPROM_INTERRUPTS)
    EEPROMIntClear(EEPROM_INT_PROGRAM);
    EEPROMIntDisable(EEPROM_INT_PROGRAM);
#endif
    /* if eeprom hardware is busy, return immediately */
    if((HWREG(EEPROM_EEDONE) & EEPROM_EEDONE_WORKING) != EEPROM_EEDONE_WORKING)
    {
        /* check for previous good program or erase */
        if((HWREG(EEPROM_EESUPP) &
                (EEPROM_EESUPP_ERETRY | EEPROM_EESUPP_PRETRY)) == 0u)
        {
            g_ui8RetryCount = MAX_RETRIES;
            executeStateMachine(); /* do next program operation */
        }
        else
        {
            /* if previous operation failed, retry 7 times unless count3  */
            if(g_ui8RetryCount != 0u)
            {
                /* reset the module, try again */
                g_ui8RetryCount--;
                SysCtlPeripheralReset(SYSCTL_PERIPH_EEPROM0);
                while(!SysCtlPeripheralReady(SYSCTL_PERIPH_EEPROM0))
                {
                }
                EEPROMInit();
                retryLastOperation();
            }
            else
            {
                g_eFeeResult = FEE_JOB_FAILED;
                g_eFeeStatus = FEE_IDLE;
                g_eFeeState = EEStateIdle;
            }

        }
    }
}

static void executeStateMachine(void)
{
    uint16_t ui16Address;
    uint8_t ui8I;
    uint8_t ui8TotalImages;

    switch(g_eFeeState)
    {
    case EEStateIdle:
        break;
    case EEStateProgrammingFirstCounter:
        /* one counter programmed, determine which and program other */
        if(readEE(g_ui16CurrentImageAddress + COUNT1_OFFSET) ==
                g_ui32CurrentImageCount)
        {
            writeEE((g_ui16CurrentImageAddress + COUNT2_OFFSET),
                    g_ui32CurrentImageCount);
        }
        else
        {
            writeEE((g_ui16CurrentImageAddress + COUNT1_OFFSET),
                    g_ui32CurrentImageCount);
        }
        g_eFeeState = EEStateProgrammingSecondCounter;
        break;
    case EEStateProgrammingSecondCounter:
        /* counters 1 and 2 are programmed, now start the data */
        g_ui16WriteBufferIndex = 0u;
        writeEE(g_ui16CurrentImageAddress + DATA_OFFSET,
                g_pui32WriteBuffer[g_ui16WriteBufferIndex]);
        g_ui16WriteBufferIndex++;
        g_eFeeState = EEStateProgrammingData;
        break;
    case EEStateProgrammingData:
        if(g_ui16WriteBufferIndex < g_ui8WriteBufferSize)
        {
            writeEE((g_ui16CurrentImageAddress + DATA_OFFSET +
                    (g_ui16WriteBufferIndex * 4u)),
                    g_pui32WriteBuffer[g_ui16WriteBufferIndex]);
            g_ui16WriteBufferIndex++;
        }
        else
        {
            writeEE((g_ui16CurrentImageAddress + CRC_OFFSET ), g_ui32CurrentCRC);
            g_eFeeState = EEStateProgrammingCRC;
        }
        break;
    case EEStateProgrammingCRC:
        /* verify data and CRC */
        ui16Address = imageAddress(g_ui8CurrentDataset, g_ui8CurrentImage);
        if(calculateEepromCRC((ui16Address + DATA_OFFSET),
                psDatasets[g_ui8CurrentDataset].ui8Words) !=
                readEE(ui16Address + CRC_OFFSET))
        {
            /* bad data or bad CRC */
            g_eFeeResult = FEE_JOB_FAILED;
            g_eFeeStatus = FEE_IDLE;
            g_eFeeState = EEStateIdle;
        }
        else
        {
            g_ui8RetryCount = 0u; /* no retries on count3 */
            writeEE((ui16Address + COUNT3_OFFSET), g_ui32CurrentImageCount);
            g_eFeeState = EEStateProgrammingThirdCounter;
        }
        break;
    case EEStateProgrammingThirdCounter:
        /* Check that counter 3 is programmed properly*/
        ui16Address = imageAddress(g_ui8CurrentDataset, g_ui8CurrentImage);
        if((readEE(ui16Address + COUNT3_OFFSET)) == g_ui32CurrentImageCount)
        {
            g_pui8CurrentImageNumber[g_ui8CurrentDataset] = g_ui8CurrentImage;
            g_eFeeResult = FEE_JOB_OK;
        }
        else
        {
            g_eFeeResult = FEE_JOB_FAILED;
        }
        g_eFeeStatus = FEE_IDLE;
        g_eFeeState = EEStateIdle;
        break;
    case EEStateInitializeDataset:
    /* uses g_ui8CurrentDataset to identify which dataset to initialize
     * Initilize image 0 with data all 0xFFFFFFFFu. Then invalidate the
     * other images by setting their count3 value to 0xFFFFFFFFu
     */

        /* counter 1, 2 and 3 programmed 0, 1, ... for each image */
        ui16Address = imageAddress(g_ui8CurrentDataset, g_ui8CurrentImage);
        if(readEE(ui16Address + COUNT2_OFFSET) != g_ui8CurrentImage)
        {
            writeEE((ui16Address + COUNT2_OFFSET), (uint32_t)g_ui8CurrentImage);
        }
        else if(readEE(ui16Address + COUNT3_OFFSET) != g_ui8CurrentImage)
        {
            writeEE((ui16Address + COUNT3_OFFSET), (uint32_t)g_ui8CurrentImage);
            g_ui8WriteBufferSize = psDatasets[g_ui8CurrentDataset].ui8Words;
            g_ui16WriteBufferIndex = 0u;
        }
        else /* program the data */
        {
            if(g_ui8WriteBufferSize != 0u)
            {
                ui16Address = ui16Address + DATA_OFFSET +
                        (g_ui16WriteBufferIndex * 4u);
                writeEE(ui16Address, 0xFFFFFFFFu);
                g_ui16WriteBufferIndex++;
                g_ui8WriteBufferSize--;
            }
            else /*  do CRC */
            {
                g_ui32CurrentCRC =
                   calculateBlankCRC(psDatasets[g_ui8CurrentDataset].ui8Words);
                writeEE((ui16Address + CRC_OFFSET), g_ui32CurrentCRC);
                g_ui32CurrentImageCount = g_ui8CurrentImage;
                g_eFeeState = EEStateFinishFormat;
            }
        }
        break;
    case EEStateFinishFormat:
        /* Mark each of the images other than image 0 as invalid by setting
         * counter3 to 0xFFFFFFFFu
         */
        g_ui8CurrentImage++;
        ui8TotalImages = (psDatasets[g_ui8CurrentDataset].ui8Images * 2u);
        if(g_ui8CurrentImage != ui8TotalImages)
        {
            ui16Address = imageAddress(g_ui8CurrentDataset, g_ui8CurrentImage);
            writeEE((ui16Address + COUNT3_OFFSET), 0xFFFFFFFFu);
        }
        else
        {
            g_ui8CurrentImage = 0u;
            /* Check that counter 3 is programmed properly*/
            ui16Address = imageAddress(g_ui8CurrentDataset, g_ui8CurrentImage);
            if((readEE(ui16Address + COUNT3_OFFSET)) == g_ui32CurrentImageCount)
            {
                g_pui8CurrentImageNumber[g_ui8CurrentDataset] = g_ui8CurrentImage;
                g_eFeeResult = FEE_JOB_OK;
                /* Check if all datasets are now valid */
                g_eFeeStatus = FEE_IDLE;
                for(ui8I = 0u; ui8I < NUMBER_OF_DATASETS; ui8I++)
                {
                    if(0xFF == g_pui8CurrentImageNumber[ui8I])
                    {
                        g_eFeeStatus = FEE_UNINIT;
                        break;
                    }
                }
            }
            else
            {
                g_eFeeResult = FEE_JOB_FAILED;
            }
            g_eFeeState = EEStateIdle;
        }
        break;
    default:
        break;
    }
}

static void retryLastOperation(void)
{
    uint16_t ui16Address;
    uint8_t ui8TotalImages;

    switch(g_eFeeState)
    {
    case EEStateProgrammingFirstCounter:
        ui16Address = imageAddress(g_ui8CurrentDataset, g_ui8CurrentImage);
        ui8TotalImages = psDatasets[g_ui8CurrentDataset].ui8Images;
        /* alternate which to write first, count1 or count2 */
        if(((g_ui32CurrentImageCount /
                ((uint32_t)ui8TotalImages * 6u)) & 1u) == 1u)
        {
            writeEE((ui16Address +  COUNT2_OFFSET), g_ui32CurrentImageCount);
        }
        else
        {
            writeEE((ui16Address + COUNT1_OFFSET), g_ui32CurrentImageCount);
        }
        break;
    case EEStateProgrammingSecondCounter:
        g_eFeeState = EEStateProgrammingFirstCounter;
        executeStateMachine();
        break;
    case EEStateProgrammingData:
        if(1u == g_ui16WriteBufferIndex)
        {
            g_eFeeState = EEStateProgrammingSecondCounter;
        }
        else
        {
            g_ui16WriteBufferIndex--;
        }
        executeStateMachine();
        break;
    case EEStateProgrammingCRC:
        writeEE((g_ui16CurrentImageAddress + CRC_OFFSET ), g_ui32CurrentCRC);
        break;
    case EEStateFinishFormat:
        g_eFeeState = EEStateInitializeDataset;
        executeStateMachine();
        break;
    case EEStateProgrammingThirdCounter:
    case EEStateInitializeDataset:
        g_eFeeResult = FEE_JOB_FAILED;
        g_eFeeStatus = FEE_IDLE;
        g_eFeeState = EEStateIdle;
        break;
    case EEStateIdle:
    default:
        break;
    }
}
/* Check for a valid image in a dataset.
 */
tFeeReturn FeeCheckDatasetValid(uint8_t ui8Dataset)
{
    uint8_t ui8Index;
    uint8_t ui8Image;
    uint8_t ui8TotalImages;
    tFeeReturn feeReturnValue;
    uint16_t ui16ImageStartAddress;
    uint32_t ui32Count1, ui32Count2, ui32Count3, ui32Crc;
    tFeeImageSortList psImageSortList[MAX_IMAGES_PER_SECTOR * 2u];

    /* create list of images with image number, image address and count3 */
    createSortedImageList(ui8Dataset, psImageSortList);
    /* now go through list in order checking if count3 is equal to count1 and
     * count2. Check that count is even for even sector numbers and odd for
     * odd sector numbers. Then check if data has valid CRC.
     */
    ui8TotalImages = (psDatasets[ui8Dataset].ui8Images * 2u);
    /* start by assuming no valid image */
    g_pui8CurrentImageNumber[ui8Dataset] = 0xFFu;
    feeReturnValue = FEE_NOT_OK;
    for(ui8Index = 0u; ui8Index < ui8TotalImages; ui8Index++)
    {
        ui16ImageStartAddress = psImageSortList[ui8Index].ui16ImageAddress;
        ui32Count3 = psImageSortList[ui8Index].ui32ImageCount3;
        ui32Count2 = readEE(ui16ImageStartAddress + COUNT2_OFFSET);
        ui32Count1 = readEE(ui16ImageStartAddress + COUNT1_OFFSET);
        if((ui32Count1 == ui32Count2) && (ui32Count1 == ui32Count3))
        {
            /* if count is odd and image number is odd, or count is even and
             * image number is even */
            if((((psImageSortList[ui8Index].ui32ImageCount3 & 1u) == 1u) &&
                ((psImageSortList[ui8Index].ui8ImageNumber & 1u) == 1u)) ||
               (((psImageSortList[ui8Index].ui32ImageCount3 & 1u) == 0u) &&
                ((psImageSortList[ui8Index].ui8ImageNumber & 1u) == 0u)))
            {
                ui32Crc = calculateEepromCRC((ui16ImageStartAddress +
                             DATA_OFFSET), psDatasets[ui8Dataset].ui8Words);
                if(ui32Crc == readEE(ui16ImageStartAddress + CRC_OFFSET))
                {
                    /* found a good image */
                    ui8Image = psImageSortList[ui8Index].ui8ImageNumber;
                    g_pui8CurrentImageNumber[ui8Dataset] = ui8Image;
                    if(ui8Index == 0u) /* image with largest count is valid */
                    {
                        feeReturnValue = FEE_OK;
                    }
                    else if((psImageSortList[ui8Index].ui32ImageCount3 + 1u ) ==
                            (psImageSortList[ui8Index - 1u].ui32ImageCount3))
                    {
                        feeReturnValue = FEE_OLD;
                    }
                    else /* assume count3 of previous image was corrupt */
                    {
                        feeReturnValue = FEE_OK;
                    }
                    break;
                }
            }
        }
    }
    return(feeReturnValue);
}

static uint32_t readEE(uint16_t eepromLocation)
{
    HWREG(EEPROM_EEBLOCK) = blockFromAddress(eepromLocation);
    HWREG(EEPROM_EEOFFSET) = offsetFromAddress(eepromLocation);
    return HWREG(EEPROM_EERDWR);
}

static void writeEE(uint16_t eepromLocation, uint32_t word)
{
    uint32_t ui32Status;

    // Make sure the EEPROM is idle before we start.
    do
    {
        ui32Status = HWREG(EEPROM_EEDONE);
    }
    while(ui32Status & EEPROM_EEDONE_WORKING);
    HWREG(EEPROM_EEBLOCK) = blockFromAddress(eepromLocation);
    HWREG(EEPROM_EEOFFSET) = offsetFromAddress(eepromLocation);
    HWREG(EEPROM_EERDWR) = word;
#if defined(EEPROM_INTERRUPTS)
    EEPROMIntEnable(EEPROM_INT_PROGRAM);
#endif
}

static int32_t sortListCompareFunc(const void *pvFirst, const void *pvSecond)
{
    int32_t i32ReturnValue;
    uint32_t ui32Count1, ui32Count2;

    ui32Count1 = *((const uint32_t *)pvFirst + 1u);
    ui32Count2 = *((const uint32_t *)pvSecond + 1u);

    if( ui32Count1 == ui32Count2)
    {
        i32ReturnValue = 0;
    }
    else if(ui32Count1 == 0xFFFFFFFFu) //Put blank counter at the end
    {
        i32ReturnValue = 1;
    }
    else if( ui32Count1 > ui32Count2)
    {
        i32ReturnValue = -1;
    }
    else
    {
        i32ReturnValue = 1;
    }
    return i32ReturnValue;
}

static uint32_t calculateRAMCRC(const uint32_t bufferAddress[], uint8_t count)
{
    uint8_t ui8Index;
    uint32_t ui32ReturnValue;
#if defined(CRC_SETTINGS)
    uint32_t ui32CrcValue;

    ui32CrcValue = CRCResultRead(CCM0_BASE, false);
#endif
    CRCConfigSet(CCM0_BASE, CRC_CFG_INIT_1 | CRC_CFG_TYPE_P4C11DB7 |
                 CRC_CFG_SIZE_32BIT);
    for(ui8Index = 0u; ui8Index < count; ui8Index++)
    {
        CRCDataWrite(CCM0_BASE, bufferAddress[ui8Index]);
    }
    ui32ReturnValue = CRCResultRead(CCM0_BASE, false);
#if defined(CRC_SETTINGS)
    CRCConfigSet(CCM0_BASE, (CRC_SETTINGS & 0x1FFFu));
    CRCSeedSet(CCM0_BASE, ui32CrcValue);
#endif
    return ui32ReturnValue;
}

static uint32_t calculateEepromCRC(uint16_t eepromAddress, uint8_t count)
{
    uint8_t ui8Index;
    uint32_t ui32ReturnValue;
#if defined(CRC_SETTINGS)
    uint32_t ui32CrcValue;

    ui32CrcValue = CRCResultRead(CCM0_BASE, false);
#endif
    CRCConfigSet(CCM0_BASE, CRC_CFG_INIT_1 | CRC_CFG_TYPE_P4C11DB7 |
                 CRC_CFG_SIZE_32BIT);

    for(ui8Index = 0u; ui8Index < count; ui8Index++)
    {
        CRCDataWrite(CCM0_BASE, readEE(eepromAddress));
        eepromAddress += 4u;
    }
    ui32ReturnValue = CRCResultRead(CCM0_BASE, false);
#if defined(CRC_SETTINGS)
    CRCConfigSet(CCM0_BASE, (CRC_SETTINGS & 0x1FFFu));
    CRCSeedSet(CCM0_BASE, ui32CrcValue);
#endif
    return ui32ReturnValue;
}

static uint32_t calculateBlankCRC(uint8_t count)
{
    uint8_t ui8Index;
    uint32_t ui32ReturnValue;
#if defined(CRC_SETTINGS)
    uint32_t ui32CrcValue;

    ui32CrcValue = CRCResultRead(CCM0_BASE, false);
#endif
    CRCConfigSet(CCM0_BASE, CRC_CFG_INIT_1 | CRC_CFG_TYPE_P4C11DB7 |
                 CRC_CFG_SIZE_32BIT);

    for(ui8Index = 0u; ui8Index < count; ui8Index++)
    {
        CRCDataWrite(CCM0_BASE, 0xFFFFFFFF);
    }
    ui32ReturnValue = CRCResultRead(CCM0_BASE, false);
#if defined(CRC_SETTINGS)
    CRCConfigSet(CCM0_BASE, (CRC_SETTINGS & 0x1FFFu));
    CRCSeedSet(CCM0_BASE, ui32CrcValue);
#endif
    return ui32ReturnValue;
}


static uint16_t imageAddress(uint8_t ui8Dataset, uint8_t ui8ImageNumber)
{
    uint16_t ui16Address;

    /* even images in sector 0 */
    ui16Address = pui16FirstSectorDatasetStart[ui8Dataset] +
            ((ui8ImageNumber >> 1u) *
            ((psDatasets[ui8Dataset].ui8Words + CONTROL_WORDS) * 4u));
    if(1u == (ui8ImageNumber & 1u))
    {   /* odd images in sector 1 */
        ui16Address += SECTOR_SIZE;
    }
    return ui16Address;
}

static uint32_t imageCount3(uint8_t ui8Dataset, uint8_t ui8ImageNumber)
{
    uint16_t ui16Address;

    ui16Address = imageAddress(ui8Dataset, ui8ImageNumber);
    return readEE(ui16Address + COUNT3_OFFSET);
}

static void
createSortedImageList(uint8_t ui8Dataset, tFeeImageSortList psList[])
{
    uint8_t ui8Index, ui8NumberOfImages;

    /* images in 2 sectors */
    ui8NumberOfImages = psDatasets[ui8Dataset].ui8Images << 1u;
    for(ui8Index = 0u; ui8Index < ui8NumberOfImages; ui8Index++)
    {
        psList[ui8Index].ui8ImageNumber = ui8Index;
        psList[ui8Index].ui16ImageAddress = imageAddress(ui8Dataset, ui8Index);
        psList[ui8Index].ui32ImageCount3 = imageCount3(ui8Dataset, ui8Index);
    }
    /* now sort these records with highest count first */
    qsort((void *)psList, (size_t)ui8NumberOfImages, sizeof(tFeeImageSortList),
          &sortListCompareFunc);
}

static uint32_t offsetFromAddress(uint16_t ui16Address)
{
    return(((uint32_t)ui16Address >> 2u) & 0x0Fu);
}

static uint32_t blockFromAddress(uint16_t ui16Address)
{
    return ((uint32_t)ui16Address >> 6u);
}
