//
// Copyright (c) 2012-2021 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
// 
//
//*****************************************************************************
/*--------------------------------------------------------------
* Revision History
*--------------------------------------------------------------
* Version Date       Author    Change ID     Description
*--------------------------------------------------------------
* 00.01.00 21Jan2021 RM Crosby 0000000000000 Initial Version
*/
#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_uart.h"
#include "driverlib/sysctl.h"
#include "driverlib/gpio.h"
#include "driverlib/eeprom.h"
#include "driverlib/uart.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "utils/uartstdio.h"
#include "feeConfig.h"
#include "fee.h"

void main(void);
static void feeDump(void);
static bool checkDataBufferGood(const uint32_t *buffer);
static void configureUART(void);
static void incrementDataset(uint32_t ui32Dataset);
static void zeroDataset(uint8_t ui8Dataset);

//*****************************************************************************
//
// Define pin to LED mapping.
//
//*****************************************************************************

//*****************************************************************************
//
//!
//! This example demonstrates the use of the high endurance flash based EEPROM
//! emulation driver.
//
//*****************************************************************************

#define USER_LED1  GPIO_PIN_0
#define USER_LED2  GPIO_PIN_1
#define SW1 GPIO_PIN_0
#define SW2 GPIO_PIN_1
#define EEPROM_WORDS 1536u

//*****************************************************************************
//
// The error routine that is called if the driver library encounters an error.
//
//*****************************************************************************
#ifdef DEBUG
void
__error__(char *pcFilename, uint32_t ui32Line)
{
}
#endif

uint32_t g_ui32SysClock;
static uint32_t g_pui32DataBuffer[4u];

void main(void)
{
    uint8_t ui8Index;
    uint32_t ui32Leds, ui32Switches;
    tFeeVersion sVersion;
    tFeeReturn eFeeReturn;

    g_ui32SysClock = SysCtlClockFreqSet((SYSCTL_XTAL_25MHZ |
                                       SYSCTL_OSC_MAIN |
                                       SYSCTL_USE_PLL |
                                       SYSCTL_CFG_VCO_240), 120000000u);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPION);
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPION))
    {
    }
    // Configure the GPIO port for the LED operation.
    GPIOPinTypeGPIOOutput(GPIO_PORTN_BASE, (USER_LED1|USER_LED2));
    SysCtlPeripheralEnable(SYSCTL_PERIPH_EEPROM0);
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_EEPROM0))
    {
    }
    // Configure GPIO pins for switch inputs
    //
    // PJ0 and J1 are used for user buttons
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOJ);
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOJ))
    {
    }
    GPIOPinTypeGPIOInput(GPIO_PORTJ_BASE, SW1 | SW2);
    GPIOPadConfigSet(GPIO_PORTJ_BASE,  SW1 | SW2,
                        GPIO_STRENGTH_2MA, GPIO_PIN_TYPE_STD_WPU);

    configureUART();
    UARTprintf("Fee Example Version 1.00\n");
    sVersion = FeeGetVersionInformation();
    UARTprintf("Fee Version %d.%02d\n",
               sVersion.ui8MajorVersion, sVersion.ui8MinorVersion );
    eFeeReturn = FeeInit();
    if(eFeeReturn == FEE_OLD)
    {
        UARTprintf("  At least on dataset reverted to old data\n");
    }
    else if(eFeeReturn == FEE_NOT_OK) // at least one dataset is bad
    {
        UARTprintf("  Fee Initialization Failed\n");
        for(ui8Index = 0u; ui8Index < NUMBER_OF_DATASETS; ui8Index++)
        {
            if(FeeCheckDatasetValid(ui8Index) == FEE_NOT_OK)
            {
                FeeFormat(ui8Index);
                UARTprintf("  Formatting dataset %d\n", ui8Index);
                while(FeeGetStatus() == FEE_BUSY)
                {
                    FeeMainFunction();
                }
                if(FeeGetJobResult() != FEE_JOB_OK)
                {
                    UARTprintf("  FAIL: FeeFormat dataset %d did not pass\n",
                               ui8Index);
                    feeDump(); /* never returns */
                }
            }
        }
        for(ui8Index = 0u; ui8Index < (NUMBER_OF_DATASETS ); ui8Index++)
        {
            FeeRead(ui8Index, g_pui32DataBuffer);
            if(g_pui32DataBuffer[0] == 0xFFFFFFFFu)
            {
                g_pui32DataBuffer[0] = 0u;
                g_pui32DataBuffer[2] = 0u;
                FeeWrite(ui8Index, g_pui32DataBuffer);
                while(FeeGetStatus() == FEE_BUSY)
                {
                        FeeMainFunction();
                }
                if(FeeGetJobResult() != FEE_JOB_OK)
                {
                    UARTprintf("  FAIL: Intial write for dataset %d failed\n",
                               ui8Index);
                    feeDump(); /* never returns */
                }
                UARTprintf("  Count in dataset %d set to zero\n", ui8Index);
            }
        }
    }
    /* Read and print  counts from datasets */
    for(ui8Index = 0u; ui8Index < NUMBER_OF_DATASETS; ui8Index++)
    {
        /* read counts from each dataset */
        eFeeReturn = FeeRead(ui8Index,g_pui32DataBuffer);
        if((eFeeReturn == FEE_OK) || (eFeeReturn == FEE_OLD))
        {
            if(eFeeReturn == FEE_OLD)
            {
                UARTprintf(" Dataset %d corrupt, reading old data\n", ui8Index);
            }
            if(checkDataBufferGood(g_pui32DataBuffer) == true)
            {
                UARTprintf(" Data from dataset %d is good\n", ui8Index);
                UARTprintf("   Count = %d\n", g_pui32DataBuffer[0]);
                UARTprintf("   Cycle count = %d\n",
                           FeeGetDatasetCounter(ui8Index));
            }
            else
            {
                /* the data read was corrupt */
                UARTprintf("Error: Data from dataset %d is corrupt:", ui8Index);
                UARTprintf("   0x%08X, 0x%08X, 0x%08X, 0x%08X\n",
                           g_pui32DataBuffer[0], g_pui32DataBuffer[1],
                           g_pui32DataBuffer[2], g_pui32DataBuffer[3]);
                zeroDataset(ui8Index);
            }
        }
        else
        {
            UARTprintf("Error: No valid data in dataset %d\n", ui8Index);
            zeroDataset(ui8Index);
        }
    }
    ui32Leds = USER_LED1;
    while(1)
    {
        // Alternate LEDs 1 and 2
        GPIOPinWrite(GPIO_PORTN_BASE, (USER_LED1|USER_LED2), ui32Leds);
        ui32Leds = ui32Leds ^ 3u;
        ui32Switches = GPIOPinRead(GPIO_PORTJ_BASE, SW1 | SW2);
        switch(ui32Switches)
        {
        case 0u: //Both buttons pressed
            incrementDataset(0u);
            incrementDataset(1u);
            break;
        case 1u: //Switch 2 pressed
            incrementDataset(1u);
            break;
        case 2u: //Switch 1 pressed
            incrementDataset(0u);
            break;
        case 3u: //No switches pressed
        default:
            break;
        }
        SysCtlDelay(g_ui32SysClock/30); /* about 100 ms */
    }
}

//*****************************************************************************
//
// Configure the UART and its pins.  This must be called before UARTprintf().
//
//*****************************************************************************
static void configureUART(void)
{
    //
    // Enable the GPIO Peripheral used by the UART.
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

    //
    // Enable UART0
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);

    //
    // Configure GPIO Pins for UART mode.
    //
    GPIOPinConfigure(GPIO_PA0_U0RX);
    GPIOPinConfigure(GPIO_PA1_U0TX);
    GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

    //
    // Initialize the UART for console I/O.
    //
    UARTStdioConfig(0, 115200, g_ui32SysClock);
}

static void feeDump(void)
{
    static uint32_t pui32Data[EEPROM_WORDS]; /* 6KB */

    // Read the entire EEPROM
    EEPROMRead(pui32Data, 0x000, EEPROM_WORDS << 2u);
    for(;;)
    {
        /* spin here forever */
    }
}

static bool checkDataBufferGood(const uint32_t *buffer)
{
    /* for this test the first four 32-bit integers should contain a number,
     * the 1's complement of that number, the number again and the 1's
     * complement again. Any other pattern is corrupt data.
     */
    if((buffer[0u] == buffer[2u]) && (buffer[1u] == buffer[3u]) &&
            (buffer[0u] == ~buffer[1u]))
    {
        return(true);
    }
    else
    {
        return(false);
    }
}

static void incrementDataset(uint32_t ui32Dataset)
{
    tFeeReturn eFeeReturn;

    eFeeReturn = FeeRead(ui32Dataset, g_pui32DataBuffer);
    if(FEE_OLD == eFeeReturn)
    {
        UARTprintf(" Dataset %d corrupt, reading old data\n", ui32Dataset);
    }
    if(false == checkDataBufferGood(g_pui32DataBuffer))
    {
        UARTprintf("Error: Data from dataset %d is corrupt:", ui32Dataset);
        feeDump(); /* never returns */
    }
    g_pui32DataBuffer[0]++;
    g_pui32DataBuffer[1] = ~g_pui32DataBuffer[0];
    g_pui32DataBuffer[2] = g_pui32DataBuffer[0];
    g_pui32DataBuffer[3] = ~g_pui32DataBuffer[0];
    FeeWrite(ui32Dataset, g_pui32DataBuffer);
    while(FeeGetStatus() != FEE_IDLE)
    {
        FeeMainFunction();
    }
    if(FeeGetJobResult() != FEE_JOB_OK)
    {
        UARTprintf("  FAIL: Write for dataset %d did not pass\n",
                   ui32Dataset);
        feeDump(); /* never returns */
    }
    else
    {
        UARTprintf("Counter in dataset %d set to %d\n",
                   ui32Dataset, g_pui32DataBuffer[0]);
    }
}

static void zeroDataset(uint8_t ui8Dataset)
{
    uint32_t ui32Buttons;

    UARTprintf("Press Switch %d to zero dataset %d\n", ui8Dataset +1u,
               ui8Dataset);
    g_pui32DataBuffer[0] = 0u;
    g_pui32DataBuffer[1] = 0xFFFFFFFFu;
    g_pui32DataBuffer[2] = 0u;
    g_pui32DataBuffer[3] = 0xFFFFFFFFu;
    for(;;)
    {
        ui32Buttons = GPIOPinRead(GPIO_PORTJ_BASE, SW1 | SW2);
        if(((ui8Dataset == 0u) && ((ui32Buttons & SW1) != SW1)) || //SW1 pressed
           ((ui8Dataset == 1u) && ((ui32Buttons & SW2) != SW2)))   //SW2 pressed
        {
            FeeWrite(ui8Dataset, g_pui32DataBuffer);
            while(FeeGetStatus() == FEE_BUSY)
            {
                FeeMainFunction();
            }
            break;
        }
    }
}
