/***************************************************************************/
/*                                                                         */
/* C6416_CACHE.C - Routine which invalidates the program cache that        */
/*                 corresponds to the block(s) of program memory           */
/*                 overwritten by a copy table entry.                      */
/*-------------------------------------------------------------------------*/
/* Do not use any of this code elsewhere without first reading the legal   */
/* disclaimer at the end of the file.                                      */
/*-------------------------------------------------------------------------*/
/*                              !!! WARNING !!!                            */
/* Do NOT call this routine in these circumstances:                        */
/* - A data block is being overlaid instead of a code block.  In that case */
/*   there is no need to invalidate any cache.                             */
/* - This code is being executed on a simulator which does not model the   */
/*   cache.  This includes the C64xx CPU Cycle Accurate Simulator.   The   */
/*   calls to the CSL routines hang on such simulators.                    */
/* - You have any of L2 memory configured as cache, and not all SRAM.  In  */
/*   that case you have to be concerned about invalidating L2 cache as     */
/*   well as the L1 cache invalidated in this routine.                     */
/*                                                                         */
/***************************************************************************/

#include <cpy_tbl.h>                  /* Defines COPY_TABLE structure      */

/*-------------------------------------------------------------------------*/
/* Use these header files when calling the Chip Support Library (CSL)      */
/* cache routines.  For more information consult online help.  From within */
/* CCS access: Help | Contents | TMS320C6000 DSP Reference | C6000 Chip    */
/* Support Library | Using CSL APIs Without DSP/BIOS Kernel.               */
/*-------------------------------------------------------------------------*/
#define CHIP_6416
#include <csl.h>
#include <csl_cache.h>

#define ROUNDUP_POW2(val, pow2) (((val) + (pow2)-1) & ~(pow2))

/*-------------------------------------------------------------------------*/
/* C6416_INVALIDATE_PROG_CACHE -                                           */
/*     Call this routine whenever a copy table entry is used to overwrite  */
/*     a block of L2 Memory configured as SRAM.  In such a case, the L1P   */
/*     program cache that corresponds to the blocks in the copy table      */
/*     entry must be invalidated in the cache.  If this step is not taken, */
/*     when calling the newly copied in routine, what gets executed is     */
/*     whatever remains in the cache from before, and not what is in L2    */
/*     SRAM.  Copying over data memory with copy tables, however, requires */
/*     no cache invalidation step.                                         */
/*-------------------------------------------------------------------------*/
/*     Documentation on the specific cache routines called can be found    */
/*     in online help.  From within CCS access: Help | Contents |          */
/*     TMS320C6000 DSP Reference | C6000 Chip Support Library | Cache      */
/*     Module.  Note these routines are defined in the CSL library         */
/*     csl6416.lib, which included with the -l option in the link.         */
/*-------------------------------------------------------------------------*/
void c6416_invalidate_prog_cache(COPY_TABLE *tp)
{
    static int need_init = 1;
    int i;
    
    if (need_init)
    {
        need_init = 0;
        CSL_init();
    }
    
    for (i = 0; i < tp->num_recs; i++)
    {
        CACHE_invL1p((unsigned char *)tp->recs[i].run_addr,
                     ROUNDUP_POW2(tp->recs[i].size, 4),
                     CACHE_NOWAIT);
    }
    
    CACHE_wait();
}

/* ***********************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR
* REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY, 
* INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS 
* FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR 
* COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE. 
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET 
* POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY 
* INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR 
* YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, 
* CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY 
* THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED 
* OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT 
* OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. 
* EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF 
* REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS 
* OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF 
* USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S 
* AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF 
* YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS 
* (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted 
* by Texas Instruments is distributed as "freeware".  You may, 
* only under TI's copyright in the Program, use and modify the 
* Program without any charge or restriction.  You may 
* distribute to third parties, provided that you transfer a 
* copy of this license to the third party and the third party 
* agrees to these terms by its first use of the Program. You 
* must reproduce the copyright notice and any other legend of 
* ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains 
* copyrighted material, trade secrets and other TI proprietary 
* information and is protected by copyright laws, 
* international copyright treaties, and trade secret laws, as 
* well as other intellectual property laws.  To protect TI's 
* rights in the Program, you agree not to decompile, reverse 
* engineer, disassemble or otherwise translate any object code 
* versions of the Program to a human-readable form.  You agree 
* that in no event will you alter, remove or destroy any 
* copyright notice included in the Program.  TI reserves all 
* rights not specifically granted under this license. Except 
* as specifically provided herein, nothing in this agreement 
* shall be construed as conferring by implication, estoppel, 
* or otherwise, upon you, any license or other right under any 
* TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
* ********************************************************* */

