/*****************************************************************************/
/*                                                                           */
/* DEMO.C - Demonstrate linker features copy tables and automatic section    */
/*          splitting.   Most of the interesting linker details are in the   */
/*          linker command file demo.cmd.                                    */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/* Do not use any of this code elsewhere without first reading the legal     */
/* legal disclaimer at the end of the file.                                  */
/*****************************************************************************/
#include <stdio.h>
#include <cpy_tbl.h>

/*---------------------------------------------------------------------------*/
/* Declare the copy tables that are created in the linker command file.      */
/* Use "far" because they reside in external memory, i.e. too far to be      */
/* accessed at an offset from the DP like the other global variables.        */
/*---------------------------------------------------------------------------*/
extern far COPY_TABLE func1_ctbl, func23_ctbl;

extern void c6416_invalidate_prog_cache(COPY_TABLE *tp);

/*---------------------------------------------------------------------------*/
/* func1 and func2/func3 share the same run address in L2 SRAM.  Both sets   */
/* of func load in external memory.  Note the load allocation of func2/func3 */
/* is split across multiple memory ranges.                                   */
/*---------------------------------------------------------------------------*/
void func1(void);
void func2(void), func3(void);

/*---------------------------------------------------------------------------*/
/* These funcs are all loaded in external memory.  Section splitting is used */
/* allocate them across a range of artificially small memory ranges.         */
/*---------------------------------------------------------------------------*/
extern void func4(void);
extern void func5(void);
extern void func6(void);
extern void func7(void);

/*---------------------------------------------------------------------------*/
/* The global variable incremented by each of the funcs.  By the time the    */
/* demo is over x = 1+2+3+4+5+6+7 = 28.                                      */
/*---------------------------------------------------------------------------*/
int x = 0;

void main()
{
    /*-----------------------------------------------------------------------*/
    /* Copy the code for func1 from EMIFA to L2RAM.  Then, invalidate the    */
    /* program cache for this memory block(s).  See c6416_cache.c for        */
    /* details.  Then, can call func1.                                       */
    /*-----------------------------------------------------------------------*/
    copy_in(&func1_ctbl);
    c6416_invalidate_prog_cache(&func1_ctbl);
    func1();
       
    /*-----------------------------------------------------------------------*/
    /* Note func2 and func3 are allocated together.  The load allocation     */
    /* is spread across multiple blocks of EMIFA.  These multiple blocks are */
    /* reflected in the multiple entries in the copy table.  All of this     */
    /* detail is comprehended in the copy table and copy_in routine.  Thus,  */
    /* this code is similar to that for func1.                               */
    /*-----------------------------------------------------------------------*/
    copy_in(&func23_ctbl);
    c6416_invalidate_prog_cache(&func23_ctbl);
    func2();
    func3();
       
    /*-----------------------------------------------------------------------*/
    /* The remaining funcs demonstrate automatic section splitting.          */
    /*-----------------------------------------------------------------------*/
    func4();
    func5();
    func6();
    func7();

    printf("linker example %s!\n", (x != 28) ? "FAILED" : "PASSED");
}

#pragma CODE_SECTION(func1, ".func1_scn")
void func1(void)
{
    printf("hit func1, run address is %p, x is %d\n", func1, x);
    x += 1;
}

#pragma CODE_SECTION(func2, ".func2_scn")
void func2(void)
{
    printf("hit func2, run address is %p, x is %d\n", func2, x);
    x += 2;
}

#pragma CODE_SECTION(func3, ".func3_scn")
void func3(void)
{
    printf("hit func3, run address is %p, x is %d\n", func3, x);
    x += 3;
}

/* ***********************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR
* REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY, 
* INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS 
* FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR 
* COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE. 
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET 
* POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY 
* INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR 
* YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, 
* CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY 
* THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED 
* OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT 
* OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. 
* EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF 
* REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS 
* OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF 
* USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S 
* AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF 
* YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS 
* (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted 
* by Texas Instruments is distributed as "freeware".  You may, 
* only under TI's copyright in the Program, use and modify the 
* Program without any charge or restriction.  You may 
* distribute to third parties, provided that you transfer a 
* copy of this license to the third party and the third party 
* agrees to these terms by its first use of the Program. You 
* must reproduce the copyright notice and any other legend of 
* ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains 
* copyrighted material, trade secrets and other TI proprietary 
* information and is protected by copyright laws, 
* international copyright treaties, and trade secret laws, as 
* well as other intellectual property laws.  To protect TI's 
* rights in the Program, you agree not to decompile, reverse 
* engineer, disassemble or otherwise translate any object code 
* versions of the Program to a human-readable form.  You agree 
* that in no event will you alter, remove or destroy any 
* copyright notice included in the Program.  TI reserves all 
* rights not specifically granted under this license. Except 
* as specifically provided herein, nothing in this agreement 
* shall be construed as conferring by implication, estoppel, 
* or otherwise, upon you, any license or other right under any 
* TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
* ********************************************************* */

