//############################################################################
//
// FILE:   Example_2833x_Flash_to_RAM_nonBIOS.c
//
// TITLE:  DSP2833x Flash to RAM Example 
//
// ASSUMPTIONS:
//
//    This program uses the DSP2833x header files already included in the  
// 	  download. 
//
//    As supplied, this project is configured for "boot to FLASH"
//    operation.  The 2833x Boot Mode table is shown below.
//    For information on configuring the boot mode of an eZdsp,
//    please refer to the documentation included with the eZdsp,
//
//       $Boot_Table:
//
//         GPIO87   GPIO86     GPIO85   GPIO84
//          XA15     XA14       XA13     XA12
//           PU       PU         PU       PU
//        ==========================================
//            1        1          1        1    Jump to Flash <- "boot to Flash"
//            1        1          1        0    SCI-A boot
//            1        1          0        1    SPI-A boot
//            1        1          0        0    I2C-A boot
//            1        0          1        1    eCAN-A boot
//            1        0          1        0    McBSP-A boot
//            1        0          0        1    Jump to XINTF x16
//            1        0          0        0    Jump to XINTF x32
//            0        1          1        1    Jump to OTP
//            0        1          1        0    Parallel GPIO I/O boot
//            0        1          0        1    Parallel XINTF boot
//            0        1          0        0    Jump to SARAM
//            0        0          1        1    Branch to check boot mode
//            0        0          1        0    Boot to flash, bypass ADC cal
//            0        0          0        1    Boot to SARAM, bypass ADC cal
//            0        0          0        0    Boot to SCI-A, bypass ADC cal
//                                              Boot_Table_End$
//
//    The program must first be compiled and then programmed into the
//    flash. 
//
// DESCRIPTION:
//	 
//	  This project provides an example of how copy code from flash to RAM directly
//	  after code startup. This will allow all code to be ran from RAM directly after 
//    booting in standalone mode. GPIO32 is toggled at a specific time selected within 
//	  this file in the ePWM1 ISR.  
//		
//############################################################################		
// Original Author: Tim Love
// Release Date: March 2008
//############################################################################


#include "DSP2833x_Device.h"     // Headerfile Include File
#include "DSP2833x_Examples.h"   // Examples Include File

// Specify LED Toggle Rate 
#define PWM1_TIMER_TBPRD 41825  // ePWM1 Timer Period (LED Toggle 1 sec)
//#define PWM1_TIMER_TBPRD 20926  // ePWM1 Timer Period (LED Toggle .5 sec)
//#define PWM1_TIMER_TBPRD 10463   // ePWM1 Timer Period (LED Toggle .25 sec)

// Prototype statements for functions found within this file.
interrupt void epwm1_timer_isr(void);
void InitEPwmTimer(void);

void main(void)
{

// Step 1. Initialize System Control:
// PLL, WatchDog, enable Peripheral Clocks
// This example function is found in the DSP280x_SysCtrl.c file.
   InitSysCtrl();

// Step 2. Initalize GPIO: 
// This example function is found in the DSP2833x_Gpio.c file and
// illustrates how to set the GPIO to it's default state.
// InitGpio();  // Skipped for this example  


// Step 3. Clear all interrupts and initialize PIE vector table:
// Disable CPU interrupts 
   DINT;

// Initialize the PIE control registers to their default state.
// The default state is all PIE interrupts disabled and flags
// are cleared.  
// This function is found in the DSP2833x_PieCtrl.c file.
   InitPieCtrl();
   
// Disable CPU interrupts and clear all CPU interrupt flags:
   IER = 0x0000;
   IFR = 0x0000;

// Initialize the PIE vector table with pointers to the shell Interrupt 
// Service Routines (ISR).  
// This will populate the entire table, even if the interrupt
// is not used in this example.  This is useful for debug purposes.
// The shell ISR routines are found in DSP2833x_DefaultIsr.c.
// This function is found in DSP2833x_PieVect.c.
   InitPieVectTable();

// Interrupts that are used in this example are re-mapped to
// ISR functions found within this file.  
   EALLOW;  // This is needed to write to EALLOW protected registers
   PieVectTable.EPWM1_INT = &epwm1_timer_isr;
   EDIS;    // This is needed to disable write to EALLOW protected registers

// Step 4. Initialize all the Device Peripherals:
// This function is found in DSP2833x_InitPeripherals.c
// InitPeripherals();  // Not required for this example
   InitEPwmTimer();    // For this example, initialize ePWM1 Timers

// Step 5. User specific code, enable interrupts:*/

   // Configure GPIO32 as output to drive LED on eZdsp
   EALLOW;
   GpioCtrlRegs.GPBDIR.bit.GPIO32 = 1;
   EDIS;

   // Enable CPU INT3 which is connected to EPWM1 INT:
   IER |= M_INT3;

   // Enable EPWM INT1 in the PIE: Group 3 interrupt 1
   PieCtrlRegs.PIEIER3.bit.INTx1 = 1;

   // Enable global Interrupts and higher priority real-time debug events:
   EINT;   // Enable Global interrupt INTM
   ERTM;   // Enable Global realtime interrupt DBGM

// Step 6. IDLE loop. Just sit and loop forever (optional):
     
   for(;;){};

} 

void InitEPwmTimer()
{

   EALLOW;
   SysCtrlRegs.PCLKCR0.bit.TBCLKSYNC = 0;     	 // Stop all the TB clocks
   EDIS;

   EPwm1Regs.TBPRD = PWM1_TIMER_TBPRD;			 // ePWM1 Timer Period 
   EPwm1Regs.TBCTL.bit.CLKDIV = 7;				 // CLKDIV = /128
   EPwm1Regs.TBCTL.bit.HSPCLKDIV = 7;			 // HSPCLKDIV = /14
   EPwm1Regs.TBCTL.bit.CTRMODE = 0;    			 // Count up
   EPwm1Regs.ETSEL.bit.INTSEL = 1;     			 // Select INT on Zero event
   EPwm1Regs.ETSEL.bit.INTEN = 1;  				 // Enable INT
   EPwm1Regs.ETPS.bit.INTPRD = 1;           	 // Generate INT on 1st event

   EALLOW;
   SysCtrlRegs.PCLKCR0.bit.TBCLKSYNC = 1;        // Start all the timers synced
   EDIS;
}

// Interrupt routines uses in this example:
interrupt void epwm1_timer_isr(void)
{
   // Toggle LED
   GpioDataRegs.GPBTOGGLE.bit.GPIO32 = 1;

   // Clear INT flag for this timer
   EPwm1Regs.ETCLR.bit.INT = 1;
   
   // Acknowledge this interrupt to receive more interrupts from group 3
   PieCtrlRegs.PIEACK.all = PIEACK_GROUP3;
	
}
