//**************************************************************************
//
//  ESP430CE1 Application Program Example
//  Display Data
//
//  This file displays the data on the LCD display
//
//  author    Stefan Schauer
//  date      07/28/2003
//  Modified  Stefan Schauer
//  date      05/27/2004
//  Changes:
//     - SetArrow function added
//
//    Version 1.0 first release for Application Report
//    06/04/04
//    Version 1.3
//    11/27/07  added support for ESP_V2
//
//**************************************************************************

/* ***********************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR
* REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY,
* INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR
* COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET
* POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY
* INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR
* YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
* CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY
* THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED
* OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT
* OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM.
* EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF
* REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS
* OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF
* USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S
* AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF
* YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS
* (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted
* by Texas Instruments is distributed as "freeware".  You may,
* only under TI's copyright in the Program, use and modify the
* Program without any charge or restriction.  You may
* distribute to third parties, provided that you transfer a
* copy of this license to the third party and the third party
* agrees to these terms by its first use of the Program. You
* must reproduce the copyright notice and any other legend of
* ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains
* copyrighted material, trade secrets and other TI proprietary
* information and is protected by copyright laws,
* international copyright treaties, and trade secret laws, as
* well as other intellectual property laws.  To protect TI's
* rights in the Program, you agree not to decompile, reverse
* engineer, disassemble or otherwise translate any object code
* versions of the Program to a human-readable form.  You agree
* that in no event will you alter, remove or destroy any
* copyright notice included in the Program.  TI reserves all
* rights not specifically granted under this license. Except
* as specifically provided herein, nothing in this agreement
* shall be construed as conferring by implication, estoppel,
* or otherwise, upon you, any license or other right under any
* TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
* ********************************************************* */

#include "device.h"
#include "fet4xx_rtclcd.h"
#include "display.h"
#include "emeter.h"
#include "parameter.h"
#include "emeter-toolkit.h"		//banner extern declaration
#include "UART0.h"				//rx buffer size
#include <string.h>

#ifdef withDisplay

// ************************************************************
// Definitions & Declarations
// ************************************************************


const char txt_idle[] = "IDLE";

const char txt_display[][7]= {
    "      ",
    "OFF   ",
    "TIME  ",
    "DATE  ",
    "VOLT  ",
    "CURR  ",
    "VPEAK ",
    "IPEAK ",
    "FREQ  ",
    "COSPHI",
    "TEMP  ",
    "POWER ",
    "ENERGY",
    "OFF   ",
  };

unsigned char Display_Mode = display_startup;
unsigned char Display_Hold = 0;
int ModeSwitchDelay = 4;

char msgin[UART_RX_Buffer_Size];

// ************************************************************
// functions
// ************************************************************

void DisplayClear(void)
{
    int i;
    char *LCD = LCDMEM;

    for (i=0; i<32; i++)      // Clear the display
    {
      LCD[i] = 0;
    }
}

void DisplayIDLE(void)
{
      LCDtext ((unsigned char *) txt_idle, 6, 6);
      LCDM16 = (LCDM16 & 0xF0) | 0x0A;   // switch all Arrows on
      Display_Hold = 5;  // Hold 'IDLE' in Display for 5 sec
}


void DisplayNextData(void)
{
    if ( (Display_Mode >= display_clock) && (Display_Mode <= display_energy) )
    {
	if (ModeSwitchDelay-- <= 0)
	{
            ModeSwitchDelay = 4;
            Display_Mode ++;
            if (Display_Mode > display_energy)
            {
               Display_Mode = display_clock;
            }
            LCDtext ((unsigned char *) txt_display[Display_Mode], 6, 6);
            Display_Hold = 2;
        }
    }
}

void UpdateArrow(int x)
{
	const int arrow_table[8] = {0,0x1,0x3,0x7,0xF,0xE,0xC,0x8};
	static int xold;
	static int pos = 0;
	if (x != xold)		// Every time argument changes, rotate the arrow one spot
        {
	   pos++;
	   xold = x;
	}
        SetArrow(arrow_table[pos & 7]);      // set Arrows
}

void DisplayData(void)
{
  int svalue;
  unsigned int value;
  unsigned long lvalue;
  float fvalue;
  unsigned long activeEnergy;

  if (Display_Mode == display_off) return ;

  // Set the display mode indication
#if PCB == TI_BOARD
  if (Display_Mode >= 1)
    LCDM1 = (unsigned char) * ((unsigned char*)(TLCD_Tab + Display_Mode));
#endif
#if PCB ==  SB_BOARD
  if (Display_Mode >= 1)
    Direct2LCD((Display_Mode & 0x0F) << 4, 7 );
#endif

  if (Display_Hold > 0)
  {
    Display_Hold--;
  }
  else
  {
    activeEnergy = (unsigned long)((float) total_energy * (float)(s_parameters.EnergyRatio)/((float)1000000*(float)3600));
	
    switch (Display_Mode)
    {
      case display_startup:
          {
              static unsigned char i = LCD_SIZE-1;
              if (i < LCD_SIZE)
              { LCDtext ((unsigned char *) &banner[0], i + 0, LCD_SIZE); }
              else
              { LCDtext ((unsigned char *) &banner[i + 0 -LCD_SIZE], LCD_SIZE - 1, LCD_SIZE); }
              i++;
              if (i > (sbanner + LCD_SIZE/2))
              {	
                  Display_Mode = display_clock;								// Switch to clock mode
                  LCDtext ((unsigned char *) "       ", LCD_SIZE, LCD_SIZE);  // Clear the display
                  LCDM16 = 0;
                  DisplayNextData();					
              }
          }
          break;
      case display_off:
          Display_Mode = display_off;
          SetupLCD();
          break;
      case display_clock:
          Dis_Clock();
          break;
      case display_date:
          Dis_Date();
          break;
      case display_volt:
          #ifdef __CROSSWORKS_MSP430
                lvalue = ((unsigned long) (V1RMS) * (s_parameters.VRatio*1000))/1000;
          #else
                lvalue = ((unsigned long) ((V1RMS) * (s_parameters.VRatio*1000))/1000);
          #endif
          LCDdecu32(lvalue, 7, LCD_SIZE, 3);
          break;
      case display_current:
          lvalue = ((unsigned long) (IRMS_HI * s_parameters.IRatio*1000))/1000;
          LCDdecu32(lvalue, 7, LCD_SIZE, 3);
          break;
      case display_vpeak:
          #ifdef __CROSSWORKS_MSP430
                lvalue = ((unsigned long) (VPEAK) * (s_parameters.VRatio*1000))/1000;
          #else
                lvalue = ((unsigned long) ((VPEAK) * (s_parameters.VRatio*1000))/1000);
          #endif
          LCDdecu32(lvalue, 7, LCD_SIZE, 3);
          break;
      case display_ipeak:
          lvalue = ((unsigned long) (IPEAK * s_parameters.IRatio*1000))/1000;
          LCDdecu32(lvalue, 7, LCD_SIZE, 3);
          break;
      case display_freq:
          if (MAINSPERIOD == 0)
            lvalue = 0;
          else
            lvalue = (unsigned long)((float)((long)1000*(long)256*(long)4096))/((float)MAINSPERIOD);
          LCDdecu32(lvalue, 7, LCD_SIZE, 3);
          break;
      case display_cosphi:
          value = (unsigned int)((((unsigned long)POWERFCT*1000)/POW_2_14));
#ifdef ESP_V2
          ((signed char)REACTENERGY_HI < 0) ? (svalue = -(signed int)value) : (svalue = value);
#else
          ((signed char)CAPIND < 0) ? (svalue = -(signed int)value) : (svalue = value);
#endif          
          LCDdec16(svalue, 5, LCD_SIZE, 3);
          break;
/*      case display_ch1:
          LCDdecu16(wfs1, 5, LCD_SIZE, 0);
          break;
      case display_ch2:
          LCDdecu16(wfs2, 5, LCD_SIZE, 0);
          break;
      case display_ch3:
          LCDdecu16(wfs3, 5, LCD_SIZE, 0);
          break;  */
      case display_temp:
          lvalue = (unsigned long) (((unsigned long)temperature * (s_parameters.ulTempGain))/100 - (s_parameters.ulTempOffset));
          LCDdecu32(lvalue, 5, LCD_SIZE, 2); // Temperature (degree)
          break;
      case display_power:
          fvalue = ((float) energy.l * (float)(s_parameters.EnergyRatio)/(1000000));
//              if (fvalue <= 1000)
//              {
//                 lvalue = (unsigned long)(fvalue*1000);
//                 LCDdecu32(lvalue, 7, 7, 6); // Active Power (kW);
//              }
//              else
//              {
          lvalue = (unsigned long)fvalue;
          LCDdecu32(lvalue, 7, LCD_SIZE, 3); // Active Power (kW);
//              }
          break;
      case display_energy:
          lvalue = activeEnergy;
          LCDdecu32(lvalue, 7, LCD_SIZE, 3); // Active Energy1 (kWh):
          break;
      case display_msgin:
          {
            static unsigned char i = LCD_SIZE-1;
            if (i < LCD_SIZE)
            { LCDtext ((unsigned char *) &msgin[0], i + 1, LCD_SIZE); }
            else
            { LCDtext ((unsigned char *) &msgin[i + 1 - LCD_SIZE], 7, LCD_SIZE); }
            i++;
            if (i >= strlen(msgin)) i = LCD_SIZE-1;
          }
          break;
      case display_calib:
          LCDtext ((unsigned char *) "CALIB", 6, 6);
          LCDM16 ^= 0xF0;			// Blink all unused segments of LCD
          break;
      case display_last:
      default:
          Display_Mode = display_off;
          SetupLCD();
          break;
    }
    if (Display_Mode > 1)
        UpdateArrow(activeEnergy);

    if (OP_Mode == idle)
      SetArrow(0x05);      // switch up/down Arrows on
    else if (Display_Mode > 1)
        UpdateArrow(activeEnergy);
  }
}

#if LCD_TEST_PATTERN >= 1
void DisplayTest(void)
{
   int i;
   int j;
   int k;
   char *LCD = LCDMEM;

   IE2 &= ~BTIE;   // HEM disable Basic Timer Interrupt
   // clear LCD memory to clear display //
    for (i=0; i<32; i++)
         LCD[i] = 0;

   // Turn on all the LCD segments //
    for (i=0; i<32; i++)
    {
      for (j=0; j<15; j++)
      {
        LCD[i] |= 1<<j;
        for (k=0; k<1000; k++)  { _NOP(); }
      }
    }

    for (k=0; k<30000; k++)      // Wait a little
    {
      _NOP();
    }

    for (i=0; i<32; i++)      // Clear the display
    {
      LCD[i] = 0;
    }



#if LCD_TEST_PATTERN >= 2
    for (i=' '; i<'a'; i++)      // Ripple through all the characters in the LCD table
    {
      for (j=6; j>0; j-- )
      {
        Char2LCD(i,j);
      }
      for (k=0; k<30000; k++)  { _NOP(); _NOP(); }
    }
	
	for (i=0; i<32; i++)      // Clear the display
    {
      LCD[i] = 0;
    }
#endif	// LCD_TEST_PATTERN >=2	

}
#endif // LCD_TEST_PATTERN >=1

#endif // withDisplay
