//**************************************************************************
//  ESP430CE1 Application Program Example
//
//  This file shows exemplarily the usage of the ESP430CE1 module for
//  a single-phase emeter with two current sensors (one shunt and one
//  current transformer).
//
//  Modified Stefan Schauer
//  date   04/29/2003
//
//  changes:
//  - added handling of requested Temp. Measurement
//
//
//    Version 1.0 first release for Application Report
//    06/04/04
//    Version 1.1
//    08/19/04  removed Reinitalisation Routine Call
//    Version 1.2
//    01/17/05  Added Temp. Compensation function for PhaseShift
//    03/01/05  Added Supply Voltage supervision with SVS
//    03/07/05  Moved checkkeys function into main routine
//    03/07/05  Removed Averaging function
//
//**************************************************************************


/* ***********************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR
* REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY,
* INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR
* COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET
* POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY
* INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR
* YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
* CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY
* THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED
* OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT
* OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM.
* EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF
* REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS
* OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF
* USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S
* AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF
* YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS
* (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted
* by Texas Instruments is distributed as "freeware".  You may,
* only under TI's copyright in the Program, use and modify the
* Program without any charge or restriction.  You may
* distribute to third parties, provided that you transfer a
* copy of this license to the third party and the third party
* agrees to these terms by its first use of the Program. You
* must reproduce the copyright notice and any other legend of
* ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains
* copyrighted material, trade secrets and other TI proprietary
* information and is protected by copyright laws,
* international copyright treaties, and trade secret laws, as
* well as other intellectual property laws.  To protect TI's
* rights in the Program, you agree not to decompile, reverse
* engineer, disassemble or otherwise translate any object code
* versions of the Program to a human-readable form.  You agree
* that in no event will you alter, remove or destroy any
* copyright notice included in the Program.  TI reserves all
* rights not specifically granted under this license. Except
* as specifically provided herein, nothing in this agreement
* shall be construed as conferring by implication, estoppel,
* or otherwise, upon you, any license or other right under any
* TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
* ********************************************************* */

#include "device.h"
#include "comms_uart.h"
#include "subroutines.h"
#include "uart0.h"
#include "TimerA.h"
#include "SendData.h"
#include "fet4xx_rtclcd.h"
#include "display.h"
#include "emeter.h"
#include "parameter.h"
#include "portfunc.h"

#include <stdio.h>
#include <math.h>

// const and var definition
const char banner[]="MSP430FE427   ";
const unsigned int sbanner = 14;
const char TempBanner[]="\rTemp: ";

unsigned int OP_Mode = idle;   // operation mode
unsigned int sys_status = 0;   // System status bits

// forward declarations

//====================================================================
  // Init Digital Hardware
void init_system(void)
{
    P1OUT = 0x03;
    P2OUT = 0x00;
    P1DIR = 0xFF;  // P1.0 + P1.1 = LED Output
                   // P1.2 - P1.7 = LCD
    P2DIR = 0xC3;  // P2.0 + P2.1 = LCD
                   // P2.2 + P2.3 = switch input
                   // P2.4 + P2.5 = UART


    //P1DIR |= 0x02; // MCLK out on - test
    //P1SEL |= 0x02; // MCLK out on - test
    P1SEL |= 0xF4;    // Enable P1.2-1.7 as LCD drivers S26-S31
    P2SEL |= 0x03;    // Enable P2.0-2.1 as LCD drivers S24-S25

    P1IE  = 0x00;  // Disable Port 1 interrupts
    P2IE  = 0x00;  // Disable Port 2 interrupts

    P1IFG = 0x00;  // Clear Port 1 interrupt Flags
    P2IFG = 0x00;  // Clear Port 2 interrupt Flags

#ifdef TIMERA_PULSE_OUTPUT
    Init_TimerA(); // Init Timer A
#endif

    FCTL2 = FWKEY | FSSEL_1 | defFlashFreq;  // Configure Flash Timing Generator

#ifdef withDisplay
#if   (LCD_SIZE == 7)
    LCDCTL = LCDON + LCD4MUX + LCDP2 + LCDP1+ LCDP0; // Enable 7-segment LCD
#else
    LCDCTL = LCDON + LCD4MUX + LCDP2 + LCDP0; // Enable 6-segment LCD
#endif
    BTCTL  = BTHOLD + BTFRFQ1; // f(LCD) = ACLK / 256 = 128 Hz

#if LCD_TEST_PATTERN > 0
   DisplayTest();
#endif

    DisplayClear();
#endif // withDisplay
}


int __low_level_init(void)
{
  WDTCTL = WDTPW + WDTHOLD;             // Stop WDT
  return (1);         // Force initialization of RAM
//  return (0);         // Skip initialization of RAM
}

//====================================================================
// Wait in low-power mode.  Handle interrupts as they occur.
void WaitLPM(void)
{
#ifdef withUARTComm
  static volatile char LPM_on = 1;
  if ((TX_Mode < tx_ch1) || (TX_Mode > tx_ch3))
      {
          if (LPM_on == 1)
          if (OP_Mode == idle)
          {  LPM3;  // switch MSP430 into Low Power Mode 3
          }
          if (OP_Mode == measure)
          {  LPM0;  // switch MSP430 into Low Power Mode 0
          }
          _NOP();
      }
    else sys_status |= NewValues;

    // process UART receive
    if((UART_Status & LineReceived) != 0)
    {
        Process_UART();
        UART_Status &= ~LineReceived;     // clear flag
        if ( OP_Mode == request_cal)
        {
          TX_Mode = tx_off;
          start_calibration();
        }
    }

    // process UART send
    if((sys_status & NewValues) != 0)
    {
       sys_status &= ~NewValues;
       SendData();
    }

    if (SVSCTL & 0x01){
       SendString("Vcc Low\r");
    }
#else
    if (OP_Mode == idle)
    {  LPM3;  // switch MSP430 into Low Power Mode 3
    }
    if (OP_Mode == measure)
    {  LPM0;  // switch MSP430 into Low Power Mode 0
    }
    _NOP();
#endif // withUARTComm

    // process Calculations
    if((sys_status & DoCalc) != 0)
    {
       sys_status &= ~DoCalc;
    }


    // Temp. Measurement requested ?
    if((sys_status & RequestTemp) != 0)
    {
        sys_status &= ~RequestTemp;
        MBOUT0= mTEMP;               // request a Temp. Measurement
    }

/*
    // Do reinitialisation
    if((sys_status & ReInit) != 0)
    {
        _DINT();
        init_esp_parameters(0);
        sys_status &= ~ReInit;      // clearset flag to request reinit
        _EINT();
    }
*/

#ifdef withDisplay
    if (SVSCTL & 0x01){
       LCDtext ((unsigned char *) "Vcc Lo", 6, 6);
    }else
    DisplayData();
#endif // withDisplay
    SVSCTL &= ~0x01;


    if((sys_status & SecTick) != 0)
    {
      static unsigned char temp_count = defTemp_repeat;

      sys_status &= ~SecTick;

      temp_count--;
      if ((temp_count == 0) && (defTemp_repeat != 0)) {
        //
        //  * Every minute a temperature measurement is requested.
        //
        temp_count = defTemp_repeat;
        sys_status |= RequestTemp;       // set mark that a Temp. Measurement is requested
      }

    }


#ifdef withTempCorrection
    if((sys_status & NewTemp) != 0)
    {
      long lvalue;

      sys_status &= ~NewTemp;
      //temperature (assume for in range from -40 to 85)
      // lvalue in CDegree *100
      lvalue = (unsigned long) (((unsigned long)temperature * (s_parameters.ulTempGain))/100 - (s_parameters.ulTempOffset));
      lvalue /= 100; // lvalue is now in degree
      if (lvalue < -40) lvalue = -40;
      if (lvalue > 85) lvalue = 85;
      lvalue = (lvalue)/10 + 4;
      // lvalue now is now in the range of 0 ( = -40) to 13 ( = 90(85)
      // so a table with 14 entries is required
      set_parameter(mSET_PHASECORR1, (int)s_parameters.pSET_PHASECORR1 + phasecorr_table[lvalue]);
      set_parameter(mSET_PHASECORR2, (int)s_parameters.pSET_PHASECORR2 + phasecorr_table[lvalue]);
      _EINT();

    }
#endif // withTempCorrection

}

//====================================================================
void main (void)
{
  WDTCTL = WDTPW + WDTHOLD;             // Stop WDT

 // Init. FLL and clocks
  init_fll(10, defSystemFreq, 32);

  // Init Digital Hardware
  init_system();

 // Init. analog front-end (sigma-delta ADC)
  init_analog_front_end();
  // Init. Embedded Signal Processing parameters
  init_esp(1);

  // Init. UART
#ifdef withUARTComm
//  InitUART(1200);
//  InitUART(9600);
  InitUART(57600);
//  InitUART(115200);
//  InitUART(230400);
  TX_Mode = tx_off;
  SendString("\r\r");
  SendString(banner);
  LCDtext ((unsigned char *) banner, 6, 6);
  SendString("Firmware Version: ");
  SendResult((unsigned char*) &firmware_version, 2);
  SendChar('\r');
  SendChar('\n');
  SendChar('\r');
#endif // withUARTComm

 // Init. RTC and Display
#ifdef withDisplay
  SetupClock();
  LCDtext ((unsigned char *) banner, 6, 6);
#endif // withDisplay

  // Init. event generation and start measurement
  start_measurement();

  SVSCTL = 0x70;  // set SVS to 2.65 V to detect voltage drops below save operating area

  _EINT(); // Enable Interrupts

  while (1)
  {
   WaitLPM();   // Wait in low-power mode.  Return after interrupt
   checkKeys(); // Check if a button has been pressed
  } // End of while (1)
} // End of main()


