//**************************************************************************
//    UART commmunication for the FE427
//
//    Stefan Schauer
//    Texas Instruments Freising
//    Date        Comments
//    =====================
//    07/01/03    Code Starts for FE427
//                Added 'SP' Command to read/modify ESP430 Parameters
//
//    Version 1.0 first release for Application Report
//    06/14/04
//    Version 1.1
//    08/19/04  added calibration of POFFSETx
//    Version 1.2
//    12/23/04  changed calibration function (+/-) to use values instead of
//                steps
//    Version 1.3
//    11/23/07  added calibration of IADAPTx value
//              added calibration of CORRCOMP value
//              added more settings for UART Baud rate
//              added Pulse output selection
//              added BitMask selection
//
//**************************************************************************

/* ***********************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR
* REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY,
* INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR
* COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET
* POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY
* INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR
* YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
* CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY
* THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED
* OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT
* OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM.
* EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF
* REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS
* OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF
* USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S
* AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF
* YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS
* (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted
* by Texas Instruments is distributed as "freeware".  You may,
* only under TI's copyright in the Program, use and modify the
* Program without any charge or restriction.  You may
* distribute to third parties, provided that you transfer a
* copy of this license to the third party and the third party
* agrees to these terms by its first use of the Program. You
* must reproduce the copyright notice and any other legend of
* ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains
* copyrighted material, trade secrets and other TI proprietary
* information and is protected by copyright laws,
* international copyright treaties, and trade secret laws, as
* well as other intellectual property laws.  To protect TI's
* rights in the Program, you agree not to decompile, reverse
* engineer, disassemble or otherwise translate any object code
* versions of the Program to a human-readable form.  You agree
* that in no event will you alter, remove or destroy any
* copyright notice included in the Program.  TI reserves all
* rights not specifically granted under this license. Except
* as specifically provided herein, nothing in this agreement
* shall be construed as conferring by implication, estoppel,
* or otherwise, upon you, any license or other right under any
* TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
* ********************************************************* */

#include "parameter.h"
#include "device.h"
#include "comms_uart.h"
#include "SendData.h"
#include "subroutines.h"
#include "display.h"
#include "uart0.h"
#include "fet4xx_rtclcd.h"
#include "emeter.h"
#include "calibration.h"

#include <math.h>
#include <stdio.h>
#include <string.h>      // Needed for strcpy()

#ifdef withUARTComm

#define   ldfs   "%ld"

// ************************************************************
// Definitions & Declarations
// ************************************************************

unsigned char Cal_Mode = 0;

const char * txt_help = "\
UART commands:\r\
 SHxx: set hour\r\
 SMxx: set minutes\r\
 SSxx: set seconds\r\
 SDxx: set day\r\
 SOxx: set month\r\
 SYxx: set year\r\
 SIxx: set calibration current\r\
 SVxx: set calibration voltage\r\
\r\
 Dx: set Display mode\r\
  1: Off\r\
  2: Time\r\
  3: Date\r\
  4: Voltage (V)\r\
  5: Current (A)\r\
  6: PeakVoltage (V)\r\
  7: PeakCurrent (A)\r\
  8: Frequency (Hz)\r\
  9: CosPhi\r\
 10: Temp\r\
 11: Power (kW)\r\
 12: Energy (kWh)\r\
 Vx: value of single measurement\r\
  1: Off\r\
  2: Time\r\
  3: Date\r\
  4: Voltage (V)\r\
  5: Current (A)\r\
  6: PeakVoltage (V)\r\
  7: PeakCurrent (A)\r\
  8: Frequency (Hz)\r\
  9: CosPhi\r\
 10: Temp\r\
 11: Power (kW)\r\
 12: Energy (kWh)\r\
\r\
 H : show help test\r\
 Tx: set test dump mode\r\
 Qx: query dump\r\
 R : reset system\r\
 Wxx: write message to LCD display\r\
 Mx: execute calibration measuerment over x*50 cycles\r\
 I : init\r\
\r\
  C0: auto calibration of U / I / P / PhaseShift\r\
  C1: calibration of Interrupt Level\r\
  C2: calibration of Phase correction 1\r\
  C3: calibration of Phase correction 2\r\
  C4: calibration of V1 Offset\r\
  C5: calibration of I1 Offset\r\
  C6: calibration of I2 Offset\r\
  C7: calibration of Gain Correction 1\r\
  C8: calibration of Gain Correction 2\r\
  C9: save settings to flash\r\
 C10: calibration of V Ratio\r\
 C11: calibration of I Ratio\r\
 C12: calibration of Energy Ratio\r\
 C13: calibration of Power1 Offset\r\
 C14: calibration of Power2 Offset\r\
 C17: calibration of Adaption I1 Correction 1\r\
 C18: calibration of Adaption I2 Correction 2\r\
 C19: calibration CMMR\r\
 +xxx : inc values for calibration\r\
 -xxx : dec values for calibration\r\
";

// ************************************************************
// functions forward declarations
// ************************************************************

int Digit0(unsigned char Register);
int Digit1(unsigned char Register);
int Hex2ASCII(int hex);

// ************************************************************
// functions
// ************************************************************

void InitUART(unsigned long baud)
{
  //UART
  // Configure USART in UART mode, 8-bit, 1-stop, no parity.
  // Hold logic in reset state while configuring other registers

  UCTL0 =  (0*PENA)      // 7 Parity Enable         0=Disabled
          |(0*PEV)      // 6 Parity Select         0= Odd
        |(0*SPB)      // 5 Stop Bit Select      0= 1 Stop bit
        |(1*CHAR)      // 4 Character Length      1= 8-bit data
        |(0*LISTEN)   // 3 Loopback control      0= Loopback disabled
        |(0*SYNC)      // 2 Synchronous Mode      0= UART mode
        |(0*MM)      // 1 Multiprocessor Mode   0= Idle-line multiproc control
        |(1*SWRST);   // 0 Software Reset         1= Logic Held in Reset State ...
                  //                             while configuring other registers

switch(baud)
{
#if (MCLK_FREQ == 4)
  case 1200:
    UTCTL0 = SSEL0;                     // UCLK = ACLK
    UBR00 = 0x1B;                         // 32k/9600
    UBR10 = 0x00;                         //
    UMCTL0 = 0x03;                        //
    break;
  case 9600:
    UTCTL0 = SSEL0;                     // UCLK = ACLK
    UBR00 = 0x03;                         // 32k/9600
    UBR10 = 0x00;                         //
    UMCTL0 = 0x4a;                        //
    break;
  case 19200:
    UTCTL0 = SSEL1;                       // UCLK = SMCLK
    UBR00 = 0xDA;                         // 4MHz/19200
    UBR10 = 0x00;                         //
    UMCTL0 = 0x55;                        //
    break;
  case 57600:
    UTCTL0 = SSEL1;                       // UCLK = SMCLK
    UBR00 = 0x48;                         // 4Mhz/57600
    UBR10 = 0x00;                         //
    UMCTL0 = 0x7B;                        //
    break;
  case 115200:
    UTCTL0 = SSEL1;                       // UCLK = SMCLK
    UBR00 = 0x24;                         // 4Mhz/115.2k
    UBR10 = 0x00;                         //
    UMCTL0 = 0x29;                        //
    break;
  case 230400:
    UTCTL0 = SSEL1;                       // UCLK = SMCLK
    UBR00=0x12;
    UBR10=0x00;
    UMCTL0=0x84; /* uart0 4194304Hz 230456bps */
    break;
#endif

#if (MCLK_FREQ == 8)
  case 1200:
    UTCTL0 = SSEL0;                     // UCLK = ACLK
    UBR00 = 0x1B;                         // 32k/9600
    UBR10 = 0x00;                         //
    UMCTL0 = 0x03;                        //
    break;
  case 9600:
    UTCTL0 = SSEL0;                     // UCLK = ACLK
    UBR00 = 0x03;                         // 32k/9600
    UBR10 = 0x00;                         //
    UMCTL0 = 0x4a;                        //
    break;
  case 57600:
    UTCTL0 = SSEL1;                       // UCLK = SMCLK
    UBR00 = 0x91;                         // 8Mhz/57600
    UBR10 = 0x00;                         //
    UMCTL0 = 0xAD;                        //
    break;
  case 230400:
    UTCTL0 = SSEL1;                       // UCLK = SMCLK
    UBR00=0x24;
    UBR10=0x00;
    UMCTL0=0x2A; /* uart0 4194304Hz 230456bps */
    break;
#endif

  default:
    break;
  }
  U0ME |= UTXE0+URXE0 ;                 // Enabled USART0 TXD/RXD

  P2SEL |= (BIT4 + BIT5);               // P2.4,5 = USART0 TXD/RXD
  P2DIR |= BIT4;                        // P2.4 output direction

  U0IFG &= ~URXIFG0;                    // Clear USART0 RX interrupt flag

  UCTL0 &= ~(SWRST);                    // 8-bit character - clr SWRST bit

//  U0IE |= URXIE0+UTXIE0;                // Enable USART0 RX + TX interrupt
  U0IE |= URXIE0;                       // Enable USART0 RX interrupt
}

void SendString(const char* ARRAY)
{
int i=0;

   while (ARRAY[i])          // transmit " String" via RS232
   {
      SendChar(ARRAY[i]);
      i++;
   }
}

void SendResult(unsigned char* ARRAY, unsigned int length)
{
   int i;

   for (i=length-1;(i+1)>0;i--)
   {  SendChar(Digit1(ARRAY[i]));  // transmit digit 1
      SendChar(Digit0(ARRAY[i]));  // transmit digit 0
   }
}

void SendFloat(uint32_t lvalue, unsigned int after)
{
   int i;
   int digitval;      // ASCII version of next character to send
   int LZSflag = 1;      // Leading-zero supression flag
   uint8_t bcd[5];

   bin2bcd32(bcd,lvalue);         // Convert from 32 bit to BCD

   for (i=10; i>after; i--)
   {
      if ((i&1) == 0)      // Test for odd or even
      {
         digitval = Digit1(bcd[(10-i)/2]);  // Even: high digit
      }
      else
      {
         digitval = Digit0(bcd[(10-i)/2]);  // Odd: low digit
      }

      if (digitval > '0')
         {LZSflag = 0;}         // Clear LZS flag when non-zero character seen

      if (  (digitval != '0')
         ||(LZSflag == 0)
         ||(i == after+1) )
      {
         SendChar(digitval);
      }
   }
   if(after > 0)
   {
      SendChar('.');
      for (i=after; i>0; i--)
      {
         if ((i&1) == 0)      // Test for odd or even
         {
            digitval = Digit1(bcd[(10-i)/2]);  // Even: high digit
         }
         else
         {
            digitval = Digit0(bcd[(10-i)/2]);  // Odd: low digit
         }

         SendChar(digitval);
      }
   }
}

void SendChar(char ch)
{
   while ((U0IFG&UTXIFG0)==0);            // wait till TX buf empty
   TXBUF0=ch;                        // transmit ch
}

int Digit0(unsigned char Register)         // return LSB nibble (digit 0)
{  int result;

   result = Hex2ASCII(0x0F & Register);
   return result;
}

int Digit1(unsigned char Register)         // return nibble (digit 1)
{  int result;

   result = Register >> 4;
   result = Hex2ASCII(0x0F & result);
   return result;
}

int Hex2ASCII(int hex)        // hexadecimal to ASCII conversion
{  int result;

   if (hex<=9)
   {  result=hex+'0'; }       // convert number
   else
   {  result=hex+('A'-10); }  // convert letter

   return result;
}


//;--------------------------------------------------------------
//; This routine processes a received UART Command
void Process_UART(void)
{
  int   temp;         // Temporary variable
  unsigned int   parm;   //

  if (UART_Status & LineReceived)
  {
     //UARTSet
     switch (UART_RX_Buffer[0])
     {
       case 'S':
       //SetClock
         switch (UART_RX_Buffer[1])
         {
           case 'H':
               HOUR = GetBCDNumber(&UART_RX_Buffer[2]);
             break;
           case 'M':
               MIN = GetBCDNumber(&UART_RX_Buffer[2]);
             break;
           case 'S':
               SEC = GetBCDNumber(&UART_RX_Buffer[2]);
             break;
            case 'Y':
               YEAR = GetBCDNumber(&UART_RX_Buffer[2]);
             break;
           case 'O':
               MONTH = GetBCDNumber(&UART_RX_Buffer[2]);
             break;
           case 'D':
               DAY = GetBCDNumber(&UART_RX_Buffer[2]);
             break;
#ifdef MAINCPU_ENERGYPULSE
           case 'E':
               cMainCPU_EnPulseMode = GetBCDNumber(&UART_RX_Buffer[2]);
             break;
#endif
           case 'B':
               testBitMask = GetHexNumber(&UART_RX_Buffer[2]);
               // Set event message request flags:
               set_parameter(mSET_EVENT, defSET_EVENT | testBitMask );
               _EINT();
             break;
           case   'V':
               calVoltage = GetNumber(&UART_RX_Buffer[2]);
             break;
           case   'I':
               calCurrent = GetNumber(&UART_RX_Buffer[2]);
             break;
#ifdef ESP_V2
           case   'X':
/*               _DINT();
               _NOP();
               MBOUT1= 0; //ESP_MEASURE;
               MBOUT0= mCMD_SET_RES_SC;
               _EINT();
*/             
             break;
#endif
           case   'P':
              {
                  unsigned int param = 0;
                  unsigned int data = 0;
                  unsigned int num = 2;
                  param = GetNumberPos(&UART_RX_Buffer[num], &num);
                  if (UART_RX_Buffer[num] == 0)
                  {
                     data = read_parameter(param);
                     SendString("0x");
                     SendResult((unsigned char*) &data, 2);
                  }
                  else
                  {
                     data = GetNumberPos(&UART_RX_Buffer[num], &num);
                     set_parameter(param, data);
                     _EINT();
                  }
                  SendString("\r");
              }
             break;
           case   'R':
              {
                  unsigned int param = 0;
                  unsigned int data = 0;
                  unsigned int num = 0;
                  unsigned int i = 0;
				  
                  param = 0xB0;num = 0;i = 0;
                  while (num < 0x10)
                  {
                     data = read_parameter(param+num);
                     SendString("0x");
                     SendResult((unsigned char*) &data, 2);
                     SendString(" ");
                     num+=2;
                     if (++i == 8){i = 0; SendString("\r");}
                  }
                  SendString("\r");
                  param = 0x200;num = 0;i = 0;
                  while (num < 0x100)
                  {
                     data = read_parameter(param+num);
                     SendString("0x");
                     SendResult((unsigned char*) &data, 2);
                     SendString(" ");
                     num+=2;
                     if (++i == 8){i = 0; SendString("\r");}
                  }
                  SendString("\r");
              }
             break;
            default:
               break;
         }
         break;
#ifdef withDisplay
       case 'D':
           //SetDisplay_Mode
             LCDtext ((unsigned char *) "       ", LCD_SIZE, LCD_SIZE);  // Clear the display
        Display_Mode = GetNumber(&UART_RX_Buffer[1]);
        ModeSwitchDelay = 60;   // Stay in commanded display mode for 1 minute before auto-updating display mode
        Display_Hold = 2;   // Display units for 2 seconds before showing values
           break;

       case 'W':
      // Write to LCD display
        Display_Mode = display_msgin;
        strcpy(msgin, (char *)&UART_RX_Buffer[1]);
        ModeSwitchDelay = 60;   // Stay in commanded display mode for 1 minute before auto-updating display mode
           break;

#endif // withDisplay
       case 'H':
       //send help text
           SendString(txt_help);
         break;

       case 'T':
       //SetTX_Mode
           TX_Mode = GetNumber(&UART_RX_Buffer[1]);

           if ((TX_Mode >= tx_ch1) && (TX_Mode <= tx_ch3))
//             set_parameter(mSET_EVENT, defSET_EVENT | WFSRDYME);
             set_parameter(mSET_EVENT, WFSRDYME);
           else
             set_parameter(mSET_EVENT, defSET_EVENT);
           _EINT();

         break;

       case 'Q':
       //Query
      temp= TX_Mode;         // Temporarily save TX_Mode
           TX_Mode = GetNumber(&UART_RX_Buffer[1]);
           SendData();
      TX_Mode = temp;         // Restore TX_Mode
         break;

       case 'V':
      //Value
           parm = GetNumber(&UART_RX_Buffer[1]);
           SendValue(parm);
         break;

       case 'R':
           WDTCTL =0;  // Generate WDT violation Reset
         break;

       case 'M':
       //SetMeasureMode
            TX_Mode = 0; //                  ; Clear running TX_Mode
            OP_Mode = request_cal;
            CalCyclCnt = defCalFreq*GetNumber(&UART_RX_Buffer[1]);    // Set Cycles
         break;

       case 'I':
       //Init
            _DINT(); // Disable Interrupts
            WDTCTL = WDTPW + WDTHOLD;             // Stop WDT
            // Init Digital Hardware
            init_system();
            // Init. FLL and clocks
            init_fll(10, defSystemFreq, 32);
           // Init. analog front-end (sigma-delta ADC)
            init_analog_front_end();
            // Init. Embedded Signal Processing parameters
            init_esp(1);
            // Init. event generation and start measurement
            start_measurement();
            _EINT(); // Enable Interrupts
         break;

#ifdef withCalibration
       case '+':
            CalPlus(Cal_Mode,GetBCDNumber(&UART_RX_Buffer[1]));
         break;

       case '-':
            CalMinus(Cal_Mode, GetBCDNumber(&UART_RX_Buffer[1]));
         break;

       case 'C':
       //Set Calibration mode
         {

            static char strbuf[20];

            _DINT(); // Disable Interrupts
         Cal_Mode = GetNumber(&UART_RX_Buffer[1]);
         SendString("Cal_Mode = ");
         SendFloat(Cal_Mode,0);
         SendString("\r");

         if (Cal_Mode == 0)
            {
            SendString("Start Calibration : expecting ");
            SendFloat(calVoltage,0); SendString("V / ");
            SendFloat(calCurrent,0); SendString("A / cosPhi = ");
            SendFloat(defCalCosPhi*1000,3); SendString("\r");

               SendString("Old Data : \r ");
               sprintf(strbuf, ldfs,(long) (s_parameters.VRatio*1000)); SendString(strbuf); SendChar('\r');
               sprintf(strbuf, ldfs,(long) (s_parameters.IRatio*1000)); SendString(strbuf); SendChar('\r');
               sprintf(strbuf, ldfs,(long) (s_parameters.EnergyRatio*1000)); SendString(strbuf); SendChar('\r');
               sprintf(strbuf, ldfs,(long) s_parameters.pSET_PHASECORR1); SendString(strbuf); SendChar('\r');
               sprintf(strbuf, ldfs,(long) s_parameters.pSET_PHASECORR2); SendString(strbuf); SendChar('\r');
         }

         UART_Status &= ~LineReceived;
         UART_RX_Bytes = 0;
         _EINT(); // Enable Interrupts

            CalMode(Cal_Mode);

            _DINT(); // Disable Interrupts

            if (Cal_Mode == 0)
            {
               SendString("Calibration Done\r");
               SendString("New Data : \r ");
               sprintf(strbuf, ldfs,(long) (s_parameters.VRatio*1000)); SendString(strbuf); SendChar('\r');
               sprintf(strbuf, ldfs,(long) (s_parameters.IRatio*1000)); SendString(strbuf); SendChar('\r');
               sprintf(strbuf, ldfs,(long) (s_parameters.EnergyRatio*1000)); SendString(strbuf); SendChar('\r');
               sprintf(strbuf, ldfs,(long) s_parameters.pSET_PHASECORR1); SendString(strbuf); SendChar('\r');
               sprintf(strbuf, ldfs,(long) s_parameters.pSET_PHASECORR2); SendString(strbuf); SendChar('\r');
            }
            if (Cal_Mode == 9) { SendString("Parameters saved\r"); }
         _EINT(); // Enable Interrupts
         }
         break;
#endif // withCalibration

     }

    UART_Status &= ~LineReceived;
    UART_RX_Bytes = 0;

  }
}
//No_Process_Uart
//        ret

#endif // withUARTComm
