//*******************************************************************************
//
//  Description "FET410_RTCwLCD": This program maintains a real time clock and
//  displays the current time on a 3.5 digit static LCD. The colon between the
//  minutes and seconds toggles on or off each second to indicate that the clock
//  is running.  The digits of the display are only rewritten each minute once
//  the elapsed seconds roll over from 60 back to 0. The fourth digit is
//  automatically blank for hours 1-9. The software is totally interrupt driven.
//  It only comes out of low power mode 3 and executes code when a Basic Timer
//  interrupt occurs, once each second.
//
//*******************************************************************************

//*******************************************************************************
//  Modified Stefan Schauer
//  date   11/03/2004
//  Changes:
//     - SetArrow function
//
//    Version 1.0 converted to C
//    11/03/2004
//
//*******************************************************************************


//*******************************************************************************
// THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR
// REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY,
// INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
// FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR
// COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
// TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET
// POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY
// INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR
// YOUR USE OF THE PROGRAM.
//
// IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
// CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY
// THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT
// OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM.
// EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF
// REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS
// OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF
// USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S
// AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF
// YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS
// (U.S.$500).
//
// Unless otherwise stated, the Program written and copyrighted
// by Texas Instruments is distributed as "freeware".  You may,
// only under TI's copyright in the Program, use and modify the
// Program without any charge or restriction.  You may
// distribute to third parties, provided that you transfer a
// copy of this license to the third party and the third party
// agrees to these terms by its first use of the Program. You
// must reproduce the copyright notice and any other legend of
// ownership on each copy or partial copy, of the Program.
//
// You acknowledge and agree that the Program contains
// copyrighted material, trade secrets and other TI proprietary
// information and is protected by copyright laws,
// international copyright treaties, and trade secret laws, as
// well as other intellectual property laws.  To protect TI's
// rights in the Program, you agree not to decompile, reverse
// engineer, disassemble or otherwise translate any object code
// versions of the Program to a human-readable form.  You agree
// that in no event will you alter, remove or destroy any
// copyright notice included in the Program.  TI reserves all
// rights not specifically granted under this license. Except
// as specifically provided herein, nothing in this agreement
// shall be construed as conferring by implication, estoppel,
// or otherwise, upon you, any license or other right under any
// TI patents, copyrights or trade secrets.
//
// You may not use the Program in non-TI devices.
//*******************************************************************************

#include "parameter.h"

#if PCB == TI_BOARD

#include "device.h"
#define _def_clock
#include "fet4xx_rtclcd.h"

//--------RAM bytes used for variables-------------------------------------------

unsigned char SEC  =  0x00;
unsigned char MIN  =  0x00;
unsigned char HOUR =  0x12;

unsigned char DAY   =  0x01;
unsigned char MONTH =  0x01;
unsigned char YEAR  =  0x04;

//-------------------------------------------------------------------------------
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
// LCD  Definitions
// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

// LCD_TYPE
// Top Digit at LCD
// SB LCD
#define ta           0x10
#define tb           0x01
#define tc           0x04
#define td           0x08
#define te           0x40
#define tf           0x20
#define tg           0x02
#define tdp          0x80

const unsigned char TLCD_Tab[] = {
            ta+tb+tc+td+te+tf,            // displays "0"
            tb+tc,                        // displays "1"
            ta+tb+td+te+tg,               // displays "2"
            ta+tb+tc+td+tg,               // displays "3"
            tb+tc+tf+tg,                  // displays "4"
            ta+tc+td+tf+tg,               // displays "5"
            ta+tc+td+te+tf+tg,            // displays "6"
            ta+tb+tc,                     // displays "7"
            ta+tb+tc+td+te+tf+tg,         // displays "8"
            ta+tb+tc+td+tf+tg,            // displays "9"
            ta+tb+tc+te+tf+tg,            // displays "A"
            tc+td+te+tf+tg,               // displays "B"
            ta+td+te+tf,                  // displays "C"
            tb+tc+td+te+tg,               // displays "D"
            ta+td+te+tf+tg,               // displays "E"
            ta+te+tf+tg,                  // displays "F"
         };


// SB LCD
#define a            0x08
#define b            0x04
#define c            0x02
#define d            0x01
#define e            0x20
#define f            0x40
#define dp           0x10
#define colon        0x80

#define g            0x4000
#define h            0x8000
#define j            0x0800
#define k            0x0400
#define m            0x0200
#define n            0x0100
#define q            0x2000   //1000h
#define p            0x1000   //2000h

//--- character definitions
// decode special Char to be placed at end of the table
const unsigned int CHAR_SPACE = 0;
const unsigned int CHAR_MINUS = g+m;

//#define CHAR_MINUS   LCD_mi
//#define LCD_mi       LCD_sp[13]
const unsigned short LCD_sp[] = {
            0,                         // displays " "
            j+p,                      // displays "!"
            b+f,                      // displays """
            a+b+c+d+e+f+g+j+m+p,      // displays "#"
            a+f+g+m+c+d+j+p,          // displays "$"
            c+f+k+q,                  // displays "%"
            d+h+j+n+q,                // displays "&"
            j,                        // displays "'"
            j+k+n+p,                  // displays "("
            h+j+p+q,                  // displays ")"
            g+h+j+k+m+n+p+q,          // displays "*"
            g+j+m+p,                  // displays "+"
            q,                        // displays ","
            g+m,                      // displays "-"
            dp,                       // displays "."
            k+q,                      // displays "/"
       };

const unsigned short LCD_Tab[] = {
           a+b+c+d+e+f+k+q,           // displays "0" with diagonal slash
           b+c,                        // displays "1"
           a+b+d+e+g+m,                // displays "2"
           a+b+c+d+g+m,                // displays "3"
           b+c+f+g+m,                  // displays "4"
           a+c+d+f+g+m,                // displays "5"
           a+c+d+e+f+g+m,              // displays "6"
           a+b+c,                      // displays "7"
           a+b+c+d+e+f+g+m,            // displays "8"
           a+b+c+d+f+g+m,              // displays "9"
           colon,                      // displays ":"
           colon+q,                    // displays ";"
           k+n,                        // displays "<"
           d+g+m,                      // displays "="
           h+q,                        // displays ">"
           a+b+m+p,                    // displays "?"
           a+b+d+e+f+g+m,              // displays "@"
           a+b+c+e+f+g+m,              // displays "A"
           c+d+e+f+g+m,                // displays "B"
           a+d+e+f,                    // displays "C"
           b+c+d+e+g+m,                // displays "D"
           a+d+e+f+g,                  // displays "E"
           a+e+f+g,                    // displays "F"
           a+c+d+e+f+m,                // displays "G"
           b+c+e+f+g+m,                // displays "H"
           j+p +a+d,                   // displays "I"
           b+c+d+e,                    // displays "J"
           e+f+g+k+n,                  // displays "K"
           d+e+f,                      // displays "L"
           b+c+e+f+h+k,                // displays "M"
           e+f+h+n+c+b,                // displays "N"
           a+b+c+d+e+f,                // displays "O" big
           a+b+e+f+g+m,                // displays "P"
           a+b+c+d+e+f+n,              // displays "Q"
           e+f+a+b+m+g+n,              // displays "R"
           a+c+d+f+g+m,                // displays "S"
           a+j+p,                      // displays "T"
           c+d+e+f+b,                  // displays "U"
           e+f+q+k,                    // displays "V"
           e+f+q+n+c+b,                // displays "W"
           h+n+q+k,                    // displays "X"
           h+k+p,                      // displays "Y"
           a+k+q+d,                    // displays "Z"

           e+f+a+d+h+q,                // displays "["
           h+n,                        // displays "\"
           a+d+b+c+k+n,                // displays "]"
           a+f+b,                      // displays "^"
           d,                          // displays "_"
           h,                          // displays "`"
     };

//LCD_Tab_Length equ   $ - LCD_Tab

//LCD_Tab_End

void Inc_Month (void);


//-------------------------------------------------------------------------------
//             Increment clock values every second
//-------------------------------------------------------------------------------
void UpdateClock (void)
{
//????              setc                             ;entry every second
              SEC = __bcd_add_short(1, SEC);         // increment seconds, add carry bit
              if (SEC == 0x60)
              {  SEC = 0;
                 MIN = __bcd_add_short(1, MIN);      // increment minutes if set carry
                 if (MIN == 0x60)                // sixty minutes elapsed?
                 {   MIN = 0;                    // yes, clear minutes
                     HOUR = __bcd_add_short(1, HOUR);   // increment hour
                     if (HOUR == 0x24)              // 24 hours elapsed?
                     {   HOUR = 0;
                        DAY = __bcd_add_short(1, DAY);   // increment DAY
                        if (MONTH == 0x02)
                        {   if (((YEAR & 0x03)==0) && (DAY == 0x29)) Inc_Month();     //Test for leap year (2 lowest bits == 0)
                            else if (((YEAR & 0x03)!=0) && (DAY == 0x28)) Inc_Month();
                        }
                        else
                        {   if (MONTH < 0x08)
                            {   if (((MONTH & 0x01)==0x01) && (DAY == 0x31)) Inc_Month();
                            else if (((MONTH & 0x01)!=0x01) && (DAY == 0x30)) Inc_Month();
                            }
                            else
                            {   if (((MONTH & 0x01)==0x01) && (DAY == 0x30)) Inc_Month();
                                else if (((MONTH & 0x01)!=0x01) && (DAY == 0x31)) Inc_Month();
                            }
                        }
                    }
                 }
              }
              sys_status |= ReInit;           //set flag to request reinit
}

void Inc_Month (void)
{
        DAY = 0x01;                      // Set day to 1
        MONTH = __bcd_add_short(1, MONTH);       // increment MONTH
        if (MONTH == 0x13)
        { MONTH = 0x01;
          YEAR = __bcd_add_short(1, YEAR);      // increment YEAR
        }
}


//-------------------------------------------------------------------------------
//             Display clock values contained in RAM Bytes MIN & HOUR.
//             CPU Registers R12, R13 and R14 used temporarily.
//-------------------------------------------------------------------------------
void Dis_Clock (void)
{
              Display_BCD(SEC,1);          // call routine to display seconds
              Display_BCD(MIN,1+2*2);      // call routine to display minutes
              Display_BCD(HOUR,1+4*2);     // call routine to display hours

              if (SEC & 0x01)
              {  LCDM4 |= colon;     // toggle Colon
                 LCDM8 |= colon;     // toggle Colon
              }
}





void Dis_Date (void)
{
              Display_BCD(YEAR,1);           // call routine to display seconds
#if   date_format == MMDDYY      // US Format
              Display_BCD(DAY,1+2*2);        // call routine to display day
              Display_BCD(MONTH,1+4*2);      // call routine to display month
#else               // European Format
              Display_BCD(MONTH,1+2*2);      // call routine to display month
              Display_BCD(DAY,1+4*2);        // call routine to display day
#endif
              if (SEC & 0x01)
              {LCDM4 |= colon;    // toggle Colon
               LCDM8 |= colon;    // toggle Colon
              }
}

//Write an BCD value into the Display
//The BCD value consists of 2 Digits
//R12 contains the the BCD value
//R14 indicates the possition on the LCD where the values should be displayed
//  0 is the right border / one digit are two steps
void Display_BCD (unsigned char data, unsigned char pos)
{
              LCDMEM[pos]   = (unsigned char)(LCD_Tab[(data & 0x0F)]      & 0xFF);
              LCDMEM[pos+1] = (unsigned char)((LCD_Tab[(data & 0x0F)]>>8) & 0xFF);

              data = data >> 4;
              LCDMEM[pos+2]   = (unsigned char)(LCD_Tab[(data & 0x0F)]      & 0xFF);
              LCDMEM[pos+3] = (unsigned char)((LCD_Tab[(data & 0x0F)]>>8) & 0xFF);
}

//-------------------------------------------------------------------------------
//             Display 1 digit on static LCD. CPU Registers R12, R14 are used
//             for the parameters
//             R12 is char (R15 in CROSSWORKS)
//             R14 is position
//             pos = 0 means right aligned digit / 1 digit are 2 steps
//-------------------------------------------------------------------------------
void Char2LCD (unsigned char data, unsigned char pos)
{
              data -= '0';                     // subtract offset of char '0'
              BCD2LCD (data, pos);
}


void BCD2LCD (unsigned char data, unsigned char pos)
{
              //data = data << 1;                // transform for word table
              Direct2LCD ((unsigned int) LCD_Tab[data], pos);
}


void Direct2LCD (unsigned int data, unsigned char pos)
{
              LCDMEM[1+(pos<<1)] = (unsigned char)( data      & 0xFF);
              LCDMEM[2+(pos<<1)] = (unsigned char)((data >>8) & 0xFF);
}


//-------------------------------------------------------------------------------
void setPoint (unsigned char pos)
{
              LCDMEM[1+(pos<<1)] |= dp;
}

//-------------------------------------------------------------------------------
void setColon (unsigned char pos)
{
              LCDMEM[1+(pos<<1)] |= colon;
}


//-------------------------------------------------------------------------------
void SetArrow(unsigned char data)
{
              LCDM16 &= ~0x0F;                  // Turn off all arrows
              LCDM16 |= data & 0x0F;            // Turn set arrows
}

//-------------------------------------------------------------------------------
void SetupClock (void)   // Configure modules and control registers
{
              int i;
#if   (LCD_SIZE == 7)
              LCDCTL = LCDON+LCD4MUX+LCDP2+LCDP0; // Softbaugh LCD 4Mux, S0-S31
#else
              LCDCTL = LCDON+LCD4MUX+LCDP2        // STK LCD 4Mux, S0-S15
#endif

#ifdef P5SEL_
              P5SEL = 0FCh                        // Enable R(LCD) + COM Lines
#endif

//SetupBT
              BTCTL = BT_ADLY_1000;            // Basic Timer interrupt = 1 second
              BTCTL |=BTFRFQ1;                 // Set fLCD = ACLK / 128
              IE2 |= BTIE;                     // enable Basic Timer interrupt

//ClearLCD
              for (i=20; i != 0; i--) LCDMEM[i] =0;
}

//-------------------------------------------------------------------------------
//             Basic Timer causes 1 second interrupt. Mode bits changed on stack
//             so CPU is returned in active upon return from the interrupt
//-------------------------------------------------------------------------------

#ifdef __IAR_SYSTEMS_ICC__
#if __VER__ < 200
interrupt[BASICTIMER_VECTOR] void BT_ISR(void)
#else
#pragma vector=BASICTIMER_VECTOR
__interrupt void BT_ISR( void )
#endif
#endif

#ifdef __CROSSWORKS_MSP430
void BT_ISR(void)   __interrupt[BASICTIMER_VECTOR]
#endif

#ifdef __TI_COMPILER_VERSION__
#pragma vector=BASICTIMER_VECTOR
__interrupt void BT_ISR( void )
#endif

{
              UpdateClock();               // update hours and minutes each
                                           // second even if not displayed
              DisplayNextData();           // Switch to next display mode
                                           // every few seconds
              LPM3_EXIT;    // back to active mode after ISR
}

#endif  // PCB == TI_BOARD
