//*******************************************************************************
//
//  Initializes FLL+ and load capacitances for the 32kHz oscillator
//
//    Lutz Bierl
//    Texas Instruments Freising
//
//    Version 1.0 first release for Application Report
//    06/04/04
//
//*******************************************************************************

/* ***********************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR
* REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY, 
* INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS 
* FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR 
* COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE. 
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET 
* POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY 
* INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR 
* YOUR USE OF THE PROGRAM.
*
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, 
* CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY 
* THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED 
* OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT 
* OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. 
* EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF 
* REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS 
* OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF 
* USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S 
* AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF 
* YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS 
* (U.S.$500).
*
* Unless otherwise stated, the Program written and copyrighted 
* by Texas Instruments is distributed as "freeware".  You may, 
* only under TI's copyright in the Program, use and modify the 
* Program without any charge or restriction.  You may 
* distribute to third parties, provided that you transfer a 
* copy of this license to the third party and the third party 
* agrees to these terms by its first use of the Program. You 
* must reproduce the copyright notice and any other legend of 
* ownership on each copy or partial copy, of the Program.
*
* You acknowledge and agree that the Program contains 
* copyrighted material, trade secrets and other TI proprietary 
* information and is protected by copyright laws, 
* international copyright treaties, and trade secret laws, as 
* well as other intellectual property laws.  To protect TI's 
* rights in the Program, you agree not to decompile, reverse 
* engineer, disassemble or otherwise translate any object code 
* versions of the Program to a human-readable form.  You agree 
* that in no event will you alter, remove or destroy any 
* copyright notice included in the Program.  TI reserves all 
* rights not specifically granted under this license. Except 
* as specifically provided herein, nothing in this agreement 
* shall be construed as conferring by implication, estoppel, 
* or otherwise, upon you, any license or other right under any 
* TI patents, copyrights or trade secrets.
*
* You may not use the Program in non-TI devices.
* ********************************************************* */


#include "device.h"

//====================================================================
/**
  * Initializes FLL+ and load capacitances for the 32kHz oscillator
  *
  * \param cl       required load capacitance in pF
  * \param fsystem  required system frequency (MCLK) in kHz
  * \param fcyrstal cyrstal frequency in kHz
  */
void init_fll(int cl, int fsystem, int fcrystal)
{
  unsigned int cint, d, dco_div_bits;
  int nplus1;
  volatile int i, j;
  //  /\  Prevent variables from being "optimized".
  
  // Choose internal load capacitors Cint for LFXT1 depending on the 
  // load capacitor CL ("cl") required by the crystal

  if (cl <= 3)      // 0pF < CL =< 3pF?
    cint= XCAP0PF;  // Use Cint = 0pF
  else if (cl <= 7) // 3pF < CL =< 7pF?
    cint= XCAP10PF; // Use Cint = 10pF
  else if (cl <= 9) // 7pF < CL =< 9pF?
    cint= XCAP14PF; // Use Cint = 14pF
  else              //  CL > 9pF!
    cint= XCAP18PF; // Use Cint = 18pF and external Cs

  // Choose the system frequency divider
  if ((fsystem/fcrystal) < 121)
    d= 1; // D = 0 (fDCOCLK/1)
  else if ((fsystem/fcrystal) < 240)
    d= 2; // D = 1 (fDCOCLK/2)
  else if ((fsystem/fcrystal) < 480)
    d= 4; // D = 2 (fDCOCLK/4)
  else
    d= 8; // D = 3 (fDCOCLK/8)
    
  nplus1= (fsystem/fcrystal)/d;
  
  switch (d)
  {
    case  1: dco_div_bits= FLLD_1; break;  // fDCOCLK/1
    case  2: dco_div_bits= FLLD_2; break;  // fDCOCLK/2
    case  4: dco_div_bits= FLLD_4; break;  // fDCOCLK/4
    default: dco_div_bits= FLLD_8;         // fDCOCLK/8
  }

  if (fsystem < 3000) // fsystem < 3MHz
    SCFI0=    0 | dco_div_bits;
  else if (fsystem <  6000) // 3MHz <= fsystem <  6MHz
    SCFI0= FN_2 | dco_div_bits;
  else if (fsystem <  9000) // 6MHz <= fsystem <  9MHz
    SCFI0= FN_3 | dco_div_bits;
  else if (fsystem < 13000) // 9MHz <= fsystem < 13MHz
    SCFI0= FN_4 | dco_div_bits;
  else
    SCFI0= FN_8 | dco_div_bits;

  FLL_CTL0= DCOPLUS | cint;
  SCFQCTL = nplus1 - 1;

  // Allow FLL+ to settle at the correct DCO tap
  for (i= 32*d*nplus1; i > 0; i--)
    for (j= 8; j > 0; j--);

} // End of fll_init()
