//--------------------------------------------------------------------------
//
//  Software for MSP430 based e-meters.
//
//  THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR
//  REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY,
//  INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
//  FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR
//  COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.
//  TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET
//  POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY
//  INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR
//  YOUR USE OF THE PROGRAM.
//
//  IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
//  CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY
//  THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED
//  OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT
//  OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM.
//  EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF
//  REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS
//  OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF
//  USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S
//  AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF
//  YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS
//  (U.S.$500).
//
//  Unless otherwise stated, the Program written and copyrighted
//  by Texas Instruments is distributed as "freeware".  You may,
//  only under TI's copyright in the Program, use and modify the
//  Program without any charge or restriction.  You may
//  distribute to third parties, provided that you transfer a
//  copy of this license to the third party and the third party
//  agrees to these terms by its first use of the Program. You
//  must reproduce the copyright notice and any other legend of
//  ownership on each copy or partial copy, of the Program.
//
//  You acknowledge and agree that the Program contains
//  copyrighted material, trade secrets and other TI proprietary
//  information and is protected by copyright laws,
//  international copyright treaties, and trade secret laws, as
//  well as other intellectual property laws.  To protect TI's
//  rights in the Program, you agree not to decompile, reverse
//  engineer, disassemble or otherwise translate any object code
//  versions of the Program to a human-readable form.  You agree
//  that in no event will you alter, remove or destroy any
//  copyright notice included in the Program.  TI reserves all
//  rights not specifically granted under this license. Except
//  as specifically provided herein, nothing in this agreement
//  shall be construed as conferring by implication, estoppel,
//  or otherwise, upon you, any license or other right under any
//  TI patents, copyrights or trade secrets.
//
//  You may not use the Program in non-TI devices.
//
//  File: emeter-foreground.c
//
//  Steve Underwood <steve-underwood@ti.com>
//  Texas Instruments Hong Kong Ltd.
//
//  $Id: emeter-foreground.c,v 1.43 2009/04/27 09:05:26 a0754793 Exp $
//
/*! \file emeter-structs.h */
//
//--------------------------------------------------------------------------
//
//  MSP430 foreground (non-interrupt) routines for e-meters
//
//  This software is appropriate for single phase and three phase e-meters
//  using a voltage sensor plus a CT or shunt resistor current sensors, or
//  a combination of a CT plus a shunt.
// 
//    Foreground process includes:
//    -Using timer tick to wait
//    -Calculating the power per channel
//    -Determine if current channel needs scaling.
//    -Determine if needs to be in low power modes.
//    -Compensate reference from temperature sensor
//
#include <stdint.h>
#include <stdlib.h>
#if !defined(__MSP430__)
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#endif
#if defined(__GNUC__)
#include <signal.h>
#endif
#include <math.h>
#include <io.h>
#include <emeter-toolkit.h>
#define __MAIN_PROGRAM__

#include "emeter-structs.h"
#if defined(MESH_NET_SUPPORT)
#include "mesh_structure.h"
#endif

int32_t xxx;
int32_t yyy;

#if !defined(__IAR_SYSTEMS_ICC__)
static __inline__ long labs(long __x);

static __inline__ long labs(long __x)
{
    return (__x < 0) ? -__x : __x;
}
#endif

void set_phase_correction(struct phase_correction_s *s, int correction)
{
    correction += 128;
    s->step = I_HISTORY_STEPS + (correction >> 8);
    correction = 127 - ((correction & 0xFF) >> 1);
    s->fir_beta = fir_coeffs[correction][0];
    s->fir_gain = fir_coeffs[correction][1];
}

#if defined(DYNAMIC_PHASE_CORRECTION_SUPPORT)
static void set_phase_gain_correction(struct phase_correction_s *s, int correction, int gain)
{
    correction += 128;
    s->step = I_HISTORY_STEPS + (correction >> 8);
    correction = 127 - ((correction & 0xFF) >> 1);
    s->fir_beta = fir_coeffs[correction][0];
    s->fir_gain = Q1_15_mul(gain, fir_coeffs[correction][1]);
}
#endif

#if defined(__HAS_SD_ADC__)
void set_sd16_phase_correction(struct phase_correction_sd16_s *s, int ph, int correction)
{
    static unsigned char volatile * const sd16_locations[NUM_CURRENT_CHANNELS] =
    {
    #if defined(SINGLE_PHASE)
        &SD16PRE_LIVE,
    #else
        &SD16PRE_CURRENT_1,
        &SD16PRE_CURRENT_2,
        &SD16PRE_CURRENT_3,
    #endif
    #if defined(NEUTRAL_MONITOR_SUPPORT)
       // &SD16PRE_NEUTRAL
    #endif
    };
    uint8_t bump;

    /* Only try to nudge the converter's timing when in normal operating mode. */
    if (operating_mode == OPERATING_MODE_NORMAL)
    {
        if ((bump = s->sd16_preloaded_offset - (correction & 0xFF)))
            *sd16_locations[ph] = bump;
    }
    /* Always store the required correction. */
    s->step = I_HISTORY_STEPS - (correction >> 8);
    s->sd16_preloaded_offset = correction;
}
#endif

#if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)  &&  defined(POWER_BALANCE_DETECTION_SUPPORT)
static int32_t test_phase_balance(int32_t live_signal, int32_t neutral_signal, int threshold)
{
    int permitted_imbalance_fraction;

    /* This may be testing between two currents, or between two powers. I normal mode it
       is testing between two power readings. In limp mode it is testing between two
       current readings. */
    /* See which one is bigger, with some tolerance built in. If the signal measured
       from the neutral is more than 6.25% or 12.5% (options) different from the signal
       measured from the live there is something wrong (maybe fraudulent tampering, or
       just something faulty). In this case we use the current measured from the
       channel with the higher signal. When the channel is reasonably balanced, we use
       the signal from the live lead. If neither signal is above the threshold we use
       a more relaxed measure of imbalance (say 25% or even 50%), to allow for the lower
       accuracy of these small signals (remember, we need the test to work well under
       transient conditions, so the threshold needs to be far more relaxed than the
       basic measurement accuracy would indicate). Assessments are persistence checked
       to avoid transient conditions causing a false change of imbalance status. */
    if (live_signal <= threshold  &&  neutral_signal <= threshold)
        permitted_imbalance_fraction = RELAXED_IMBALANCE_FRACTION;
    else
        permitted_imbalance_fraction = PERMITTED_IMBALANCE_FRACTION;
    /* We have a signal strong enough for proper assessment. */
    if ((phase->status & PHASE_UNBALANCED))
    {
        /* We are looking for the restoration of balance between the
           live and neutral. */
        if ((live_signal - (live_signal >> permitted_imbalance_fraction)) < neutral_signal
            &&
            neutral_signal < (live_signal + (live_signal >> permitted_imbalance_fraction)))
        {
            /* Things might be balanced, but persistence check to be sure. */
            if (--current_unbalanced <= -PHASE_UNBALANCED_PERSISTENCE_CHECK)
            {
                /* Things look genuinely balanced. */
                phase->status &= ~(PHASE_UNBALANCED | CURRENT_FROM_NEUTRAL);
                current_unbalanced = 0;
            }
        }
        else
        {
            current_unbalanced = 0;
            /* The imbalanced might have swapped around - check. */
            /* Here we just choose the greater signal each block, as we have
                already confirmed (i.e. debounced) the imbalance condition. */
            if (neutral_signal > live_signal)
                phase->status |= CURRENT_FROM_NEUTRAL;
            else
                phase->status &= ~CURRENT_FROM_NEUTRAL;
        }
    }
    else
    {
        /* We are looking for the live and neutral becoming unbalanced. */
        if ((live_signal - (live_signal >> permitted_imbalance_fraction)) > neutral_signal
            ||
            neutral_signal > (live_signal + (live_signal >> permitted_imbalance_fraction)))
        {
            /* Things might be unbalanced, but persistence check to be sure. */
            if (++current_unbalanced >= PHASE_UNBALANCED_PERSISTENCE_CHECK)
            {
                /* Things look genuinely unbalanced. */
                current_unbalanced = 0;
                phase->status |= PHASE_UNBALANCED;
                if (neutral_signal > live_signal)
                    phase->status |= CURRENT_FROM_NEUTRAL;
                else
                    phase->status &= ~CURRENT_FROM_NEUTRAL;
            }
        }
        else
        {
            current_unbalanced = 0;
        }
    }
    /* Clear the unbalanced detection, so we don't display unbalanced. This should eliminate
       flashing of the LED if the are transient conditions causing false indications of
       imbalance. */
    if (live_signal <= threshold  &&  neutral_signal <= threshold)
        phase->status &= ~PHASE_UNBALANCED;
    if ((phase->status & CURRENT_FROM_NEUTRAL))
        return  neutral_signal;
    return  live_signal;
}
#endif

#if defined(MAINS_FREQUENCY_SUPPORT)
#if defined(SINGLE_PHASE)
int16_t frequency(void)
#else
int16_t frequency(struct phase_parms_s *phase, struct phase_nv_parms_s const *phase_nv)
#endif
{
    int32_t x;
    //int step;
    //int offset;

    /* Calculate the mains frequency in 1/100Hz increments, based on the mains
       period assessment from the background activity. */
#if defined(LIMP_MODE_SUPPORT)
    if (operating_mode == OPERATING_MODE_LIMP)
    {
        /* In limp mode there is no voltage waveform, so we get the frequency from
           the current in the active lead. This may fail to measure frequency
           correctly for very low currents, and very distorted current waveforms. */
    #if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)
        if ((phase->status & CURRENT_FROM_NEUTRAL))
            x = phase->metrology.neutral.mains_period;
        else
    #endif
            x = phase->metrology.current.mains_period;
    }
    else
    {
#endif
        /* Normally we get the mains frequency from the voltage. Voltage is always
           present, and is not subject to the same level of distortion as the current
           waveform with difficult loads. */
#if defined(REACTIVE_POWER_BY_QUADRATURE_SUPPORT)
        /* We have a whole cycle period in the upper 16 bits, but we want the delay for 90 degrees, so we shift 2
           extra bits for that. */
        x = (phase->metrology.mains_period >> 18);
        set_phase_correction(&phase->metrology.current.quadrature_correction[0], x + phase_nv->current.phase_correction[0]);
    #if GAIN_STAGES > 1
        set_phase_correction(&phase->metrology.current.quadrature_correction[1], x + phase_nv->current.phase_correction[1]);
    #endif
    #if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)
        set_phase_correction(&phase->metrology.neutral.quadrature_correction[0], x + nv_parms.seg_a.s.neutral.phase_correction[0]);
        #if GAIN_STAGES > 1
        set_phase_correction(&phase->metrology.neutral.quadrature_correction[1], x + nv_parms.seg_a.s.neutral.phase_correction[1]);
        #endif
    #endif
#endif
        x = phase->metrology.mains_period;
#if defined(LIMP_MODE_SUPPORT)
    }
#endif
    x = (int32_t) SAMPLES_PER_10_SECONDS*256L*10L/(x >> 16);
    return  x;
}
#endif

#if defined(VRMS_SUPPORT)
    #if defined(SINGLE_PHASE)
int32_t voltage(void)
    #else
int32_t voltage(struct phase_parms_s *phase, struct phase_nv_parms_s const *phase_nv)
    #endif
{
    int16_t i;
    int32_t x;

    /* Calculate the RMS voltage in 10mV increments. Return -1 for overrange
       (i.e. ADC clip). */
    if ((phase->status & V_OVERRANGE))
        return -1;
    x = div_sh48(phase->metrology.dot_prod_logged.V_sq, 26 - 2*ADC_BITS, phase->metrology.dot_prod_logged.sample_count);
    #if defined(LIMP_MODE_SUPPORT)
    if (operating_mode == OPERATING_MODE_LIMP)
        i = phase_nv->V_rms_limp_scale_factor;
    else
    #endif
        i = phase_nv->V_rms_scale_factor;
    x = isqrt32(x);
    x = (x >> 12)*i;
    x >>= 14;
    return x;
}
#endif

#if defined(IRMS_SUPPORT)
    #if defined(SINGLE_PHASE)
int32_t current(void)
    #else
int32_t current(struct phase_parms_s *phase, struct phase_nv_parms_s const *phase_nv, int ph)
    #endif
{
    #if defined(DYNAMIC_PHASE_CORRECTION_SUPPORT)
    int16_t i;
    int16_t j;
    int32_t z;
    int16_t gain;
    #endif
    int32_t x;
    #if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)
    int32_t y;
    #endif

    /* Calculate the RMS current in 1mA increments. Return -1 for overrange
       (i.e. ADC clip). A side effect of this routine is it updates the dynamic
       phase correction settings, based on the newly calculated current. */
    /* We always have to work out the properly scaled current from both leads, in
       order to work out the FIR coeffs for the next block. */
    #if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)
    if ((phase->status & I_NEUTRAL_OVERRANGE))
    {
        y = -1;
        #if defined(PRECALCULATED_PARAMETER_SUPPORT)
        phase->metrology.neutral.I_rms = y;
        #endif
        #if defined(DYNAMIC_PHASE_CORRECTION_SUPPORT)
        /* Use the last step in the phase shift table (i.e. the 55A and up step) */
        i = 6 + 50 + 20;
        j = 0;
        #endif
    }
    else
    {
        #if ADC_BITS == 16
        y = div48(phase->metrology.neutral.dot_prod_logged.I_sq[0], phase->metrology.neutral.dot_prod_logged.sample_count);
        #else
        y = div_sh48(phase->metrology.neutral.dot_prod_logged.I_sq[0], 32 - 2*ADC_BITS, phase->metrology.neutral.dot_prod_logged.sample_count);
        #endif
        y = isqrt32(y) - nv_parms.seg_a.s.neutral.ac_offset;
        #if defined(LIMP_MODE_SUPPORT)
        if (operating_mode == OPERATING_MODE_LIMP)
            y = (y >> 12)*nv_parms.seg_a.s.neutral.I_rms_limp_scale_factor[0];
        else
        #endif
            y = (y >> 12)*nv_parms.seg_a.s.neutral.I_rms_scale_factor[0];
        y >>= 14;
        #if defined(PRECALCULATED_PARAMETER_SUPPORT)
        phase->metrology.neutral.I_rms = y;
        #endif
        #if defined(DYNAMIC_PHASE_CORRECTION_SUPPORT)
        if (y < 0)
        {
            /* This should not happen, but just in case... */
            i = 0;
            j = 0;
        }
        else if (y < 5000)
        {
            /* 0.1A steps up to 5A */
            i = y/100;
            j = y%100;
        }
        else if (y < 25000)
        {
            /* 1A steps up to 25A */
            i = (y - 5000)/1000 + 50;
            j = (y%1000)/10;
        }
        else if (y < 55000)
        {
            /* 5A steps up to 60A */
            i = (y - 25000)/5000 + 50 + 20;
            j = (y%5000)/50;
        }
        else
        {
            /* Stick at the 55A position */
            i = 6 + 50 + 20;
            j = 0;
        }
        #endif
    }
        #if defined(DYNAMIC_PHASE_CORRECTION_SUPPORT)
    gain = (uint8_t) phase_corrections[i][NEUTRAL_CURRENT_CHANNEL][1];
    j = ((phase_corrections[i + 1][NEUTRAL_CURRENT_CHANNEL][0] - phase_corrections[i][NEUTRAL_CURRENT_CHANNEL][0])*j)/100;
    i = nv_parms.seg_a.s.neutral.phase_correction + phase_corrections[i][NEUTRAL_CURRENT_CHANNEL][0] + j;
    /* Adjust the phase shift to allow for the current mains frequency. This is just a linear
       approximation, but should be good enough over the frequency ranges we must cover. */
    if (frequency_phase_factor[NEUTRAL_CURRENT_CHANNEL])
    {
        /* The frequency correction is phase-shift dependant, and is proportional to the square of
           the phase-shift. */
        z = i*i;
        z *= ((int) phase->frequency - MAINS_NOMINAL_FREQUENCY*100);
        //z = z*frequency_phase_factor[NEUTRAL_CURRENT_CHANNEL];
        z >>= 16;
        /* Half bit round */
        z++;
        i -= z;
    }
    gain = 32767 - (gain << 2);
    /* Update the whole group quickly, so the interrupt routine uses a consistent pair of
       step and beta values. */
            #if defined(__MSP430_HAS_ADC12__)  ||  defined(__MSP430_HAS_ADC10__)
    set_phase_gain_correction(&phase->metrology.neutral.in_phase_correction[0], i, gain);
            #elif defined(__HAS_SD_ADC__)
                #if defined(SINGLE_PHASE)
    set_sd16_phase_correction(&phase->metrology.neutral.in_phase_correction[0], ph, i);
                #endif
            #endif
        #endif
    #endif
    if ((phase->status & I_OVERRANGE))
    {
        x = -1;
    #if defined(PRECALCULATED_PARAMETER_SUPPORT)
        phase->metrology.current.I_rms = x;
    #endif
    #if defined(DYNAMIC_PHASE_CORRECTION_SUPPORT)
        /* Use the last step in the table (i.e. the 55A and up step) */
        i = 6 + 50 + 20;
        j = 0;
    #endif
    }
    else
    {
    #if ADC_BITS == 16
        x = div48(phase->metrology.current.dot_prod_logged.I_sq[0], phase->metrology.current.dot_prod_logged.sample_count);
    #else
        x = div_sh48(phase->metrology.current.dot_prod_logged.I_sq[0], 32 - 2*ADC_BITS, phase->metrology.current.dot_prod_logged.sample_count);
    #endif
        x = isqrt32(x) - phase_nv->current.ac_offset;

    #if defined(LIMP_MODE_SUPPORT)
        if (operating_mode == OPERATING_MODE_LIMP)
            x = (x >> 12)*phase_nv->current.I_rms_limp_scale_factor[0];
        else
    #endif
            x = (x >> 12)*phase_nv->current.I_rms_scale_factor[0];
        x >>= 14;
    #if defined(PRECALCULATED_PARAMETER_SUPPORT)
        phase->metrology.current.I_rms = x;
    #endif
    #if defined(DYNAMIC_PHASE_CORRECTION_SUPPORT)
        if (x < 0)
        {
            /* This should not happen, but just in case... */
            i = 0;
            j = 0;
        }
        else if (x < 5000)
        {
            /* 0.1A steps up to 5A */
            i = x/100;
            j = x%100;
        }
        else if (x < 25000)
        {
            /* 1A steps up to 25A */
            i = (x - 5000)/1000 + 50;
            j = (x%1000)/10;
        }
        else if (x < 55000)
        {
            /* 5A steps up to 60A */
            i = (x - 25000)/5000 + 50 + 20;
            j = (x%5000)/50;
        }
        else
        {
            /* Stick at the 55A position */
            i = 6 + 50 + 20;
            j = 0;
        }
    #endif
    }
    #if defined(DYNAMIC_PHASE_CORRECTION_SUPPORT)
        #if !defined(SINGLE_PHASE)
    gain = (uint8_t) phase_corrections[i][ph][1];
    j = ((phase_corrections[i + 1][ph][0] - phase_corrections[i][ph][0])*j)/100;
    i = phase_nv->current.phase_correction + phase_corrections[i][ph][0] + j;
        #else
    gain = (uint8_t) phase_corrections[i][0][1];
    j = ((phase_corrections[i + 1][0][0] - phase_corrections[i][0][0])*j)/100;
    i = phase_nv->current.phase_correction + phase_corrections[i][0][0] + j;
        #endif
    /* Adjust the phase shift to allow for the current mains frequency. This is just a linear
       approximation, but should be good enough over the frequency ranges we must cover. */
        #if defined(SINGLE_PHASE)
    if (frequency_phase_factor[0])
        #else
    if (frequency_phase_factor[ph])
        #endif
    {
        /* The frequency correction is phase-shift dependant, and is proportional to the square of
           the phase-shift. */
        z = i*i;
        z *= ((int) phase->frequency - MAINS_NOMINAL_FREQUENCY*100);
        //z = z*frequency_phase_factor[ph];
        z >>= 16;
        /* Half bit round */
        z++;
        i -= z;
    }
    gain = 32767 - (gain << 2);
    /* Update the whole group quickly, so the interrupt routine uses a consistent pair of
       step and beta values. */
        #if defined(__MSP430_HAS_ADC12__)  ||  defined(__MSP430_HAS_ADC10__)
    set_phase_gain_correction(&phase->metrology.current.in_phase_correction[0], i, gain);
        #elif defined(__HAS_SD_ADC__)
            #if defined(SINGLE_PHASE)
    set_sd16_phase_correction(&phase->metrology.current.in_phase_correction[0], ph, i);
            #else
    set_sd16_phase_correction(&phase->metrology.current.in_phase_correction[0], ph, phase_nv->current.phase_correction[0]);
            #endif
        #endif
    #endif
    #if defined(LIMP_MODE_SUPPORT)
    if (operating_mode == OPERATING_MODE_LIMP)
    {
        /* We need to work out for ourselves which is the relevant current
           to use. */
        #if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)  &&  defined(POWER_BALANCE_DETECTION_SUPPORT)
        x = test_phase_balance(x, y, PHASE_UNBALANCED_THRESHOLD_CURRENT);
          /* In limp mode we have no way to determine if the phase is reversed,
             so just say it is not. */
        phase->status &= ~PHASE_REVERSED;
        #endif
    }
    else
    #endif
    {
        /* The power calculation has provided us which is the appropriate
           current to use. */
    #if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)
        if ((phase->status & CURRENT_FROM_NEUTRAL))
               x = y;
    #endif
    }
    return  x;
}
#endif

#if !defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)  &&  defined(IRMS_SUPPORT)
int32_t neutral_current(void)
{
    int32_t x;

    /* Calculate the RMS current in 1mA increments. Return -1 for overrange
       (i.e. ADC clip). A side effect of this routine is it updates the dynamic
       phase correction settings, based on the newly calculated current. */
    if ((neutral.status & I_OVERRANGE))
    {
        x = -1;
    }
    else
    {
    #if ADC_BITS == 16
        x = div48(neutral.metrology.I_sq_accum_logged[0], neutral.metrology.sample_count_logged);
    #else
        x = div_sh48(neutral.metrology.I_sq_accum_logged[0], 32 - 2*ADC_BITS, neutral.metrology.sample_count_logged);
    #endif
        x = isqrt32(x);
    #if defined(LIMP_MODE_SUPPORT)
        if (operating_mode == OPERATING_MODE_LIMP)
            x = (x >> 12)*nv_parms.seg_a.s.neutral.I_rms_limp_scale_factor[1];
        else
    #endif
            x = (x >> 12)*nv_parms.seg_a.s.neutral.I_rms_scale_factor[0];
        x >>= 14;
    }
    return  x;
}
#endif

#if defined(SINGLE_PHASE)
int32_t active_power(void)
#else
int32_t active_power(struct phase_parms_s *phase, struct phase_nv_parms_s const *phase_nv)
#endif
{
    int32_t x;
    int16_t i;
#if defined(NEUTRAL_MONITOR_SUPPORT)  ||  GAIN_STAGES > 1
    int32_t y;
#endif
#if defined(PHASE_REVERSED_DETECTION_SUPPORT)
    int reversed;
#endif

    /* We can only do real power assessment in full operating mode. */
    /* If we have neutral monitoring for a single phase meter, we need to measure
       both power levels, and decide between them. Issues to be assessed here are
       whether one or both leads show reverse power, and whether the power levels
       are balanced. */
#if defined(PHASE_REVERSED_DETECTION_SUPPORT)
    /* If we find a negative power level we may be genuinely feeding power to the grid,
       or we may be seeing a tamper condition. This is application dependant. */
    reversed = FALSE;
#endif

    x = div_sh48(phase->metrology.current.dot_prod_logged.P_active[0], 27 - 2*ADC_BITS, phase->metrology.current.dot_prod_logged.sample_count);
    i = phase_nv->current.P_scale_factor[0];
#if defined(PHASE_CORRECTION_SUPPORT)  &&  (defined(__MSP430_HAS_ADC12__)  ||  defined(__MSP430_HAS_ADC10__))
    i = Q1_15_mul(i, phase->metrology.current.in_phase_correction[0].fir_gain);
#endif
    x = mul48(x, i);
#if GAIN_STAGES > 1
    x >>= 2;
    y = div_sh48(phase->metrology.current.dot_prod_logged.P_active[1], 27 - 2*ADC_BITS, phase->metrology.current.dot_prod_logged.sample_count);
    i = phase_nv->current.P_scale_factor[1];
#if defined(PHASE_CORRECTION_SUPPORT)  &&  (defined(__MSP430_HAS_ADC12__)  ||  defined(__MSP430_HAS_ADC10__))
    i = Q1_15_mul(i, phase->metrology.current.in_phase_correction[1].fir_gain);
#endif
    x += mul48(y, i);
#endif
#if defined(PRECALCULATED_PARAMETER_SUPPORT)
    phase->metrology.current.active_power = x;
#endif
#if defined(PHASE_REVERSED_DETECTION_SUPPORT)
    if (x < 0)
    {
    #if defined(PHASE_REVERSED_IS_TAMPERING)
        x = -x;
    #endif
        phase->status |= I_REVERSED;
        if (x > PHASE_REVERSED_THRESHOLD_POWER)
            reversed = TRUE;
    }
    else
    {
        phase->status &= ~I_REVERSED;   
    }
#endif
#if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)
    y = div_sh48(phase->metrology.neutral.dot_prod_logged.P_active[0], 27 - 2*ADC_BITS, phase->metrology.neutral.dot_prod_logged.sample_count);
    i = nv_parms.seg_a.s.neutral.P_scale_factor[0];
    #if defined(PHASE_CORRECTION_SUPPORT)  &&  (defined(__MSP430_HAS_ADC12__)  ||  defined(__MSP430_HAS_ADC10__))
    i = Q1_15_mul(i, phase->metrology.neutral.in_phase_correction[0].fir_gain);
    #endif
    y = mul48(y, i);
    #if defined(PRECALCULATED_PARAMETER_SUPPORT)
    phase->metrology.neutral.active_power = y;
    #endif
    #if defined(PHASE_REVERSED_DETECTION_SUPPORT)
    if (y < 0)
    {
    #if defined(PHASE_REVERSED_IS_TAMPERING)
        y = -y;
    #endif
        phase->status |= I_NEUTRAL_REVERSED;
        if (y > PHASE_REVERSED_THRESHOLD_POWER)
            reversed = TRUE;
    }
    else
    {
        phase->status &= ~I_NEUTRAL_REVERSED;  
    }
    #endif
    #if defined(POWER_BALANCE_DETECTION_SUPPORT)
        #if defined(SINGLE_PHASE)
    x = test_phase_balance(x, y, PHASE_UNBALANCED_THRESHOLD_POWER);
        #else
    x = test_phase_balance(phase, x, y, PHASE_UNBALANCED_THRESHOLD_POWER);
        #endif
    if ((phase->status & PHASE_UNBALANCED))
    {
        /* When the phase is unbalanced we only look for reversed current in the 
           lead with the higher current. If we do not impose this restriction, coupling
           through a parasitic CT power supply transformer can cause the reverse condition
           to be raised incorrectly. If there is no parasitic supply this test is probably
           a waste of time. */
        if ((phase->status & CURRENT_FROM_NEUTRAL))
            reversed = phase->status & I_NEUTRAL_REVERSED;
        else
            reversed = phase->status & I_REVERSED;
    }
    #endif
#endif
#if defined(PHASE_REVERSED_DETECTION_SUPPORT)
    if ((phase->status & PHASE_REVERSED))
    {
        if (!reversed)
        {
            if (--current_reversed <= -PHASE_REVERSED_PERSISTENCE_CHECK)
            {
                phase->status &= ~PHASE_REVERSED;
                current_reversed = 0;
            }
        }
        else
        {
            current_reversed = 0;
        }
    }
    else
    {
        if (reversed)
        {
            if (++current_reversed >= PHASE_REVERSED_PERSISTENCE_CHECK)
            {
                phase->status |= PHASE_REVERSED;
                current_reversed = 0;
            }
        }
        else
        {
            current_reversed = 0;
        }
    }
#endif
    return  x;
}

#if defined(REACTIVE_POWER_SUPPORT)  &&  defined(REACTIVE_POWER_BY_QUADRATURE_SUPPORT)
    #if defined(SINGLE_PHASE)
int32_t reactive_power(void)
    #else
int32_t reactive_power(struct phase_parms_s *phase, struct phase_nv_parms_s const *phase_nv)
    #endif
{
    int32_t x;
    int16_t i;
#if GAIN_STAGES > 1
    int32_t y;
#endif

    /* We can only do real power assessment in full operating mode. */
    #if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)
    /* If we have neutral monitoring for a single phase meter, we need to use whichever
       channel has been selected by the anti-tamper validation scheme. */
    if ((phase->status & CURRENT_FROM_NEUTRAL))
    {
        x = div_sh48(phase->metrology.neutral.dot_prod_logged.P_reactive[0], 27 - 2*ADC_BITS, phase->metrology.neutral.dot_prod_logged.sample_count);
        i = nv_parms.seg_a.s.neutral.P_scale_factor[0];
        #if defined(REACTIVE_POWER_BY_QUADRATURE_SUPPORT)
        i = Q1_15_mul(i, phase->metrology.neutral.quadrature_correction[0].fir_gain);
        #endif
    }
    else
    {
    #endif
        x = div_sh48(phase->metrology.current.dot_prod_logged.P_reactive[0], 27 - 2*ADC_BITS, phase->metrology.current.dot_prod_logged.sample_count);
        i = phase_nv->current.P_scale_factor[0];
    #if defined(REACTIVE_POWER_BY_QUADRATURE_SUPPORT)
        i = Q1_15_mul(i, phase->metrology.current.quadrature_correction[0].fir_gain);
    #endif
    #if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)
    }
    #endif
    x = mul48(x, i);
    #if GAIN_STAGES > 1
    x >>= 2;
    #endif
#if GAIN_STAGES > 1
    y = div_sh48(phase->metrology.current.dot_prod_logged.P_reactive[1], 27 - 2*ADC_BITS, phase->metrology.dot_prod_logged.sample_count);
    i = phase_nv->current.P_scale_factor[1];
    #if defined(REACTIVE_POWER_BY_QUADRATURE_SUPPORT)
    i = Q1_15_mul(i, phase->metrology.current.quadrature_correction[1].fir_gain);
    #endif
    x += mul48(y, i);
#endif
#if defined(PHASE_REVERSED_DETECTION_SUPPORT)
    if (x < 0)
    {
    #if defined(PHASE_REVERSED_IS_TAMPERING)
        x = -x;
    #endif
    }
#endif
    x <<= 2;
    return  x;
}
#endif

#if defined(APPARENT_POWER_SUPPORT)
    #if defined(SINGLE_PHASE)
int32_t apparent_power(void)
    #else
int32_t apparent_power(struct phase_parms_s *phase, struct phase_nv_parms_s const *phase_nv)
    #endif
{
    #if defined(REACTIVE_POWER_BY_QUADRATURE_SUPPORT)
    int32_t p;
    int32_t x;
    int shift;

    /* Calculate apparent (VA) power in 0.01W increments */
    p = labs(phase->readings.active_power);
    x = labs(phase->readings.reactive_power);
    /* Justify for optimal accuracy */
    shift = 0;
    while ((p & 0xFFFF8000)  ||  (x & 0xFFFF8000))
    {
        shift++;
        p >>= 1;
        x >>= 1;
    }
    x = isqrt32(p*p + x*x);
    x >>= (16 - shift);
    #else
    int16_t i;
    int32_t x;
    int32_t y;

    /* Calculate VA power in 0.01W increments */
        #if defined(SD16CCTL0_)
    x = div_sh48(phase->V_sq_accum_logged, -6, phase->metrology.sample_count_logged);
        #else
    x = div_sh48(phase->V_sq_accum_logged, 2, phase->metrology.sample_count_logged);
        #endif
        #if defined(LIMP_MODE_SUPPORT)
    if (operating_mode == OPERATING_MODE_LIMP)
        i = phase_nv->V_rms_limp_scale_factor;
    else
        #endif
        i = phase_nv->V_rms_scale_factor;
    x = isqrt32(x);
    x = (x >> 12)*i;
    x >>= 14;


        #if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)
    if ((phase->status & CURRENT_FROM_NEUTRAL))
    {
            #if defined(SD16CCTL0_)
        y = div48(phase->metrology.neutral.I_sq_accum_logged[0], phase->metrology.sample_count_logged);
            #else
        y = div_sh48(phase->metrology.neutral.I_sq_accum_logged[0], 8, phase->metrology.sample_count_logged);
            #endif
        y = isqrt32(y);
            #if defined(LIMP_MODE_SUPPORT)
        if (operating_mode == OPERATING_MODE_LIMP)
            y = (y >> 12)*nv_parms.s.neutral.I_rms_limp_scale_factor[0];
        else
            #endif
            y = (y >> 12)*nv_parms.s.neutral.I_rms_scale_factor[0];
        y >>= 14;
    }
    else
    {
        #endif
        #if defined(SD16CCTL0_)
        y = div48(phase->metrology.current.dot_prod_logged.I_sq[0], phase->metrology.current.dot_prod_logged.sample_count);
        #else
        y = div_sh48(phase->metrology.current.current.dot_prod_logged.I_sq[0], 8, phase->metrology.current.dot_prod_logged.sample_count);
        #endif
        y = isqrt32(y);
        #if defined(LIMP_MODE_SUPPORT)
        if (operating_mode == OPERATING_MODE_LIMP)
            y = (y >> 12)*phase_nv->current.I_rms_limp_scale_factor[0];
        else
        #endif
            y = (y >> 12)*phase_nv->current.I_rms_scale_factor[0];
        y >>= 14;
        #if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)
    }
        #endif
    x *= y;
    x /= 1000;
    #endif
    return  x;
}
#endif

#if defined(POWER_FACTOR_SUPPORT)
    #if defined(SINGLE_PHASE)
int16_t power_factor(void)
    #else
int16_t power_factor(struct phase_parms_s *phase, struct phase_nv_parms_s const *phase_nv)
    #endif
{
    int32_t p;
    int32_t x;

    p = labs(phase->readings.active_power);
    #if defined(APPARENT_POWER_SUPPORT)
    x = labs(phase->readings.apparent_power);
    #endif
    if (p  &&  x)
    {
        /* Justify for optimal accuracy */
        while ((p & 0x40000000) == 0  &&  (x & 0x40000000) == 0)
        {
            p <<= 1;
            x <<= 1;
        }
        x >>= 16;
        p /= x;
        p *= 10000;
        p >>= 16;
        /* Don't let a little imprecision cause strange answers */
        if (p > 10000)
            p = 10000;
    }
    else
    {
        p = 0;
    }
    /* Use a negative PF to indicate an inductive load */
    #if defined(SINGLE_PHASE)  &&  defined(NEUTRAL_MONITOR_SUPPORT)
    if ((phase->status & CURRENT_FROM_NEUTRAL))
    {
        if (phase->metrology.neutral.leading < 0)
            p = -p;
    }
    else
    #endif
    {
        if (phase->metrology.current.leading < 0)
            p = -p;
    }
    return p;
}
#endif
