//--------------------------------------------------------------------------
//
//  Software for MSP430 based e-meters.
//
//  THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR
//  REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY, 
//  INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS 
//  FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR 
//  COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE. 
//  TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET 
//  POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY 
//  INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR 
//  YOUR USE OF THE PROGRAM.
//
//  IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL, 
//  CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY 
//  THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED 
//  OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT 
//  OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM. 
//  EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF 
//  REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS 
//  OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF 
//  USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S 
//  AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF 
//  YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS 
//  (U.S.$500).
//
//  Unless otherwise stated, the Program written and copyrighted 
//  by Texas Instruments is distributed as "freeware".  You may, 
//  only under TI's copyright in the Program, use and modify the 
//  Program without any charge or restriction.  You may 
//  distribute to third parties, provided that you transfer a 
//  copy of this license to the third party and the third party 
//  agrees to these terms by its first use of the Program. You 
//  must reproduce the copyright notice and any other legend of 
//  ownership on each copy or partial copy, of the Program.
//
//  You acknowledge and agree that the Program contains 
//  copyrighted material, trade secrets and other TI proprietary 
//  information and is protected by copyright laws, 
//  international copyright treaties, and trade secret laws, as 
//  well as other intellectual property laws.  To protect TI's 
//  rights in the Program, you agree not to decompile, reverse 
//  engineer, disassemble or otherwise translate any object code 
//  versions of the Program to a human-readable form.  You agree 
//  that in no event will you alter, remove or destroy any 
//  copyright notice included in the Program.  TI reserves all 
//  rights not specifically granted under this license. Except 
//  as specifically provided herein, nothing in this agreement 
//  shall be construed as conferring by implication, estoppel, 
//  or otherwise, upon you, any license or other right under any 
//  TI patents, copyrights or trade secrets.
//
//  You may not use the Program in non-TI devices.
//
//  File: basic_display.c
//
//  Steve Underwood <steve-underwood@ti.com>
//  Texas Instruments Hong Kong Ltd.
//
//  $Id: emeter-basic-display.c,v 1.20 2008/01/08 07:26:53 a0754793 Exp $
//
/*! \file emeter-structs.h */
//
//--------------------------------------------------------------------------
//
//  MSP430 simple LCD display routines for e-meters
//
#include <stdint.h>
#include <stdlib.h>
#include "io.h"
#include "emeter-toolkit.h"
#include "emeter.h"
#include "emeter-structs.h"



#include "lcd-segments.h"

/* Define hex digits and the minus sign to match the allocation of segments we are using. */
const lcd_cell_t lcd_digit_table[] =
{
    CHAR_0,
    CHAR_1,
    CHAR_2,
    CHAR_3,
    CHAR_4,
    CHAR_5,
    CHAR_6,
    CHAR_7,
    CHAR_8,
    CHAR_9,
    CHAR_A,
    CHAR_B,
    CHAR_C,
    CHAR_D,
    CHAR_E,
    CHAR_F
};

    #if defined(USE_STARBURST)
const lcd_cell_t lcd_alpha_table[] =
{
    CHAR_A,
    CHAR_B,
    CHAR_C,
    CHAR_D,
    CHAR_E,
    CHAR_F,
    CHAR_G,
    CHAR_H,
    CHAR_I,
    CHAR_J,
    CHAR_K,
    CHAR_L,
    CHAR_M,
    CHAR_N,
    CHAR_O,
    CHAR_P,
    CHAR_Q,
    CHAR_R,
    CHAR_S,
    CHAR_T,
    CHAR_U,
    CHAR_V,
    CHAR_W,
    CHAR_X,
    CHAR_Y,
    CHAR_Z
};
    #endif

    #if defined(ASCII_LCD)
static const lcd_cell_t lcd_ascii_table[] =
{
    CHAR_SPACE,
    CHAR_SPACE,         //exclamation
    CHAR_DOUBLEQUOTE,
    CHAR_SPACE,         //hash
    CHAR_DOLLAR,
    CHAR_SPACE,         //percent
    CHAR_SPACE,         //ampersand
    CHAR_QUOTE,
    CHAR_LEFTBRACKET,
    CHAR_RIGHTBRACKET,
    CHAR_ASTERISK,
    CHAR_PLUS,
    CHAR_SPACE,         //comma
    CHAR_MINUS,
    CHAR_SPACE,         //full stop
    CHAR_SLASH,
    CHAR_0,
    CHAR_1,
    CHAR_2,
    CHAR_3,
    CHAR_4,
    CHAR_5,
    CHAR_6,
    CHAR_7,
    CHAR_8,
    CHAR_9,
    CHAR_SPACE,         //colon
    CHAR_SPACE,         //semi-colon
    CHAR_LT,
    CHAR_EQUALS,
    CHAR_GT,
    CHAR_QUESTION,
    CHAR_SPACE,         //at sign
    CHAR_A,
    CHAR_B,
    CHAR_C,
    CHAR_D,
    CHAR_E,
    CHAR_F,
    CHAR_G,
    CHAR_H,
    CHAR_I,
    CHAR_J,
    CHAR_K,
    CHAR_L,
    CHAR_M,
    CHAR_N,
    CHAR_O,
    CHAR_P,
    CHAR_Q,
    CHAR_R,
    CHAR_S,
    CHAR_T,
    CHAR_U,
    CHAR_V,
    CHAR_W,
    CHAR_X,
    CHAR_Y,
    CHAR_Z,
    CHAR_LEFTBRACKET,
    CHAR_BACKSLASH,
    CHAR_RIGHTBRACKET,
    CHAR_CARET,
    CHAR_UNDERSCORE,
    CHAR_BACKQUOTE,
    CHAR_a,
    CHAR_b,
    CHAR_C,
    CHAR_d,
    CHAR_e,
    CHAR_f,
    CHAR_g,
    CHAR_h,
    CHAR_i,
    CHAR_j,
    CHAR_k,
    CHAR_l,
    CHAR_m,
    CHAR_n,
    CHAR_o,
    CHAR_p,
    CHAR_q,
    CHAR_r,
    CHAR_s,
    CHAR_t,
    CHAR_u,
    CHAR_v,
    CHAR_w,
    CHAR_x,
    CHAR_y,
    CHAR_z,
    CHAR_LEFTBRACKET,
    CHAR_VERTICALBAR,
    CHAR_RIGHTBRACKET,
    CHAR_SPACE,         //squiggle
    CHAR_SPACE          //delete
};

void lcd_text(char *s, int pos)
{
    int x;

    if (lcd_pos_step < 0)
        pos = -pos;
    if (abs(lcd_pos_step) > 1)
        pos <<= 1;
    pos = lcd_pos_base + pos;
    while (*s)
    {
        x = lcd_ascii_table[*s++ - 0x20];
        LCDMEM[pos] = x & 0xFF;
        pos += (lcd_pos_step >> 1);
        LCDMEM[pos] = x >> 8;
        pos += (lcd_pos_step >> 1);
    }
}
    #endif

const lcd_cell_t char_minus = CHAR_MINUS;

const int lcd_cells = LCD_CELLS;
const int lcd_pos_base = LCD_POS_BASE;
const int lcd_pos_step = LCD_POS_STEP;

static const lcd_cell_t lcd_high[TEXT_MESSAGE_LENGTH] =
{
    CHAR_H,
    CHAR_i,
    CHAR_g,
    CHAR_h,
    CHAR_SPACE,
    CHAR_SPACE,
};

static const lcd_cell_t lcd_startup[TEXT_MESSAGE_LENGTH] =
{
    #if defined(USE_STARBURST)
    CHAR_S,
    CHAR_T,
    CHAR_A,
    CHAR_R,
    CHAR_T,
    CHAR_SPACE,
    #else
    CHAR_S,
    CHAR_t,
    CHAR_a,
    CHAR_r,
    CHAR_t,
    CHAR_SPACE,
    #endif
};

static const lcd_cell_t lcd_no_power[TEXT_MESSAGE_LENGTH] =
{
    CHAR_b,
    CHAR_l,
    CHAR_SPACE,
    CHAR_o,
    CHAR_u,
    CHAR_t,
};

static const lcd_cell_t lcd_4v2_power[TEXT_MESSAGE_LENGTH] =
{
    CHAR_SPACE,
    CHAR_SPACE,
    CHAR_SPACE,
    CHAR_4,
    #if defined(USE_STARBURST)
    CHAR_V,
    #else
    CHAR_U,
    #endif
    CHAR_2,
};

static const lcd_cell_t lcd_normal_power[TEXT_MESSAGE_LENGTH] =
{
    CHAR_SPACE,
    CHAR_SPACE,
    CHAR_SPACE,
    CHAR_8,
    #if defined(USE_STARBURST)
    CHAR_V,
    #else
    CHAR_U,
    #endif
    CHAR_4,
};


enum
{
    DISPLAY_STAGE_PHASE_ENERGY,
    DISPLAY_STAGE_VOLTAGE,
    DISPLAY_STAGE_CURRENT,
    DISPLAY_STAGE_TOTAL_ACTIVE_POWER,
        #if defined(TEMPERATURE_SUPPORTx)
    DISPLAY_STAGE_TEMPERATURE,
        #endif
    DISPLAY_STAGE_LAST
};

static int8_t display_stage;

        #if !defined(USE_STARBURST)
#define LCDcharsx LCDchars
        #else
void LCDcharsx(const lcd_cell_t *s, int pos, int len)
{
    if (lcd_pos_step < 0)
        pos = -pos;
    if (abs(lcd_pos_step) > 1)
        pos <<= 1;
    pos = lcd_pos_base + pos;
    for (  ;  len > 0;  --len)
    {
        LCDMEM[pos] = *s & 0xFF;
        pos += (lcd_pos_step >> 1);
        LCDMEM[pos] = *s >> 8;
        s++;
        pos += (lcd_pos_step >> 1);
    }
}
        #endif

void display_clear_line_1(void)
{
    int i;
    
    for (i = FIRST_ROW_START;  i < FIRST_ROW_START + FIRST_ROW_CHARS;  i++)
        LCDchar(CHAR_SPACE, i);
}

void display_clear_line_2(void)
{
    int i;
    
    for (i = SECOND_ROW_START;  i < SECOND_ROW_START + SECOND_ROW_CHARS;  i++)
        LCDchar(CHAR_SPACE, i);
}

void display_power_fail_message(void)
{
    LCDchar(CHAR_SPACE, 1);
    LCDcharsx(lcd_no_power, FIRST_POSITION, TEXT_MESSAGE_LENGTH);
}

void display_startup_message(void)
{
    LCDchar(CHAR_SPACE, 1);
    LCDcharsx(lcd_startup, FIRST_POSITION, TEXT_MESSAGE_LENGTH);
}

void display_power_4v2_message(void)
{
    LCDchar(CHAR_SPACE, 1);
    LCDcharsx(lcd_4v2_power, FIRST_POSITION, TEXT_MESSAGE_LENGTH);
}

void display_power_normal_message(void)
{
    LCDchar(CHAR_SPACE, 1);
    LCDcharsx(lcd_normal_power, FIRST_POSITION, TEXT_MESSAGE_LENGTH);
}

static void LCDicon(int pos, int on)
{
    static const lcd_cell_t segs[] =
    {
        SEG_a,
        SEG_b,
        SEG_c,
        SEG_d,
        SEG_e,
        SEG_f,
        #if !defined(USE_STARBURST)
        SEG_g,
        SEG_h,
        #else
        SEG_i,
        SEG_h,
        SEG_12,
        SEG_1,
        SEG_3,
        SEG_5,
        SEG_6,
        SEG_7,
        SEG_9,
        SEG_11
        #endif
    };

    LCDmodify_char(segs[pos >> 5], pos & 0x1F, on);
}

static void LCDoverrange(void)
{
    LCDcharsx(lcd_high, FIRST_POSITION, TEXT_MESSAGE_LENGTH);
}

static void display_vrms(void)
{
    int32_t x;

    /* Display RMS voltage in 0.1V or 0.01V increments */
    if (phase->V_rms == 0xFFFF)
        x = -1;
    else
        x = phase->V_rms;
            #if !defined(ICON_VOLTAGE)  &&  defined(DISPLAY_TYPE_POSITION)
                #if defined(USE_STARBURST)
    LCDchar(CHAR_V, DISPLAY_TYPE_POSITION);
                #else
    LCDchar(CHAR_U, DISPLAY_TYPE_POSITION);
                #endif
            #endif
    if (x < 0)
    {
        LCDoverrange();
    }
    else
    {
            #if defined(VOLTAGE_DISPLAY_DIVISOR)
        x /= VOLTAGE_DISPLAY_DIVISOR;
            #endif
        LCDdecu32(x, FIRST_POSITION, NUMBER_WIDTH, VOLTAGE_RESOLUTION);
        LCDicon(ICON_DECIMAL_VOLTAGE, TRUE);
    }
    LCDicon(ICON_V, TRUE);
            #if defined(ICON_VOLTAGE)
    LCDicon(ICON_VOLTAGE, TRUE);
            #endif
}

static __inline__ void display_irms(void)
{
    int32_t x;

    /* Display RMS current in 1mA increments */
    if (phase->I_rms == 0xFFFF)
        x = -1;
    else
        x = phase->I_rms;
            #if !defined(ICON_CURRENT)  &&  defined(DISPLAY_TYPE_POSITION)
    LCDchar(CHAR_C, DISPLAY_TYPE_POSITION);
            #endif

    if (x < 0)
    {
        LCDoverrange();
    }
    else
    {
        x /= CURRENT_DISPLAY_DIVISOR;
        LCDdecu32(x, FIRST_POSITION, NUMBER_WIDTH, CURRENT_RESOLUTION);
        LCDicon(ICON_DECIMAL_CURRENT, TRUE);
    }
    LCDicon(ICON_A, TRUE);
            #if defined(ICON_CURRENT)
    LCDicon(ICON_CURRENT, TRUE);
            #endif
}


static __inline__ void display_total_consumed_energy(void)
{
    //Display energy in 0.1kWh increments
            #if !(defined(ICON_kW)  &&  defined(ICON_H))  &&  !defined(ICON_kWH)  &&  defined(DISPLAY_TYPE_POSITION)
    LCDchar(CHAR_E, DISPLAY_TYPE_POSITION);
            #endif
    LCDdecu32(total_consumed_energy/100L, FIRST_POSITION, NUMBER_WIDTH, ENERGY_RESOLUTION);
    LCDicon(ICON_DECIMAL_ENERGY, TRUE);
            #if defined(ICON_kWH)
    LCDicon(ICON_kWH, TRUE);
            #else
    LCDicon(ICON_kW, TRUE);
    LCDicon(ICON_H, TRUE);
            #endif
}

static __inline__ void display_phase_consumed_energy(void)
{
    //Display energy in 0.1kWh increments
            #if !(defined(ICON_kW)  &&  defined(ICON_H))  &&  !defined(ICON_kWH)  &&  defined(DISPLAY_TYPE_POSITION)
    LCDchar(CHAR_E, DISPLAY_TYPE_POSITION);
            #endif
    LCDdecu32(phase->consumed_energy/100L, FIRST_ENERGY_POSITION, NUMBER_WIDTH, ENERGY_RESOLUTION);
    LCDicon(ICON_DECIMAL_ENERGY, TRUE);
            #if defined(ICON_kWH)
    LCDicon(ICON_kWH, TRUE);
            #else
    LCDicon(ICON_kW, TRUE);
    LCDicon(ICON_H, TRUE);
            #endif
}

static __inline__ void display_total_active_power(void)
{
    /* Display total power (all phased summed) in 0.01W increments */
    LCDchar(CHAR_SPACE, 2);
    LCDdecu32(total_active_power, FIRST_ACTIVE_POWER_POSITION, NUMBER_WIDTH, POWER_RESOLUTION);
    LCDicon(ICON_DECIMAL_POWER, TRUE);
    LCDicon(ICON_kW, TRUE);
}

static __inline__ void display_phase_power(void)
{
    //Display per phase power in 0.01W increments
    LCDdecu32(phase->active_power, FIRST_ACTIVE_POWER_POSITION, NUMBER_WIDTH, POWER_RESOLUTION);
    LCDicon(ICON_DECIMAL_POWER, TRUE);
    LCDicon(ICON_kW, TRUE);
}

void update_display(void)
{
    int skip_stage;
    do
    {
        skip_stage = FALSE;
        LCDchar(CHAR_SPACE, DISPLAY_TYPE_POSITION);
        #if defined(ICON_ONLY_1)
        LCDchar(0, ICON_ONLY_1);
        #endif
        #if defined(ICON_ONLY_2)
        LCDchar(0, ICON_ONLY_2);
        #endif
        #if defined(ICON_ONLY_3)
        LCDchar(0, ICON_ONLY_3);
        #endif
        #if defined(PHASE_POSITION)
        LCDchar(CHAR_SPACE, PHASE_POSITION);
        #endif
        display_clear_line_1();
        display_clear_line_2();
        switch (display_stage)
        {
        case DISPLAY_STAGE_PHASE_ENERGY:
            display_phase_consumed_energy();
            display_phase_power();
            break;
        case DISPLAY_STAGE_CURRENT:
            display_irms();
            break;
        case DISPLAY_STAGE_VOLTAGE:
            display_vrms();
            break;
    #if defined(TEMPERATURE_SUPPORTx)
        case DISPLAY_STAGE_TEMPERATURE:
            display_temperature();
            break;
    #endif
        case DISPLAY_STAGE_TOTAL_ACTIVE_POWER:
            display_total_active_power();
            display_total_consumed_energy();
            break;
        }


        if (++display_stage >= DISPLAY_STAGE_LAST) 
            display_stage = 0;
    }
    while (skip_stage);
    #if defined(DEDICATED_TIME_FIELD)
    display_current_time();
    #endif
    #if defined(ICON_BATTERY)
    LCDicon(ICON_BATTERY, TRUE);
    #endif
    if (operating_mode == OPERATING_MODE_LIMP)
        LCDchar(CHAR_L, DISPLAY_TYPE_POSITION);
}


