//******************************************************************************
//   MSP430xG46x Demo - IHD Receive
//
//   Description: This code displays the Active Power Reading sent from an SMB.
//    For this code to work, the IHD's CC2530 must be programmed with the receiver code.
//    The SMB has a Zigbee module that transmits the Active Power Reading to the
//    IHD's Zigbee module.  The IHD's Zigbee Module receives this information 
//    and writes a message to the IHD's USCIA0 UART.  This code translates the 
//    message sent on the UART by the IHD's Zigbee module into the Active Power
//    Reading and displays the Active Power Reading.  By pressing the S2 
//    pushbutton, the Active Power can be sent out the RS232 to a computer for 
//    an additional display option at a baud rate of 9600 bps. 
//    By pressing the S1 pushbutton, the IHD can enter a low power "SLEEP" mode.
//    Baud rate is 9600bps
//    ACLK = LFXT1 = 32768Hz, MCLK = SMCLK = default DCO = 32 x ACLK = 8 MHz
//   //* An external watch crystal between XIN & XOUT is required for ACLK *//	
//
#include  <msp430xG46x.h>

// LCD segment definitions.
#define h 0x80
#define e 0x40
#define g 0x20
#define f 0x10
#define d 0x08
#define c 0x04
#define b 0x02
#define a 0x01

const char char_gen[] = {                   // definitions for digits
  a+b+c+d+e+f,                              // Displays "0"
  b+c,                                      // Displays "1"
  a+b+d+e+g,                                // Displays "2"
  a+b+c+d+g,                                // Displays "3"
  b+c+f+g,                                  // Displays "4"
  a+c+d+f+g,                                // Displays "5"
  a+c+d+e+f+g,                              // Displays "6"
  a+b+c,                                    // Displays "7"
  a+b+c+d+e+f+g,                            // Displays "8"
  a+b+c+d+f+g                               // Displays "9"
};

int Packbyte=0;                             //The current byte number in the message sent 
                                            //from the Zigbee Transmitter
int length=0;                               //The length of the current message received 
int TXon=0;                                 //Integer that corresponds to whether information 
                                            //is sent out the RS-232
unsigned long  value=0;                     //Holds the Active Power Reading sent by the SMB 
unsigned long Value_Array[3];               //Array used to store the bytes corresponding
                                            //to the Active Power Reading in a received message  
unsigned char temp;                         //temporary variable;
unsigned int thou,hun,ten,unit,ten_thou;    //Active Power Digits that are displayed on the LCD
int init=0;                               
volatile unsigned int i; 
void main(void)
{
 

  WDTCTL = WDTPW+WDTHOLD;                   // Stop WDT
  FLL_CTL0 |= XCAP14PF;                     // Configure load caps
  P1IE |= BIT0 + BIT1;                      // P1.0 and P1.1 Interrupt enabled
  P1IES |= BIT0 + BIT1;                     // P1.0 and P1.1 hi/low edge
  P1IFG &= ~BIT0 + BIT1;                    // P1.0 and P1.1 IFG Cleared
   
                                            // Configure unused Ports to reduce
                                            // power consumption
  P1DIR |= BIT6+BIT7;                                    
  P1OUT &= ~(BIT6+ BIT7);                       
  P2DIR |= BIT6+BIT7;                            
  P2OUT &= ~(BIT6+ BIT7);                      
  P3DIR |= BIT0+ BIT4 + BIT5 + BIT6 + BIT7;      
  P3OUT &= ~(BIT0+ BIT4 + BIT5 + BIT6 + BIT7);  
  P4DIR |= BIT3 + BIT4 + BIT6 + BIT7 + BIT5;             
  P4OUT &= ~ (BIT3 + BIT4 + BIT6 + BIT7 + BIT5);          
  P5DIR |= BIT0 + BIT5 + BIT6 + BIT7;             
  P5OUT &= ~ (BIT0 + BIT5 + BIT6 + BIT7);        
  P6DIR |= BIT0 + BIT1 + BIT2 ;                         
  P6OUT &= ~(BIT0 + BIT1 + BIT2+BIT4+BIT5+BIT3+BIT6+BIT7) ;            
  P7DIR |=  0xFF;
  P7OUT =0; 
  P10DIR |= BIT6+BIT7;                           
  P10OUT &= ~(BIT6+ BIT7);                      
 
  do
  {
  IFG1 &= ~OFIFG;                           // Clear OSCFault flag
  for (i = 0x47FF; i > 0; i--);             // Time for flag to set
  }
  while ((IFG1 & OFIFG));                   // OSCFault flag still set?
   for( i = 0; i < 20; i ++)
  {
    LCDMEM[i] = 0;                          // Clear LCD
  }
  
 
  P5SEL =BIT2+BIT3+BIT4;                                        
  P2SEL |= BIT4 + BIT5;                     // USCI_A0 RXD/TXD
  UCA0CTL1 |= UCSSEL_2;                     
  UCA0BR0 = 0x09;                           // 1MHz 115200
  UCA0BR1 = 0x00;                           // 1MHz 115200
  UCA0MCTL = 0x02;                          // Modulation
  UCA0CTL1 &= ~UCSWRST;                     // Initialize USCI state machine
  
  ME2 |= UTXE1 + URXE1;                     // Enable USART1 TXD/RXD
  U1CTL |= CHAR;                            // 8-bit character
  U1TCTL |= SSEL0;                          // UCLK= ACLK
 
   U1BR0 = 0x03;                             // 9600
   U1BR1 = 0x00;                             // 1MHz 115200
   U1MCTL = 0x4A;                            // 1MHz 115200 modulation
 
   U1CTL &= ~SWRST;                          // Initialize USART state machine
 
  
   LCDAPCTL0 = 0x7E;                         // Enable Segments 4-24
                                            //(segment 0-3 of MCU are not connected)
   LCDACTL = LCDON+LCD4MUX;                  //4mux LCD, ACLK/32
                                            //Display IHD430 at Startup
   LCDMEM[2]=a+ f + b + c + e + d;
   LCDMEM[3]=a + b + g + c + d;
   LCDMEM[4]= b + f + g + c;
   LCDMEM[5]= b + g + e + c + d;
   LCDMEM[6]= f + g + e + c;
   LCDMEM[7]= f + e;
   IE2 |= UCA0RXIE;                            //Enable USCI_A0 RX interrupt;
  _BIS_SR(LPM3_bits + GIE);                 // Enter LPM3, interrupts enabled
                        
}


// Port 1 interrupt service routine
#pragma vector=PORT1_VECTOR
__interrupt void Port1_ISR (void)
{
  static unsigned int i;
  if(P1IFG & 0x01)                           //When the toggle Sleep state button is pressed
  {                          
    P1IE &= ~0x01;                           //Used to debounce switch
    UCA0CTL1 ^= UCSWRST;                     //Initialize USCI state machine
    IE2 ^= UCA0RXIE;                         //If in sleep mode, do not receive.  
                                             //If coming out of sleep mode, receive.
    
    P2SEL ^= BIT4 + BIT5;                    //Disable the USCIA0 to save power
                                             // Initialize USCI state machine
    ME2 ^= UTXE1 + URXE1;                    // Disable/Enable USART1 TXD/RXD
    U1CTL ^= SWRST;                          // Initialize USART state machine
   
    for( i = 0; i < 20; i ++)                //Clear LCD
    {
      LCDMEM[i] = 0;                          
     }
    if(!(IE2&UCA0RXIE))
    {                                        //If in sleep mode, display "SLEEP"
      LCDMEM[2]=a+b+f+g+e;
      LCDMEM[3]=a+f+g+e+d;
      LCDMEM[4]=a+f+g+e+d;
      LCDMEM[5]=f+e+d;
      LCDMEM[6]=a+f+g+c+d;
      if(TXon==1)
      {                                     //If in sleep mode and transmitting using the RS-232, stop transmitting
                                            //through the RS-232 USART
        P4SEL&=~3;
      }
    }
    else                                   //If coming out of sleep mode, turn transmission through the RS-232 back on 
    {                                      //if it was on before sleeping and display IHD430.
        LCDMEM[2]=a+ f + b + c + e + d;
        LCDMEM[3]=a + b + g + c + d;
        LCDMEM[4]= b + f + g + c;
        LCDMEM[5]= b + g + e + c + d;
        LCDMEM[6]= f + g + e + c;
        LCDMEM[7]= f + e;
      
      if(TXon==1)
      {
        P4SEL|=3;
      }
    }
   
    P1IFG &= ~0x01;                        // P1.0 IFG Cleared
    P1IE |= 0x01;
    
  }
  else if(P1IFG & 0x02 )                 //If the button to activate/deactivate transmission 
  {                                      //to the RS-232 is pressed
    if (IE2&UCA0RXIE)                    //Only toggle when not in the sleep state
    {
      P1IE &= ~0x02;
      TXon^=1;                           //TXon=1 transmit; TXon=0 do not transmit
      P4SEL^=3;                          //Disable the UART when not transmitting; 
                                         //enable when transmitting 
      LCDMEM[12]^=2;                     //Toggle between whether to display the 
                                         //"Tx" character or not 
    }
    P1IFG &= ~0x02;                      // P1.1 IFG Cleared
    P1IE |= 0x02;
  }                        
}

#pragma vector=USCIAB0RX_VECTOR
__interrupt void USCIA0RX_ISR (void)
{
 
  temp=UCA0RXBUF;
  if(Packbyte==0)
  {                                     //The first byte received
    if(temp==0xFE) Packbyte=1;          //Prevent Garbage characters from being  
                                        //mistaken as the start of a message
  }
  else if(Packbyte==1) 
  {                                     //Set the length equal to the second byte 
    length=temp;
    Packbyte++;
  }
  else if(Packbyte==2 || Packbyte==3)  //Corresponds to the command bytes
  {                                    //which are not needed in the value calculation
    Packbyte++;
  }
  else if(Packbyte==length+4)
  {                                   //The last byte received        
      Packbyte=0;                     //Reset for the next message
      for(int i=0; i<length; i++)
      {                               //Calculate the value that is represented by the message
       value+=Value_Array[i]<<(i*8); 
      }
        ten_thou=0;
        thou=0;
        hun=0;
        ten=0;
        unit=0; 
      while(value>=10000)
      {
        ten_thou++;
        value-=10000;
      }
      if(ten_thou<10) LCDMEM[7]=0;
      while(value>=1000){
        thou++;
        value-=1000;
      }
      while(value>=100){
        hun++;
        value-=100;
      }
      while(value>=10){
        ten++;
        value-=10;
      }
      while(value>=1){
        unit++;
        value-=1;
      }                               //Display the number's digits on the LCD
      if(TXon==1)                     //If transmit option is on, also transmit through the UART
      {                         
       if(ten_thou<10){
        LCDMEM[6] = char_gen[ten_thou]; 
        TXBUF1= ten_thou+'0';         
        while (!(TXEPT & U1TCTL));    // USART1 TX buffer ready?
        
        LCDMEM[5] = char_gen[thou];
       TXBUF1= thou+'0';
        while (!(TXEPT & U1TCTL));   // USART1 TX buffer ready?
        
        LCDMEM[4] = char_gen[hun]; 
        TXBUF1= hun+'0';
        while (!(TXEPT & U1TCTL));   // USART1 TX buffer ready?
        
        LCDMEM[3] = char_gen[ten]|h;
        TXBUF1= '.';
        while (!(TXEPT & U1TCTL));   // USART1 TX buffer ready?
         TXBUF1= ten+ '0';
        while (!(TXEPT & U1TCTL));   // USART1 TX buffer ready?
       
        LCDMEM[2] = char_gen[unit];
        TXBUF1= unit+'0';
        while (!(TXEPT & U1TCTL));   // USART1 TX buffer ready?
        
        LCDMEM[11] |= 0x10;
        TXBUF1= ' ';
        while (!(TXEPT & U1TCTL));   // USART1 TX buffer ready?
        
        LCDMEM[11] ^= 0xE0;
        TXBUF1= 'W';
        while (!(TXEPT & U1TCTL));   // USART1 TX buffer ready?
        
        LCDMEM[12]= 0x24;
        LCDMEM[12]|=(TXon<<1);
        
        while (!(TXEPT & U1TCTL));   // USART1 TX buffer ready?
        TXBUF1= 0x0A;                //Send New Line Feed through RS-232
        
        while (!(TXEPT & U1TCTL));    //USART1 TX buffer ready?
        TXBUF1= 0x0D;                 //Send Carriage Return through RS-232
        while (!(TXEPT & U1TCTL));    
       }
     }
      else                            //If transmit option is off, 
                                      //just display the data on the LCD
      {
         if(ten_thou<10)
         {
            LCDMEM[6] = char_gen[ten_thou]; 
            LCDMEM[5] = char_gen[thou];
            LCDMEM[4] = char_gen[hun]; 
            LCDMEM[3] = char_gen[ten]|h;
            LCDMEM[2] = char_gen[unit];
            LCDMEM[11] |= 0x10;
            LCDMEM[11] ^= 0xE0;
            LCDMEM[12]= 0x24;
            LCDMEM[12]|=(TXon<<1);
         } 
      }
   }
  else                               //Data Byte received
  {
    Value_Array[Packbyte-4]=temp;   //Store each received character to later 
    Packbyte++;                     //calculate the sent Active Power Reading  
  }
}
