/*********************************************************************
* Filename: example_rtm.c                                            *
*                                                                    *
* Author: David M. Alter, Texas Instruments Inc.                     *
*                                                                    *
* Last Modified: 06/10/02                                            *
*                                                                    *
* Description: This program illustrates basic initialization and     *
* operation of the LF2407 DSP.  The following peripherals are        *
* exercised:                                                         *
*  1) Timer 2 is configured to generate a 250ms period interrupt.    *
*  2) The quad LED bank on the LF2407 EVM is sequenced in the        *
*     Timer2 ISR.                                                    *
*  3) The IOPC0 pin is toggled in the Timer2 ISR.                    *
*  4) Timer 1 is configured to drive 20KHz 25% duty cycle symmetric  *
*     PWM on the PWM1 pin.                                           *
* Additionally, the Code Composer realtime monitor has been          *
* incorporated into the program.                                     *
*                                                                    *
*                                                                    *
*********************************************************************/

/*********************************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR        *
* REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY,             *
* INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS       *
* FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR             *
* COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.         *
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET         *
* POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY                *
* INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR         *
* YOUR USE OF THE PROGRAM.                                           *
*                                                                    *
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL,        *
* CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY          *
* THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED         *
* OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT         *
* OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM.        *
* EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF          *
* REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS        *
* OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF          *
* USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S             *
* AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF         *
* YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS                *
* (U.S.$500).                                                        *
*                                                                    *
* Unless otherwise stated, the Program written and copyrighted       *
* by Texas Instruments is distributed as "freeware".  You may,       *
* only under TI's copyright in the Program, use and modify the       *
* Program without any charge or restriction.  You may                *
* distribute to third parties, provided that you transfer a          *
* copy of this license to the third party and the third party        *
* agrees to these terms by its first use of the Program. You         *
* must reproduce the copyright notice and any other legend of        *
* ownership on each copy or partial copy, of the Program.            *
*                                                                    *
* You acknowledge and agree that the Program contains                *
* copyrighted material, trade secrets and other TI proprietary       *
* information and is protected by copyright laws,                    *
* international copyright treaties, and trade secret laws, as        *
* well as other intellectual property laws.  To protect TI's         *
* rights in the Program, you agree not to decompile, reverse         *
* engineer, disassemble or otherwise translate any object code       *
* versions of the Program to a human-readable form.  You agree       *
* that in no event will you alter, remove or destroy any             *
* copyright notice included in the Program.  TI reserves all         *
* rights not specifically granted under this license. Except         *
* as specifically provided herein, nothing in this agreement         *
* shall be construed as conferring by implication, estoppel,         *
* or otherwise, upon you, any license or other right under any       *
* TI patents, copyrights or trade secrets.                           *
*                                                                    *
* You may not use the Program in non-TI devices.                     *
*********************************************************************/

/*** Build options ***/
#define		REALTIME	1				/* 1=use realtime monitor, 0=don't use */


/*** Address Definitions ***/
#include        "f2407_c.h"

#define         DAC0    port0000        /* EVM DAC register 0 (I/O space) */
ioport unsigned port0000;               /* '24xx compiler specific keyword */

#define         DAC1    port0001        /* EVM DAC register 1 (I/O space) */
ioport unsigned port0001;               /* '24xx compiler specific keyword */

#define         DAC2    port0002        /* EVM DAC register 2 (I/O space) */
ioport unsigned port0002;               /* '24xx compiler specific keyword */

#define         DAC3    port0003        /* EVM DAC register 3 (I/O space) */
ioport unsigned port0003;               /* '24xx compiler specific keyword */

#define         DACUD   port0004        /* EVM DAC update register (I/O space) */
ioport unsigned port0004;               /* '24xx compiler specific keyword */

#define         DIPSWCH port0008        /* EVM DIP switch (I/O space) */
ioport unsigned port0008;               /* '24xx compiler specific keyword */

#define         LED     port000C        /* EVM LED bank (I/O space) */
ioport unsigned port000C;               /* '24xx compiler specific keyword */


/*** Constant Definitions ***/
#define         timer2_per    58594     /* 250ms timer2 period with a 1/128
                                           timer prescaler and 30MHz CPUCLK */

#define         pwm_half_per  750       /* period/2, 20KHz symmetric PWM with
                                           a 30MHz CPUCLK */
                                           
#define         pwm_duty      563       /* 25% PWM duty cycle */

                                   
/*** Global Variable Definitions ***/
unsigned int    LED_index;              /* LED_index */
unsigned int    count;                  /* counter */

/****************************** MAIN ROUTINE ***************************/
void main(void)
{

/*** Configure the System Control and Status registers ***/
    *SCSR1 = 0x00FD;
/*
 bit 15        0:      reserved
 bit 14        0:      CLKOUT = CPUCLK
 bit 13-12     00:     IDLE1 selected for low-power mode
 bit 11-9      000:    PLL x4 mode
 bit 8         0:      reserved
 bit 7         1:      1 = enable ADC module clock
 bit 6         1:      1 = enable SCI module clock
 bit 5         1:      1 = enable SPI module clock
 bit 4         1:      1 = enable CAN module clock
 bit 3         1:      1 = enable EVB module clock
 bit 2         1:      1 = enable EVA module clock
 bit 1         0:      reserved
 bit 0         1:      clear the ILLADR bit
*/

    *SCSR2 = (*SCSR2 | 0x000B) & 0x000F;
/*
 bit 15-6      0's:    reserved
 bit 5         0:      do NOT clear the WD OVERRIDE bit
 bit 4         0:      XMIF_HI-Z, 0=normal mode, 1=Hi-Z'd
 bit 3         1:      disable the boot ROM, enable the FLASH
 bit 2     no change   MP/MC* bit reflects state of MP/MC* pin
 bit 1-0      11:      11 = SARAM mapped to prog and data
*/


/*** Disable the watchdog timer ***/
    *WDCR  = 0x00E8;
/*
 bits 15-8     0's:     reserved
 bit 7         1:       clear WD flag
 bit 6         1:       disable the dog
 bit 5-3       101:     must be written as 101
 bit 2-0       000:     WDCLK divider = 1
*/


/*** Setup external memory interface for LF2407 EVM ***/
    WSGR = 0x0040;
/*
 bit 15-11     0's:    reserved
 bit 10-9      00:     bus visibility off
 bit 8-6       001:    1 wait-state for I/O space
 bit 5-3       000:    0 wait-state for data space
 bit 2-0       000:    0 wait state for program space
*/


/*** Setup shared I/O pins ***/
    *MCRA = 0x0040;                     /* group A pins */
/*
 bit 15        0:      0=IOPB7,     1=TCLKINA
 bit 14        0:      0=IOPB6,     1=TDIRA
 bit 13        0:      0=IOPB5,     1=T2PWM/T2CMP
 bit 12        0:      0=IOPB4,     1=T1PWM/T1CMP
 bit 11        0:      0=IOPB3,     1=PWM6
 bit 10        0:      0=IOPB2,     1=PWM5
 bit 9         0:      0=IOPB1,     1=PWM4
 bit 8         0:      0=IOPB0,     1=PWM3
 bit 7         0:      0=IOPA7,     1=PWM2
 bit 6         1:      0=IOPA6,     1=PWM1
 bit 5         0:      0=IOPA5,     1=CAP3
 bit 4         0:      0=IOPA4,     1=CAP2/QEP2
 bit 3         0:      0=IOPA3,     1=CAP1/QEP1
 bit 2         0:      0=IOPA2,     1=XINT1
 bit 1         0:      0=IOPA1,     1=SCIRXD
 bit 0         0:      0=IOPA0,     1=SCITXD
*/

    *MCRB = 0xFE00;                     /* group B pins */
/*
 bit 15        1:      0=reserved,  1=TMS2 (always write as 1)
 bit 14        1:      0=reserved,  1=TMS  (always write as 1)
 bit 13        1:      0=reserved,  1=TD0  (always write as 1)
 bit 12        1:      0=reserved,  1=TDI  (always write as 1)
 bit 11        1:      0=reserved,  1=TCK  (always write as 1)
 bit 10        1:      0=reserved,  1=EMU1 (always write as 1)
 bit 9         1:      0=reserved,  1=EMU0 (always write as 1)
 bit 8         0:      0=IOPD0,     1=XINT2/ADCSOC
 bit 7         0:      0=IOPC7,     1=CANRX
 bit 6         0:      0=IOPC6,     1=CANTX
 bit 5         0:      0=IOPC5,     1=SPISTE
 bit 4         0:      0=IOPC4,     1=SPICLK
 bit 3         0:      0=IOPC3,     1=SPISOMI
 bit 2         0:      0=IOPC2,     1=SPISIMO
 bit 1         0:      0=IOPC1,     1=BIO*
 bit 0         0:      0=IOPC0,     1=W/R*
*/

    *MCRC = 0x0000;                     /* group C pins */
/*
 bit 15        0:      reserved
 bit 14        0:      0=IOPF6,     1=IOPF6
 bit 13        0:      0=IOPF5,     1=TCLKINB
 bit 12        0:      0=IOPF4,     1=TDIRB
 bit 11        0:      0=IOPF3,     1=T4PWM/T4CMP
 bit 10        0:      0=IOPF2,     1=T3PWM/T3CMP
 bit 9         0:      0=IOPF1,     1=CAP6
 bit 8         0:      0=IOPF0,     1=CAP5/QEP4
 bit 7         0:      0=IOPE7,     1=CAP4/QEP3
 bit 6         0:      0=IOPE6,     1=PWM12
 bit 5         0:      0=IOPE5,     1=PWM11
 bit 4         0:      0=IOPE4,     1=PWM10
 bit 3         0:      0=IOPE3,     1=PWM9
 bit 2         0:      0=IOPE2,     1=PWM8
 bit 1         0:      0=IOPE1,     1=PWM7
 bit 0         0:      0=IOPE0,     1=CLKOUT    
*/


/*** Configure IOPC0 pin as an output ***/
    *PCDATDIR = *PCDATDIR | 0x0100;


/*** Setup timers 1 and 2, and the PWM configuration ***/
    *T1CON = 0x0000;                    /* disable timer 1 */
    *T2CON = 0x0000;                    /* disable timer 2 */

    *GPTCONA = 0x0000;                  /* configure GPTCONA */
/*     
 bit 15        0:      reserved
 bit 14        0:      T2STAT, read-only
 bit 13        0:      T1STAT, read-only
 bit 12-11     00:     reserved
 bit 10-9      00:     T2TOADC, 00 = no timer2 event starts ADC
 bit 8-7       00:     T1TOADC, 00 = no timer1 event starts ADC
 bit 6         0:      TCOMPOE, 0 = Hi-z all timer compare outputs
 bit 5-4       00:     reserved
 bit 3-2       00:     T2PIN, 00 = forced low
 bit 1-0       00:     T1PIN, 00 = forced low
*/


/* Timer 1: configure to clock the PWM on PWM1 pin */
/* Symmetric PWM, 20KHz carrier frequency, 25% duty cycle */
    *T1CNT = 0x0000;                    /* clear timer counter */
    *T1PR = pwm_half_per;               /* set timer period */
    *DBTCONA = 0x0000;                  /* deadband units off */
    *CMPR1 = pwm_duty;                  /* set PWM1 duty cycle */

    *ACTRA = 0x0002;                    /* PWM1 pin set active high */     
/*
 bit 15        0:      space vector dir is CCW (don't care)
 bit 14-12     000:    basic space vector is 000 (dont' care)
 bit 11-10     00:     PWM6/IOPB3 pin forced low
 bit 9-8       00:     PWM5/IOPB2 pin forced low
 bit 7-6       00:     PWM4/IOPB1 pin forced low
 bit 5-4       00:     PWM3/IOPB0 pin forced low
 bit 3-2       00:     PWM2/IOPA7 pin forced low
 bit 1-0       10:     PWM1/IOPA6 pin active high
*/

     *COMCONA = 0x8200;                 /* configure COMCON register */
/*
 bit 15        1:      1 = enable compare operation
 bit 14-13     00:     00 = reload CMPRx regs on timer 1 underflow
 bit 12        0:      0 = space vector disabled
 bit 11-10     00:     00 = reload ACTR on timer 1 underflow
 bit 9         1:      1 = enable PWM pins
 bit 8-0       0's:    reserved
*/


     *T1CON = 0x0840;                   /* configure T1CON register */
/*     
 bit 15-14     00:     stop immediately on emulator suspend
 bit 13        0:      reserved
 bit 12-11     01:     01 = continous-up/down count mode
 bit 10-8      000:    000 = x/1 prescaler
 bit 7         0:      reserved in T1CON
 bit 6         1:      TENABLE, 1 = enable timer
 bit 5-4       00:     00 = CPUCLK is clock source
 bit 3-2       00:     00 = reload compare reg on underflow
 bit 1         0:      0 = disable timer compare
 bit 0         0:      reserved in T1CON
*/


/* Timer 2: configure to generate a 250ms periodic interrupt */
    *T2CNT = 0x0000;                    /* clear timer counter */
    *T2PR = timer2_per;                 /* set timer period */

    *T2CON = 0xD740;                    /* configure T2CON register */
/*     
 bit 15-14     11:     stop immediately on emulator suspend
 bit 13        0:      reserved
 bit 12-11     10:     10 = continous-up count mode
 bit 10-8      111:    111 = x/128 prescaler
 bit 7         0:      T2SWT1, 0 = use own TENABLE bit
 bit 6         1:      TENABLE, 1 = enable timer
 bit 5-4       00:     00 = CPUCLK is clock source
 bit 3-2       00:     00 = reload compare reg on underflow
 bit 1         0:      0 = disable timer compare
 bit 0         0:      SELT1PR, 0 = use own period register
*/

/*** Other setup ***/
    LED_index = 0x0001;                 /* initialize the LED index */
    count = 0;                          /* initialize counter */

/*** Setup the core interrupts ***/
    *IMR = 0x0000;                      /* clear the IMR register */
    *IFR = 0x007F;                      /* clear any pending core interrupts */

	#if REALTIME != 0                   /* RTM is in use */
    	*IMR = 0x0044;                  /* enable desired core interrupt + RTM interrupt */
	#else								/* RTM not in use */
        *IMR = 0x0004;                  /* enable desired core interrupts */
    #endif	


/*** Setup the event manager interrupts ***/
    *EVAIFRA = 0xFFFF;                  /* clear all EVA group A interrupts */
    *EVAIFRB = 0xFFFF;                  /* clear all EVA group B interrupts */
    *EVAIFRC = 0xFFFF;                  /* clear all EVA group C interrupts */
    *EVAIMRA = 0x0000;                  /* enable desired EVA group A interrupts */
    *EVAIMRB = 0x0001;                  /* enable desired EVA group B interrupts */
    *EVAIMRC = 0x0000;                  /* enable desired EVA group C interrupts */

    *EVBIFRA = 0xFFFF;                  /* clear all EVB group A interrupts */
    *EVBIFRB = 0xFFFF;                  /* clear all EVB group B interrupts */
    *EVBIFRC = 0xFFFF;                  /* clear all EVB group C interrupts */
    *EVBIMRA = 0x0000;                  /* enable desired EVB group A interrupts */
    *EVBIMRB = 0x0000;                  /* enable desired EVB group B interrupts */
    *EVBIMRC = 0x0000;                  /* enable desired EVB group C interrupts */

/*** Initialize the realtime monitor ***/
    #if REALTIME != 0
		asm(" .ref MON_RT_CNFG");
		asm(" CALL MON_RT_CNFG");
	#endif

/*** Enable global interrupts ***/
    asm(" CLRC INTM");                  /* enable global interrupts */

/*** Proceed with main routine ***/
    while(1)
    {
    asm(" NOP");
    asm(" NOP");                           /* endless loop, wait for interrupt */
    }

}                                       /* end of main() */


/********************** INTERRUPT SERVICE ROUTINES *********************/
interrupt void timer2_isr(void)
{

     *EVAIFRB = *EVAIFRB & 0x0001;      /* clear T2PINT flag */

/*** Sequence the LED bank on the LF2407 EVM ***/
     LED = LED_index;                   /* light the LEDs */
     LED_index = LED_index << 1;        /* left shift LED index */
     if(LED_index == 0x0010) LED_index = 0x0001;    /* reset LED index */

/*** Increment the counter ***/
     count++;                           /* increment counter */
     if(count == 10) count = 0;         /* force rollover */     

/*** Toggle the IOPC0 pin ***/
     *PCDATDIR = *PCDATDIR ^ 0x0001;    /* XOR the IOPC0 bit to toggle the pin */

}
