**********************************************************************
* Filename: example.asm                                              *
*                                                                    *
* Author: David M. Alter, Texas Instruments Inc.                     *
*                                                                    *
* Last Modified: 03/14/01                                            *
*                                                                    *
* Description: This program illustrates basic initialization and     *
* operation of the LF2407 DSP.  The following peripherals are        *
* exercised:                                                         *
*  1) Timer 2 is configured to generate a 250ms period interrupt.    *
*  2) The LED bank on the LF2407 EVM is sequenced in the Timer2 ISR. *
*  3) The IOPC0 pin is toggled in the Timer2 ISR.                    *
*  4) Timer 1 is configured to drive 20KHz 25% duty cycle symmetric  *
*     PWM on the PWM1 pin.                                           *
*                                                                    *
**********************************************************************

**********************************************************************
* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR        *
* REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY,             *
* INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS       *
* FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR             *
* COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.         *
* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET         *
* POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY                *
* INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR         *
* YOUR USE OF THE PROGRAM.                                           *
*                                                                    *
* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL,        *
* CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY          *
* THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED         *
* OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT         *
* OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM.        *
* EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF          *
* REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS        *
* OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF          *
* USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S             *
* AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF         *
* YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS                *
* (U.S.$500).                                                        *
*                                                                    *
* Unless otherwise stated, the Program written and copyrighted       *
* by Texas Instruments is distributed as "freeware".  You may,       *
* only under TI's copyright in the Program, use and modify the       *
* Program without any charge or restriction.  You may                *
* distribute to third parties, provided that you transfer a          *
* copy of this license to the third party and the third party        *
* agrees to these terms by its first use of the Program. You         *
* must reproduce the copyright notice and any other legend of        *
* ownership on each copy or partial copy, of the Program.            *
*                                                                    *
* You acknowledge and agree that the Program contains                *
* copyrighted material, trade secrets and other TI proprietary       *
* information and is protected by copyright laws,                    *
* international copyright treaties, and trade secret laws, as        *
* well as other intellectual property laws.  To protect TI's         *
* rights in the Program, you agree not to decompile, reverse         *
* engineer, disassemble or otherwise translate any object code       *
* versions of the Program to a human-readable form.  You agree       *
* that in no event will you alter, remove or destroy any             *
* copyright notice included in the Program.  TI reserves all         *
* rights not specifically granted under this license. Except         *
* as specifically provided herein, nothing in this agreement         *
* shall be construed as conferring by implication, estoppel,         *
* or otherwise, upon you, any license or other right under any       *
* TI patents, copyrights or trade secrets.                           *
*                                                                    *
* You may not use the Program in non-TI devices.                     *
**********************************************************************


;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Global symbol declarations
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                .def start, timer2_isr

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Address definitions
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                .include f2407.h

DAC0            .set    0000h           ;EVM DAC register 0 (I/O space)
DAC1            .set    0001h           ;EVM DAC register 1 (I/O space)
DAC2            .set    0002h           ;EVM DAC register 2 (I/O space)
DAC3            .set    0003h           ;EVM DAC register 3 (I/O space)
DACUD           .set    0004h           ;EVM DAC update register (I/O space)
DIPSWCH         .set    0008h           ;EVM DIP switch (I/O space)
LED             .set    000Ch           ;EVM LED bank (I/O space)

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Constant definitions
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
timer2_per      .set    58594           ;250ms timer2 period with a 1/128
                                        ;timer prescaler and 30MHz CPUCLK

pwm_half_per    .set    750             ;period/2, 20KHz symmetric PWM with
                                        ;a 30MHz CPUCLK
                                        
pwm_duty        .set    563             ;25% PWM duty cycle

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Uninitialized global variable definitions
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                .bss    temp1,1         ;general purpose variable
                .bss    LED_index,1     ;LED index


**********************************************************************
*                     M A I N   R O U T I N E                        *
**********************************************************************

        .text
start:

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Configure the System Control and Status Registers
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        LDP     #DP_PF1         ;set data page

        SPLK    #0000000011111101b, SCSR1
*                ||||||||||||||||
*                FEDCBA9876543210
* bit 15        0:      reserved
* bit 14        0:      CLKOUT = CPUCLK
* bit 13-12     00:     IDLE1 selected for low-power mode
* bit 11-9      000:    PLL x4 mode
* bit 8         0:      reserved
* bit 7         1:      1 = enable ADC module clock
* bit 6         1:      1 = enable SCI module clock
* bit 5         1:      1 = enable SPI module clock
* bit 4         1:      1 = enable CAN module clock
* bit 3         1:      1 = enable EVB module clock
* bit 2         1:      1 = enable EVA module clock
* bit 1         0:      reserved
* bit 0         1:      clear the ILLADR bit

        LACC    SCSR2                   ;ACC = SCSR2 register
        OR      #0000000000001011b      ;OR in bits to be set
        AND     #0000000000001111b      ;AND out bits to be cleared
*                ||||||||||||||||
*                FEDCBA9876543210
* bit 15-6      0's:    reserved
* bit 5         0:      do NOT clear the WD OVERRIDE bit
* bit 4         0:      XMIF_HI-Z, 0=normal mode, 1=Hi-Z'd
* bit 3         1:      disable the boot ROM, enable the FLASH
* bit 2     no change   MP/MC* bit reflects the state of the MP/MC* pin
* bit 1-0      11:      11 = SARAM mapped to prog and data (default)

        SACL    SCSR2                   ;store to SCSR2 register

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Disable the watchdog timer
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        LDP     #DP_PF1                 ;set data page

        SPLK    #0000000011101000b, WDCR
*                ||||||||||||||||
*                FEDCBA9876543210
* bits 15-8     0's      reserved
* bit 7         1:       clear WD flag
* bit 6         1:       disable the dog
* bit 5-3       101:     must be written as 101
* bit 2-0       000:     WDCLK divider = 1

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Setup external memory interface for LF2407 EVM
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        LDP     #temp1                  ;set data page

        SPLK    #0000000001000000b, temp1
*                ||||||||||||||||
*                FEDCBA9876543210
* bit 15-11     0's:    reserved
* bit 10-9      00:     bus visibility off
* bit 8-6       001:    1 wait-state for I/O space
* bit 5-3       000:    0 wait-state for data space
* bit 2-0       000:    0 wait state for program space

        OUT     temp1, WSGR

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Setup shared I/O pins
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        LDP     #DP_PF2                 ;set data page

        SPLK    #0000000001000000b,MCRA ;group A pins
*                ||||||||||||||||
*                FEDCBA9876543210
* bit 15        0:      0=IOPB7,     1=TCLKINA
* bit 14        0:      0=IOPB6,     1=TDIRA
* bit 13        0:      0=IOPB5,     1=T2PWM/T2CMP
* bit 12        0:      0=IOPB4,     1=T1PWM/T1CMP
* bit 11        0:      0=IOPB3,     1=PWM6
* bit 10        0:      0=IOPB2,     1=PWM5
* bit 9         0:      0=IOPB1,     1=PWM4
* bit 8         0:      0=IOPB0,     1=PWM3
* bit 7         0:      0=IOPA7,     1=PWM2
* bit 6         1:      0=IOPA6,     1=PWM1
* bit 5         0:      0=IOPA5,     1=CAP3
* bit 4         0:      0=IOPA4,     1=CAP2/QEP2
* bit 3         0:      0=IOPA3,     1=CAP1/QEP1
* bit 2         0:      0=IOPA2,     1=XINT1
* bit 1         0:      0=IOPA1,     1=SCIRXD
* bit 0         0:      0=IOPA0,     1=SCITXD

        SPLK    #1111111000000000b,MCRB ;group B pins
*                ||||||||||||||||
*                FEDCBA9876543210
* bit 15        1:      0=reserved,  1=TMS2 (always write as 1)
* bit 14        1:      0=reserved,  1=TMS  (always write as 1)
* bit 13        1:      0=reserved,  1=TD0  (always write as 1)
* bit 12        1:      0=reserved,  1=TDI  (always write as 1)
* bit 11        1:      0=reserved,  1=TCK  (always write as 1)
* bit 10        1:      0=reserved,  1=EMU1 (always write as 1)
* bit 9         1:      0=reserved,  1=EMU0 (always write as 1)
* bit 8         0:      0=IOPD0,     1=XINT2/ADCSOC
* bit 7         0:      0=IOPC7,     1=CANRX
* bit 6         0:      0=IOPC6,     1=CANTX
* bit 5         0:      0=IOPC5,     1=SPISTE
* bit 4         0:      0=IOPC4,     1=SPICLK
* bit 3         0:      0=IOPC3,     1=SPISOMI
* bit 2         0:      0=IOPC2,     1=SPISIMO
* bit 1         0:      0=IOPC1,     1=BIO*
* bit 0         0:      0=IOPC0,     1=W/R*

        SPLK    #0000000000000000b,MCRC ;group C pins
*                ||||||||||||||||
*                FEDCBA9876543210
* bit 15        0:      reserved
* bit 14        0:      0=IOPF6,     1=IOPF6
* bit 13        0:      0=IOPF5,     1=TCLKINB
* bit 12        0:      0=IOPF4,     1=TDIRB
* bit 11        0:      0=IOPF3,     1=T4PWM/T4CMP
* bit 10        0:      0=IOPF2,     1=T3PWM/T3CMP
* bit 9         0:      0=IOPF1,     1=CAP6
* bit 8         0:      0=IOPF0,     1=CAP5/QEP4
* bit 7         0:      0=IOPE7,     1=CAP4/QEP3
* bit 6         0:      0=IOPE6,     1=PWM12
* bit 5         0:      0=IOPE5,     1=PWM11
* bit 4         0:      0=IOPE4,     1=PWM10
* bit 3         0:      0=IOPE3,     1=PWM9
* bit 2         0:      0=IOPE2,     1=PWM8
* bit 1         0:      0=IOPE1,     1=PWM7
* bit 0         0:      0=IOPE0,     1=CLKOUT

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Configure IOPC0 pin as an output
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        LDP     #DP_PF2                 ;set data page
        LACC    #0100h                  ;ACC = 0100h
        OR      PCDATDIR                ;OR in PCDATDIR register
        SACL    PCDATDIR                ;store result to PCDATDIR

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Setup the software stack
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
stk_len        .set    100              ;stack length
stk            .usect  "stack",stk_len  ;reserve space for stack

        LAR     AR1, #stk               ;AR1 is the stack pointer

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Setup timers 1 and 2, and the PWM configuration
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        LDP     #DP_EVA                 ;set data page
        SPLK    #0000h, T1CON           ;disable timer 1
        SPLK    #0000h, T2CON           ;disable timer 2

        SPLK    #0000000000000000b, GPTCONA
*                ||||||||||||||||
*                FEDCBA9876543210
* bit 15        0:      reserved
* bit 14        0:      T2STAT, read-only
* bit 13        0:      T1STAT, read-only
* bit 12-11     00:     reserved
* bit 10-9      00:     T2TOADC, 00 = no timer2 event starts ADC
* bit 8-7       00:     T1TOADC, 00 = no timer1 event starts ADC
* bit 6         0:      TCOMPOE, 0 = Hi-z all timer compare outputs
* bit 5-4       00:     reserved
* bit 3-2       00:     T2PIN, 00 = forced low
* bit 1-0       00:     T1PIN, 00 = forced low

;Timer 1: Configure to clock the PWM on PWM1 pin.
;Symmetric PWM, 20KHz carrier frequency, 25% duty cycle
        SPLK    #0000h, T1CNT           ;clear timer counter
        SPLK    #pwm_half_per, T1PR     ;set timer period
        SPLK    #0000h, DBTCONA         ;deadband units off
        SPLK    #pwm_duty, CMPR1        ;set PWM duty cycle

        SPLK    #0000000000000010b, ACTRA
*                ||||||||||||||||
*                FEDCBA9876543210
* bit 15        0:      space vector dir is CCW (don't care)
* bit 14-12     000:    basic space vector is 000 (dont' care)
* bit 11-10     00:     PWM6/IOPB3 pin forced low
* bit 9-8       00:     PWM5/IOPB2 pin forced low
* bit 7-6       00:     PWM4/IOPB1 pin forced low
* bit 5-4       00:     PWM3/IOPB0 pin forced low
* bit 3-2       00:     PWM2/IOPA7 pin forced low
* bit 1-0       10:     PWM1/IOPA6 pin active high

        SPLK    #1000001000000000b, COMCONA
*                ||||||||||||||||
*                FEDCBA9876543210
* bit 15        1:      1 = enable compare operation
* bit 14-13     00:     00 = reload CMPRx regs on timer 1 underflow
* bit 12        0:      0 = space vector disabled
* bit 11-10     00:     00 = reload ACTR on timer 1 underflow
* bit 9         1:      1 = enable PWM pins
* bit 8-0       0's:    reserved

        SPLK    #0000100001000000b, T1CON
*                ||||||||||||||||
*                FEDCBA9876543210
* bit 15-14     00:     stop immediately on emulator suspend
* bit 13        0:      reserved
* bit 12-11     01:     01 = continous-up/down count mode
* bit 10-8      000:    000 = x/1 prescaler
* bit 7         0:      reserved in T1CON
* bit 6         1:      TENABLE, 1 = enable timer
* bit 5-4       00:     00 = CPUCLK is clock source
* bit 3-2       00:     00 = reload compare reg on underflow
* bit 1         0:      0 = disable timer compare
* bit 0         0:      reserved in T1CON

;Timer 2: configure to generate a 250ms periodic interrupt
        SPLK    #0000h, T2CNT           ;clear timer counter
        SPLK    #timer2_per, T2PR       ;set timer period
        
        SPLK    #1101011101000000b, T2CON
*                ||||||||||||||||
*                FEDCBA9876543210
* bit 15-14     11:     stop immediately on emulator suspend
* bit 13        0:      reserved
* bit 12-11     10:     10 = continous-up count mode
* bit 10-8      111:    111 = x/128 prescaler
* bit 7         0:      T2SWT1, 0 = use own TENABLE bit
* bit 6         1:      TENABLE, 1 = enable timer
* bit 5-4       00:     00 = CPUCLK is clock source
* bit 3-2       00:     00 = reload compare reg on underflow
* bit 1         0:      0 = disable timer compare
* bit 0         0:      SELT1PR, 0 = use own period register

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Other setup
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;LED index initialization
        LDP     #LED_index              ;set data page
        SPLK    #1h, LED_index          ;initialize the LED index

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Setup the core interrupts
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        LDP     #0h                     ;set data page
        SPLK    #0h,IMR                 ;clear the IMR register
        SPLK    #111111b,IFR            ;clear any pending core interrupts
        SPLK    #000100b,IMR            ;enable desired core interrupts

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Setup the event manager interrupts
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        LDP     #DP_EVA                 ;set data page
        SPLK    #0FFFFh, EVAIFRA        ;clear all EVA group A interrupts
        SPLK    #0FFFFh, EVAIFRB        ;clear all EVA group B interrupts
        SPLK    #0FFFFh, EVAIFRC        ;clear all EVA group C interrupts
        SPLK    #00000h, EVAIMRA        ;enabled desired EVA group A interrupts
        SPLK    #00001h, EVAIMRB        ;enabled desired EVA group B interrupts
        SPLK    #00000h, EVAIMRC        ;enabled desired EVA group C interrupts

        LDP     #DP_EVB                 ;set data page
        SPLK    #0FFFFh, EVBIFRA        ;clear all EVB group A interrupts
        SPLK    #0FFFFh, EVBIFRB        ;clear all EVB group B interrupts
        SPLK    #0FFFFh, EVBIFRC        ;clear all EVB group C interrupts
        SPLK    #00000h, EVBIMRA        ;enabled desired EVB group A interrupts
        SPLK    #00000h, EVBIMRB        ;enabled desired EVB group B interrupts
        SPLK    #00000h, EVBIMRC        ;enabled desired EVB group C interrupts

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Enable global interrupts
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        CLRC    INTM                    ;enable global interrupts

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;Main loop
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
loop:
        NOP
        B       loop                    ;branch to loop   


**********************************************************************
*  I N T E R R U P T  S E R V I C E  R O U T I N E S                 *
**********************************************************************

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;GP Timer 2 period interrupt (core interrupt INT3)
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

timer2_isr:

;Context save to the software stack
        MAR     *,AR1                   ;ARP=stack pointer
        MAR     *+                      ;skip one stack location (required)
        SST     #1, *+                  ;save ST1
        SST     #0, *+                  ;save ST0
        SACH    *+                      ;save ACCH
        SACL    *+                      ;save ACCL

;Clear the T2PINT interrupt flag
        LDP     #DP_EVA                 ;set data page
        SPLK    #00001h, EVAIFRB        ;clear T2PINT flag

;Sequence the LED bank on the LF2407 EVM
        LDP     #LED_index              ;set data page
        OUT     LED_index, LED          ;light the LED
        LACC    LED_index,1             ;load LED index with left shift of 1
        SACL    LED_index               ;store updated index
        SUB     #0010h                  ;subtract the mask
        BCND    done, LT                ;branch if index not ready for reset
        SPLK    #1h, LED_index          ;reset LED index to 1
done:

;Toggle the IOPC0 pin
        LDP     #DP_PF2                 ;set data page
        LACC    #0001h                  ;ACC = 0001h
        XOR     PCDATDIR                ;XOR the IOPC0 bit to toggle the pin
        SACL    PCDATDIR                ;store result to PCDATDIR

;context restore from the software stack
        MAR     *, AR1                  ;ARP = AR1
        MAR     *-                      ;SP points to last entry
        LACL    *-                      ;restore ACCL
        ADD     *-,16                   ;restore ACCH
        LST     #0, *-                  ;restore ST0
        LST     #1, *-                  ;restore ST1, unskip one stack location
        CLRC    INTM                    ;re-enable interrupts
        RET                             ;return from the interrupt
