/* =============================    edma.c    ============================== */

// ============================    INCLUDES   =================================
#include "dsk6455cfg.h"
#include "main.h"	// contains program-wide definitions like BUFFSIZE, etc.
#include "edma_int_dispatcher.h"
#include "edma_resource_allocation.h"
#include <csl.h>
#include <csl_edma3.h>

// The following code allows you to change the TCC values in
// edma_resource_allocation.h without having to make any additional
// code changes due to whether your TCC is greater or less than 31.
#if TCC_EDMA_MCBSP1_RCV > 31
	#define TCC_EDMA_MCBSP1_RCV_MASK_LOW  (0)
	#define TCC_EDMA_MCBSP1_RCV_MASK_HIGH (1<<TCC_EDMA_MCBSP1_RCV-32)
#else
	#define TCC_EDMA_MCBSP1_RCV_MASK_LOW  (1<<TCC_EDMA_MCBSP1_RCV)
	#define TCC_EDMA_MCBSP1_RCV_MASK_HIGH (0)
#endif

#if TCC_EDMA_MCBSP1_XMT > 31
	#define TCC_EDMA_MCBSP1_XMT_MASK_LOW  (0)
	#define TCC_EDMA_MCBSP1_XMT_MASK_HIGH (1<<TCC_EDMA_MCBSP1_XMT-32)
#else
	#define TCC_EDMA_MCBSP1_XMT_MASK_LOW  (1<<TCC_EDMA_MCBSP1_XMT)
	#define TCC_EDMA_MCBSP1_XMT_MASK_HIGH (0)
#endif

// =========    PROTOTYPES      ================================================
void edma_mcbsp_rcv_handler(void);
void edma_mcbsp_xmt_handler(void);


// =========    GLOBALS       ================================================

// 1 Edma3Obj and 1 Edma3Handle (points to Edma3Obj) are needed for entire EDMA3
// should be global such that edma_int_dispatcher can access them
CSL_Edma3Obj                edmaObj;
CSL_Edma3Handle 			hEdmaModule;

/******************************************************************************
 *  FUNCTION: setup_edma
 *
 *  PARAMETERS: none
 *
 *  DESCRIPTION: Sets up the EDMA3 to handle data flow to/from McBSP1
 *
 *  RETURNS: none
 *
 *****************************************************************************/
void setup_edma(void)
{
	CSL_Edma3Context            context;
	CSL_Status                  status;   

	// ChannelHandles point to ChannelObj
	CSL_Edma3ChannelObj         chObjXmt, chObjRcv;
	CSL_Edma3ChannelHandle      hChannelXmt, hChannelRcv;

	// Edma3ParamHandles point to specific PaRAM Sets after calling CSL_edma3GetParamHandle
	CSL_Edma3ParamHandle        hParam, hParamPing, hParamPong;

	// This is the argument for manipulating EDMA interrupts (enable, disable, clear).
	// It contains a bit masks (intr and intrh) for the interrupts to be manipulated.
	CSL_Edma3CmdIntr            edmaIntr;

	// These structures contain info such as the channel to be opened
	CSL_Edma3ChannelAttr        chAttrXmt, chAttrRcv;

	//PaRAM settings for generic 1 dimension EDMA receive from McBSP 1. code needs
	// to set up dstAddr and linkBcntrld during run time
	// ref: spru966 table 2-2 & table 2-3
	CSL_Edma3ParamSetup gParamSetupRcv = {  // PaRAM Set Structure for receive ping buffer
	    CSL_EDMA3_OPT_MAKE                  // option -      OPT
	        (CSL_EDMA3_ITCCH_DIS,     // DIS = no intermediate chaining  
	        CSL_EDMA3_TCCH_DIS,       // DIS = no final chaining
	        CSL_EDMA3_ITCINT_DIS,     // DIS = no intermediate CPU interrupt 
	        CSL_EDMA3_TCINT_EN,       // EN = generate CPU interrupt at the end of transfer
	        TCC_EDMA_MCBSP1_RCV, 	  // TCC value (0-63)
	        CSL_EDMA3_TCC_NORMAL, 	  // in normal mode interrupt occurs when transfer completes
	        CSL_EDMA3_FIFOWIDTH_NONE, // always set to "NONE"
	        CSL_EDMA3_STATIC_DIS,     // only useful with QDMA
	        CSL_EDMA3_SYNC_A,         // A-sync wait for an event for each array transferred
	        CSL_EDMA3_ADDRMODE_INCR,  // always use "INCR"
	        CSL_EDMA3_ADDRMODE_INCR), // always use "INCR"
	    CSL_MCBSP_1_RX_EDMA_REGS,	 	// srcAddr = 0x3400 0000: McBSP-1 Data Receive Register
	    CSL_EDMA3_CNT_MAKE(2,NUM_CHANNELS), // aCntbCnt: ACNT = 2, BCNT = NUM_CHANNELS
	    (uint32_t) NULL,                	// dstAddr: We will map to a Receive buffer address in the code
	    CSL_EDMA3_BIDX_MAKE(0,2*BUFFSIZE/NUM_CHANNELS),       // srcDstBidx: (SRCBIDX, DSTBIDX)
	    (uint32_t) NULL,  				// linkBcntrld: We will link this field to PaRam Sets
	    CSL_EDMA3_CIDX_MAKE(0,2-2*BUFFSIZE/NUM_CHANNELS),       // srcDstCidx: (SRCCIDX, DSTCIDX)
	    (uint32_t) BUFFSIZE/NUM_CHANNELS// cCnt:  (CCNT)
	};

	// PaRAM settings for generic 1 dimension EDMA transmit to McBSP 1. The code
	// needs to set up srcAddr and linkBcntrld during run time
	CSL_Edma3ParamSetup gParamSetupXmt = {  // PaRAM Set Structure for transmit ping buffer
	    CSL_EDMA3_OPT_MAKE                      // option -      OPT
	        (CSL_EDMA3_ITCCH_DIS,     // DIS = no intermediate chaining  
	        CSL_EDMA3_TCCH_DIS,       // DIS = no final chaining
	        CSL_EDMA3_ITCINT_DIS,     // DIS = no intermediate CPU interrupt 
	        CSL_EDMA3_TCINT_EN,       // EN = generate CPU interrupt at the end of transfer
	        TCC_EDMA_MCBSP1_XMT, 	  // TCC value (0-63)
	        CSL_EDMA3_TCC_NORMAL, 	  // in normal mode interrupt occurs when transfer completes
	        CSL_EDMA3_FIFOWIDTH_NONE, // always set to "NONE"
	        CSL_EDMA3_STATIC_DIS,     // only useful with QDMA
	        CSL_EDMA3_SYNC_A,         // A-sync wait for an event for each array transferred
	        CSL_EDMA3_ADDRMODE_INCR,  // always use "INCR"
	        CSL_EDMA3_ADDRMODE_INCR), // always use "INCR"
	    (uint32_t) NULL,                	// srcAddr - We will map to a Transmit buffer address in the code
	    CSL_EDMA3_CNT_MAKE(2,NUM_CHANNELS), // aCntbCnt: ACNT = 2, BCNT = NUM_CHANNELS
		CSL_MCBSP_1_TX_EDMA_REGS,		// dstAddr = 0x3400 0010: McBSP-1 Data Transmit Register
	    CSL_EDMA3_BIDX_MAKE(2*BUFFSIZE/NUM_CHANNELS,0),       // srcDstBidx: (SRCBIDX, DSTBIDX)
	    (uint32_t) NULL,  				// linkBcntrld: We will link this field to PaRam Sets
	    CSL_EDMA3_CIDX_MAKE(2-2*BUFFSIZE/NUM_CHANNELS,0),       // srcDstCidx: (SRCCIDX, DSTCIDX)
	    (uint32_t) BUFFSIZE/NUM_CHANNELS// cCnt:  (CCNT)
	};

// ****************************************************************************
// ************************** Module Initialization ***************************
// **************************** Get Module Handle *****************************
// ****************************************************************************

	// CSL step 1: Module initialization
    status = CSL_edma3Init(&context);
    
	// CSL step 2: Module open
	// initializes the edmaObj structure with pointer to EDMA registers
	// returns hEdmaModule such that it points to edmaObj
    hEdmaModule = CSL_edma3Open(&edmaObj,CSL_EDMA3,NULL,&status);

// ****************************************************************************
// ************************** Channel Initialization **************************
// ******************** EDMA setup for McBSP1 Receive Event *******************
// ****************************************************************************
   
	// EDMA Channel Step 1: Get Channel Handle
	// Initialize the chObjRcv structure (pointer to EDMA registers, etc.).
	// Get handle (hChannelRcv) to channel object (chObjRcv).
    chAttrRcv.regionNum = CSL_EDMA3_REGION_GLOBAL;
    chAttrRcv.chaNum = CSL_EDMA3_CHA_REVT1; // SPRS276, table 7-4: EDMA Channel Synchronization Events
    hChannelRcv = CSL_edma3ChannelOpen(&chObjRcv, CSL_EDMA3, &chAttrRcv, &status);   
        
	// EDMA Channel Step 2: Set up the channel
    CSL_edma3HwChannelSetupParam(hChannelRcv, PSET_EDMA_MCBSP1_RCV);	// maps channel to PSET
	CSL_edma3HwChannelSetupQue(hChannelRcv, CSL_EDMA3_QUE_1);			// maps channel to queue

	// EDMA Channel Step 3: Get PaRAM Set Handles
    hParam = CSL_edma3GetParamHandle(hChannelRcv, PSET_EDMA_MCBSP1_RCV, &status);
    hParamPing = CSL_edma3GetParamHandle(hChannelRcv, PSET_EDMA_MCBSP1_RCV_PING, &status);        
    hParamPong = CSL_edma3GetParamHandle(hChannelRcv, PSET_EDMA_MCBSP1_RCV_PONG, &status);

	// EDMA Channel Step 4: Write to the PaRAM sets
    gParamSetupRcv.dstAddr = (uint32_t)gBufferRcv[0];	//point destination to PING buffer
    gParamSetupRcv.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamPong, NUM_CHANNELS);
    CSL_edma3ParamSetup(hParam, &gParamSetupRcv); 
    CSL_edma3ParamSetup(hParamPing, &gParamSetupRcv);
    gParamSetupRcv.dstAddr = (uint32_t)gBufferRcv[1]; //point destination to PONG buffer
    gParamSetupRcv.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamPing, NUM_CHANNELS);
    CSL_edma3ParamSetup(hParamPong, &gParamSetupRcv); 

	// EDMA Channel Step 5: Set EDMA Event Enable Registers (EERH + EER)
	// clear any pending event and then enable channel
	CSL_edma3HwChannelControl(hChannelRcv,CSL_EDMA3_CMD_CHANNEL_CLEAR,NULL);
    CSL_edma3HwChannelControl(hChannelRcv,CSL_EDMA3_CMD_CHANNEL_ENABLE,NULL);

	// Plug the handlers for the EDMA completion interrupts into our edma
	// interrupt dispatcher
	edma_int_hook(TCC_EDMA_MCBSP1_RCV, (EdmaTccHandler)&edma_mcbsp_rcv_handler);

// ****************************************************************************
// ************************** Channel Initialization **************************
// ******************* EDMA setup for McBSP1 Transmit Event *******************
// ****************************************************************************

	// EDMA Channel Step 1: Get Channel Handle
	// Initialize the chObjRcv structure (pointer to EDMA registers, etc.).
	// Get handle (hChannelRcv) to channel object (chObjRcv).
    chAttrXmt.regionNum = CSL_EDMA3_REGION_GLOBAL;
    chAttrXmt.chaNum = CSL_EDMA3_CHA_XEVT1; // SPRS276, table 7-4: EDMA Channel Synchronization Events
    hChannelXmt = CSL_edma3ChannelOpen(&chObjXmt, CSL_EDMA3, &chAttrXmt, &status);   
        
	// EDMA Channel Step 2: Set up channel 
    CSL_edma3HwChannelSetupParam(hChannelXmt, PSET_EDMA_MCBSP1_XMT);		// maps channel to PSET
	CSL_edma3HwChannelSetupQue(hChannelXmt, CSL_EDMA3_QUE_1);			// maps channel to queue

	// EDMA Channel Step 3: Get PaRAM Set Handles
    hParam = CSL_edma3GetParamHandle(hChannelXmt, PSET_EDMA_MCBSP1_XMT, &status);
    hParamPing = CSL_edma3GetParamHandle(hChannelXmt, PSET_EDMA_MCBSP1_XMT_PING, &status);        
    hParamPong = CSL_edma3GetParamHandle(hChannelXmt, PSET_EDMA_MCBSP1_XMT_PONG, &status);

	// EDMA Channel Step 4: Write to the PaRAM sets
    gParamSetupXmt.srcAddr = (uint32_t)gBufferXmt[0]; //point source to PING buffer
    gParamSetupXmt.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamPong, NUM_CHANNELS);
    CSL_edma3ParamSetup(hParam, &gParamSetupXmt);
    CSL_edma3ParamSetup(hParamPing, &gParamSetupXmt); //write to PaRam Set PING
    gParamSetupXmt.srcAddr = (uint32_t)gBufferXmt[1]; //point source to PONG buffer
    gParamSetupXmt.linkBcntrld = CSL_EDMA3_LINKBCNTRLD_MAKE(hParamPing, NUM_CHANNELS);
    CSL_edma3ParamSetup(hParamPong, &gParamSetupXmt); //write to PaRam Set PONG

	// EDMA Channel Step 5: Set EDMA Event Enable Registers (EERH + EER)
	// clear any pending event and then enable channel
	CSL_edma3HwChannelControl(hChannelXmt,CSL_EDMA3_CMD_CHANNEL_CLEAR,NULL);
    CSL_edma3HwChannelControl(hChannelXmt,CSL_EDMA3_CMD_CHANNEL_ENABLE,NULL);

	// Plug the handlers for the EDMA completion interrupts into our edma
	// interrupt dispatcher
	edma_int_hook(TCC_EDMA_MCBSP1_XMT, (EdmaTccHandler)&edma_mcbsp_xmt_handler);

// ****************************************************************************
// ********************* Module Initialization Continued **********************
// ***************************** Interrupt Setup ******************************
// ****************************************************************************

	// Clear any pending interrupts then enable interrupts
    edmaIntr.region = CSL_EDMA3_REGION_GLOBAL ;
	edmaIntr.intr = TCC_EDMA_MCBSP1_RCV_MASK_LOW | TCC_EDMA_MCBSP1_XMT_MASK_LOW;
	edmaIntr.intrh = TCC_EDMA_MCBSP1_RCV_MASK_HIGH | TCC_EDMA_MCBSP1_XMT_MASK_HIGH;
	CSL_edma3HwControl(hEdmaModule, CSL_EDMA3_CMD_INTRPEND_CLEAR, &edmaIntr);
    CSL_edma3HwControl(hEdmaModule, CSL_EDMA3_CMD_INTR_ENABLE, &edmaIntr);    

}

/******************************************************************************
 *  FUNCTION: edma_mcbsp_rcv_handler
 *
 *  PARAMETERS: none
 *
 *  DESCRIPTION: gets called by edma_int_dispatcher when the EDMA completion
 *               interrupt for the McBSP receive happens
 *
 *  RETURNS: none
 *
 *****************************************************************************/
void edma_mcbsp_rcv_handler(void)
{
	SEM_post(&rcvBuffReady);
}

/******************************************************************************
 *  FUNCTION: edma_mcbsp_xmt_handler
 *
 *  PARAMETERS: none
 *
 *  DESCRIPTION: gets called by edma_int_dispatcher when the EDMA completion
 *               interrupt for the McBSP transmit happens
 *
 *  RETURNS: none
 *
 *****************************************************************************/
void edma_mcbsp_xmt_handler(void)
{
	SEM_post(&xmtBuffReady);
}

/* ==========================    END OF FILE    =========================== */
