/* ============================    tasks.c    ============================== */

// =============================   INCLUDES  ==================================
#include "audio_appcfg.h"	// BIOS object definitions (semaphores, etc.)
#include <dsk6455_dip.h>	// access to BSL DIP switch API
#include <dsk6455_led.h> // from the BSL
#include <stdint.h>  // used for definitions like uint32_t, ...
#include <string.h>		// memcpy
#include "main.h"	// definition of BUFFSIZE, etc.
#include "mcbsp_aic23.h"	// definitions of pDRR and pDXR
#include <csl_edma3.h>	// used for a polling workaround

// =========    Global       ================================================

// make volatile so compiler doesn't optimize
// global such that you can change load through watch window
volatile int32_t load=50, inner_loop=235000;


/*
 *  FUNCTION: tskDipLED
 *
 *  PARAMETERS: none
 *
 *  DESCRIPTION: This is a periodic function designed to blink an LED and
 *               add CPU load when a dip switch is depressed
 *
 *  RETURNS: none
 *
 */
void tskDipLED(void)
{
// make volatile so compiler doesn't optimize
volatile int32_t i, j;

	while(1)
	{
		SEM_pend(&???, SYS_FOREVER);

		DSK6455_LED_toggle(0);		// toggle LED D10

		// If dip switch 0 on SW1 is depressed then insert dummy load.
		// The CPU load can be configured through the watch window by
		// changing the variable called "load".
		if (DSK6455_DIP_get(0)==0)
		{
			// dummy load
			for(i=0; i<load; i++)
			{
				for(j=0; j<inner_loop; j++);
			}
		}
	}
}


/*
 *  FUNCTION: getBuffTsk
 *
 *  PARAMETERS: none
 *
 *  DESCRIPTION: This function does all the buffer management.  We are doing
 *               double-buffering in this code.  It keeps track of whether it
 *               is writing to "ping" or "pong".  Once it has filled a buffer
 *               it posts a semaphore allowing the procBuffTsk function to run
 *               and process the buffer (i.e. memcpy rcv to xmt).
 *
 *  RETURNS: none
 *
 */
void getBuffTsk(void)
{
	uint32_t rcv_count=0, xmt_count=0;
	int16_t rcv_ping=0, xmt_ping=0;

	CSL_Edma3Obj                edmaObj;
	CSL_Edma3Handle 			hEdmaModule;
	CSL_Edma3Context            context;
	CSL_Status                  status;

	// initialize the EDMA handle to use for register access
    status = CSL_edma3Init(&context);
    hEdmaModule = CSL_edma3Open(&edmaObj,CSL_EDMA3,NULL,&status);

	while(1)
	{
		// look for McBSP RRDY
		// There is a problem with reading SPCR in the McBSP regs
		// so as a workaround the EDMA Event Register is used
		if( (hEdmaModule->regs->ER&0x8000) == 0x8000 )
		{
			hEdmaModule->regs->ECR = 0x8000;	// clear corresponding flag

			// Read data from DRR and put in receive buffer
			gBufferRcv[rcv_ping][rcv_count++] = *pDRR;	// 16-bit read from DRR
			if (rcv_count == BUFFSIZE)
			{
				rcv_count=0;
				rcv_ping ^= 1;
				SEM_post(&???);  //post Receive semaphore
			}
		}

		// look for McBSP XRDY
		// There is a problem with reading SPCR in the McBSP regs
		// so as a workaround the EDMA Event Register is used
		if( (hEdmaModule->regs->ER&0x4000) == 0x4000 )
		{
			hEdmaModule->regs->ECR = 0x4000;	// clear corresponding flag

			// Read data from transmit buffer and output to DXR
			*pDXR = gBufferXmt[xmt_ping][xmt_count++];	// 16-bit write to DXR
			if (xmt_count == BUFFSIZE)
			{
				xmt_count=0;
				xmt_ping ^= 1;
				SEM_post(&???);
			}
		}
	}
}


/*
 *  FUNCTION: procBuffTsk
 *
 *  PARAMETERS: none
 *
 *  DESCRIPTION:  This function processes the buffer of receive data.  Currently
 *                the function simply copies receive to transmit (i.e. it does
 *                a pass through).  In a real application this would be
 *                replaced by a signal processing algorithm.
 *
 *  RETURNS: none
 *
 */
void procBuffTsk(void)
{
	int16_t pingPong = PING;

	while (1)
	{
	    SEM_pend(&???, SYS_FOREVER); //pend on receiver buffer ready
	    SEM_pend(&???, SYS_FOREVER); //pend on transmit buffer ready

		// following this comment line is where you would put your DSP algorithm
		memcpy(gBufferXmt[pingPong], gBufferRcv[pingPong], BUFFSIZE*sizeof(int16_t));
	    pingPong ^= 1; // switch buffer index for the next run

	}		
}

/* ==========================    END OF FILE    =========================== */

