/**************************************************************************************************
  Filename:       hal_drivers.c
  Revised:        $Date: 2011-09-23 10:34:21 -0700 (Fri, 23 Sep 2011) $
  Revision:       $Revision: 27694 $

  Description:    This file contains the interface to the Drivers Service.


  Copyright 2005-2011 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/


/**************************************************************************************************
 *                                            INCLUDES
 **************************************************************************************************/
#include "hal_types.h"
#include "OSAL.h"
#include "hal_drivers.h"
#include "hal_adc.h"
#if (defined HAL_DMA) && (HAL_DMA == TRUE)
  #include "hal_dma.h"
#endif
#include "hal_flash.h"
#if ((defined HAL_I2C) && (HAL_I2C == TRUE))
  #include "hal_i2c.h"
#endif
#if (defined HAL_KEY) && (HAL_KEY == TRUE)
#include "hal_key.h"
#endif
#if (defined HAL_CAPSENSE) && (HAL_CAPSENSE == TRUE)
#include "hal_capsense.h"
#endif
#include "hal_lcd.h"
#include "hal_led.h"
#include "hal_timer.h"
#include "hal_uart.h"
#include "hal_sleep.h"
#include "hal_vddmon.h"
#if (defined HAL_AES) && (HAL_AES == TRUE)
  #include "hal_aes.h"
#endif
#if (defined HAL_MOTION) && (HAL_MOTION == TRUE)
  #include "hal_gyro.h"
  #include "hal_motion.h"
#endif
#if (defined HAL_SPI) && (HAL_SPI == TRUE)
  #include "hal_spi.h"
#endif
#if (defined HAL_BUZZER) && (HAL_BUZZER == TRUE)
  #include "hal_buzzer.h"
#endif
#if (defined HAL_GPIO_DBG) && (HAL_GPIO_DBG == TRUE)
  #include "hal_gpiodbg.h"
#endif

/**************************************************************************************************
 *                                            MACROS
 **************************************************************************************************/

/**************************************************************************************************
 *                                          CONSTANTS
 **************************************************************************************************/


/**************************************************************************************************
 *                                          TYPEDEFS
 **************************************************************************************************/


/**************************************************************************************************
 *                                      GLOBAL VARIABLES
 **************************************************************************************************/
uint8 Hal_TaskID;

extern void HalLedUpdate( void ); /* Notes: This for internal only so it shouldn't be in hal_led.h */

/**************************************************************************************************
 *                                      FUNCTIONS - API
 **************************************************************************************************/

/**************************************************************************************************
 * @fn      Hal_Init
 *
 * @brief   Hal Initialization function.
 *
 * @param   task_id - Hal TaskId
 *
 * @return  None
 **************************************************************************************************/
void Hal_Init( uint8 task_id )
{
  /* Register task ID */
  Hal_TaskID = task_id;

  /* HalMotionInit() is called here instead of HalDriverInit() because it requires
   * osal services, which aren't available at that time.
   */
#if (defined HAL_MOTION) && (HAL_MOTION == TRUE)
  HalMotionInit();
#endif
}

/**************************************************************************************************
 * @fn      Hal_DriverInit
 *
 * @brief   Initialize HW - These need to be initialized before anyone.
 *
 * @param   task_id - Hal TaskId
 *
 * @return  None
 **************************************************************************************************/
void HalDriverInit (void)
{
  /* VDDMON */
#if (defined HAL_VDDMON) && (HAL_VDDMON == TRUE)
  HalVddMonInit();
#endif

  /* ADC */
#if (defined HAL_ADC) && (HAL_ADC == TRUE)
  HalAdcInit();
#endif

  /* DMA */
#if (defined HAL_DMA) && (HAL_DMA == TRUE)
  // Must be called before the init call to any module that uses DMA.
  HalDmaInit();
#endif

  /* AES */
#if (defined HAL_AES) && (HAL_AES == TRUE)
  HalAesInit();
#endif

  /* LCD */
#if (defined HAL_LCD) && (HAL_LCD == TRUE)
  HalLcdInit();
#endif

  /* LED */
#if (defined HAL_LED) && (HAL_LED == TRUE)
  HalLedInit();
#endif

  /* UART */
#if (defined HAL_UART) && (HAL_UART == TRUE)
  HalUARTInit();
#endif

  /* KEY */
#if (defined HAL_KEY) && (HAL_KEY == TRUE)
  HalKeyInit();
#endif

  /* SPI */
#if (defined HAL_SPI) && (HAL_SPI == TRUE)
  HalSpiInit();
#endif

  /* Buzzer */
#if (defined HAL_BUZZER) && (HAL_BUZZER == TRUE)
  HalBuzzerInit();
#endif

  /* GPIO Debug */
#if (defined HAL_GPIO_DBG) && (HAL_GPIO_DBG == TRUE)
  HalGpioDbgInit();
#endif
  
  /* Cap Sense*/  
#if (defined HAL_CAPSENSE) && (HAL_CAPSENSE == TRUE)
  HalCapSenseInit();
#endif
}

/**************************************************************************************************
 * @fn          halDriverBegPM
 *
 * @brief       This function is called before entering PM so that drivers can be put into their
 *              lowest power states.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void halDriverBegPM(void)
{
#if ((defined HAL_LED) && (HAL_LED == TRUE))
  HalLedEnterSleep();
#endif
#if ((defined HAL_KEY) && (HAL_KEY == TRUE))
  HalKeyEnterSleep();
#endif
#if ((defined HAL_I2C) && (HAL_I2C == TRUE))
  HalI2CEnterSleep();
#endif
}

/**************************************************************************************************
 * @fn          halDriverEndPM
 *
 * @brief       This function is called after exiting PM so that drivers can be restored to their
 *              ready power states.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void halDriverEndPM(void)
{
#if ((defined HAL_LED) && (HAL_LED == TRUE))
  HalLedExitSleep();
#endif
#if ((defined HAL_KEY) && (HAL_KEY == TRUE))
  HalKeyExitSleep();
#endif
#if ((defined HAL_I2C) && (HAL_I2C == TRUE))
  HalI2CExitSleep();
#endif
}

/**************************************************************************************************
 * @fn      Hal_ProcessEvent
 *
 * @brief   Hal Process Event
 *
 * @param   task_id - Hal TaskId
 *          events - events
 *
 * @return  None
 **************************************************************************************************/
uint16 Hal_ProcessEvent( uint8 task_id, uint16 events )
{
  uint8 *msgPtr;

  (void)task_id;  // Intentionally unreferenced parameter

  if ( events & SYS_EVENT_MSG )
  {
    msgPtr = osal_msg_receive(Hal_TaskID);

    while (msgPtr)
    {
      /* Do something here - for now, just deallocate the msg and move on */

      /* De-allocate */
      osal_msg_deallocate( msgPtr );
      /* Next */
      msgPtr = osal_msg_receive( Hal_TaskID );
    }
    return events ^ SYS_EVENT_MSG;
  }

  if ( events & HAL_LED_BLINK_EVENT )
  {
#if (defined (BLINK_LEDS)) && (HAL_LED == TRUE)
    HalLedUpdate();
#endif /* BLINK_LEDS && HAL_LED */
    return events ^ HAL_LED_BLINK_EVENT;
  }

  if (events & HAL_KEY_EVENT)
  {

#if (defined HAL_KEY) && (HAL_KEY == TRUE)
    /* Check for keys */
    HalKeyPoll();

    /* if interrupt disabled, do next polling */
    if (!Hal_KeyIntEnable)
    {
      osal_start_timerEx( Hal_TaskID, HAL_KEY_EVENT, 100);
    }
#endif // HAL_KEY

    return events ^ HAL_KEY_EVENT;
  }
  if
    (events & HAL_CAPSENSE_EVENT)
  {

#if (defined HAL_CAPSENSE) && (HAL_CAPSENSE == TRUE)
    /* Check for CapSense buttons */
    HalCapSensePoll();
    
    //Restart the osal timer
    osal_start_timerEx(Hal_TaskID, HAL_CAPSENSE_EVENT, HAL_CAPSENSE_POLLING_VALUE);
    
#endif // HAL_KEY

    return events ^ HAL_CAPSENSE_EVENT;
  }

#if (defined HAL_MOTION) && (HAL_MOTION == TRUE) && (!defined NO_POINTING_LIB)
  if (events & HAL_MOTION_EVENT)
  {
    uint16 eventHandled = HalMotionHandleOSEvent( events );
    return events ^ eventHandled;
  }

#if (defined HAL_I2C) && (HAL_I2C == TRUE)
  if (events & HAL_GYRO_REGISTER_ACCESS_READY_EVENT)
  {
    HalGyroHandleGyroRegisterAccessReadyEvent();
    return events ^ HAL_GYRO_REGISTER_ACCESS_READY_EVENT;
  }
#endif // #if (defined HAL_I2C) && (HAL_I2C == TRUE)
#endif // #if (defined HAL_MOTION) && (HAL_MOTION == TRUE)

#if (defined HAL_BUZZER) && (HAL_BUZZER == TRUE)
  if (events & HAL_BUZZER_EVENT)
  {
    HalBuzzerStop();
    return events ^ HAL_BUZZER_EVENT;
  }
#endif

  /* Nothing interested, discard the message */
  return 0;

}

/**************************************************************************************************
 * @fn      Hal_ProcessPoll
 *
 * @brief   This routine will be called by OSAL to poll UART, TIMER...
 *
 * @param   task_id - Hal TaskId
 *
 * @return  None
 **************************************************************************************************/
void Hal_ProcessPoll ()
{
#if (defined HAL_VDDMON) && (HAL_VDDMON == TRUE)
  HalVddMonPoll();
#endif
#if ((defined HAL_I2C) && (HAL_I2C == TRUE) && (defined HAL_I2C_POLLED) && (HAL_I2C_POLLED == TRUE))
  HalI2CPoll();
#endif
#if (defined HAL_UART) && (HAL_UART == TRUE)
  HalUARTPoll();
#endif
#if (defined HAL_SPI) && (HAL_SPI == TRUE)
  HalSpiPoll();
#endif
#if (defined HAL_HID) && (HAL_HID == TRUE)
  extern void usbHidProcessEvents(void);
  usbHidProcessEvents();
#endif
}

/**************************************************************************************************
*/
