/*
 * Copyright (C) {2013} Texas Instruments Incorporated - http://www.ti.com/
 *
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the  
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/******************************************************************************/
//  Description: The capacitive touch library function calls are used to
//               detect changes in capacitance caused by an approaching hand.
//               The API calls TI_CAPT_Raw is used to determine a detection
//               while updating the baseline tracking mechanism.
//
//               Once the approaching is detected LED is illuminated.
//               After the measurement the MSP430 enters LPM3 (sleep mode)
//               for ~117ms and then turns off the LED before the next measurement.
//               The measurement time is approximately 29ms.
//
//               The capacitance measurement is performed by comparing the
//               frequency of the pinOsc found in the digital IO of the
//               MSP430G2xx2/3 value line family and the frequency of the
//               internal digitally controlled oscillator (DCO). Please
//               refer to the Capacitive Touch Library Users guide for
//               further description.
//
//
//                 MSP430G2                       10CM PROXIMITY DETECTION
//                 LAUNCHPAD                            BOOSTERPACK
//             -----------------                     -----------------
//            |                 |                   |                 |
//    USB>----|VCC          P1.4|---<Input,Prox<----|Electrode        |
//            |                 |                   |                 |
//            |                 |                   |                 |
//            |             P1.7|-->Output,"LED">---|LED2         LED1|
//            |                 |                   |                 |
//            |              GND|--------- ---------|GND              |
//             -----------------          |          -----------------
//                                       \|/
//
//            Measurement Time: ~29ms
//                              25K PinOsc Cycles (~hz)
//            Sleep (LPM3) Time: 117ms
//            Threshold       : 200 counts
//            Distance        : 10 cm
//
//  Developed with CCS Version : 5.3.0.00090,
//                               w/support for GCC extensions(--gcc)
//
//
//
//  Travis Hwang
//  Texas Instruments, Inc.
//******************************************************************************/

#include "msp430.h"
#include <stdint.h>

//************************************************
// FAST RO DEFINITIONS
//************************************************
#define fRO_PINOSC_TA0_TA1 24


//************************************************
// TIMER SOURCE CLK DIVIDERS
//************************************************
#define TIMER_ACLK    0x0100					// Possible timer source clock dividers
#define TIMER_SMCLK   0x0200					// ACLK and SMCLK are only used


//************************************************
// POWER SOURCE SELECT
//************************************************
// If required, Accumulation Cycle can be selected for better sensitivity
//#define USB_POWER								// Accumulation cycle is set as 25000
#define BATT_POWER								// Higher sensitivity for using battery power


//************************************************
// CONFIGURATIONS
//************************************************
#define RO_TYPE									// Relaxation Oscillator method
#define RO_PINOSC_TYPE							// RO is generated by Pin Oscillator
#define TIMER0A0_GATE							// TimerA0 is used for Gate time
#define TIMER1_OVERFLOW_COUNT					// TimerA1 is used for counting

#define TOTAL_NUMBER_OF_ELEMENTS 1				// Just 1 electode is used for proximity sensing
#define MAXIMUM_NUMBER_OF_ELEMENTS_PER_SENSOR  1


//************************************************
// TIME BETWEEN EACH MEASUREMENT
//************************************************
#define ACTIVE_DELAY 1400						// 1200 12Khz VLO cycles ~117ms


//************************************************
// OPERATION CHECKING SETTING
//************************************************
#define LED		BIT7;							// Port 1.7 is used for LED blinking

//************************************************
// Enable TI TouchPro GUI
//************************************************
//#define TouchPro								// If required, TouchPro GUI can be used
#ifdef TouchPro									// Only 16-bit counting is available for display
uint16_t temp;									// because TouchPro doesn't support 32-bit value
#endif


//************************************************
// VARIABLES FOR ACTUAL MEASURING
//************************************************
uint32_t raw;									// Number of counts for 1 Gate time
uint32_t baseCnt[TOTAL_NUMBER_OF_ELEMENTS];		// Base count to compare actual count
uint16_t count;									// Actually measured count
uint32_t measCnt[MAXIMUM_NUMBER_OF_ELEMENTS_PER_SENSOR];// Measured count(converted to 32bit)


//************************************************
// STRUCTURE FOR SENSOR AND ELEMET
//************************************************
const struct Element proxE;
const struct Sensor proxS;

struct Element{

#ifdef RO_PINOSC_TYPE
	// These register address definitions are needed for each sensor only
	//   when using the PinOsc method
	uint8_t *inputPxselRegister;				// PinOsc: port selection address
	uint8_t *inputPxsel2Register;				// PinOsc: port selection 2 address
#endif
	uint16_t inputBits;							// Input bit definition
	uint16_t threshold;							// specific threshold for each button
	uint16_t maxResponse;						// Special Case: Slider max counts
};


struct Sensor{
  // the method acts as the switch to determine which HAL is called
	uint8_t halDefinition;						// COMPARATOR_TYPE (RO), RC, etc
												// RO_COMPA, RO_COMPB, RO_PINOSC
												// RC_GPIO, RC_COMPA, RC_COMPB
												// FAST_SCAN_RO

	uint8_t numElements;						// Number of elements within group
	uint8_t baseOffset;							// Offset within the global base_cnt array

	struct Element const *arrayPtr[MAXIMUM_NUMBER_OF_ELEMENTS_PER_SENSOR];
												// an array of pointers

//*****************************************************************************
// Timer definitions
//  The basic premise is to count a number of clock cycles within a time
//  period, where either the clock source or the timer period is a function
//  of the element capacitance.
//
// RC Method:
//          Period: accumulationCycles * charge and discharge time of RC
//          circuit where C is capacitive touch element
//
//          clock source: measGateSource/sourceScale
// RO Method:
//          Period: accumulationCycles*measGateSource/sourceScale
//                  (with WDT sourceScale = 1, accumulationCycles is WDT control
//                   register settings)
//
//          clock source: relaxation oscillator where freq is a function of C
//
// fRO Method:
//          Period: accumulationCycles * 1/freq, freq is a function of C
//
//          clock source: measGateSource/sourceScale

	uint16_t measGateSource;					// RC+FastRO: measurement timer source,
												// {ACLK, TACLK, SMCLK}
												// Comp_RO+PinOsc: gate timer source,
												// {ACLK, TACLK, SMCLK}
	uint16_t sourceScale;						// Comp_RO+FastRO+PinOsc: gate timer,
												// TA/TB/TD, scale: 1,1/2,1/4,1/8
												// RC+FastRO: measurement timer, TA/TB/TD
												// scale: 16, 8, 4, 2, 1, ?, ?, 1/8

	uint16_t accumulationCycles;				// Gate time setting using PinOsc AccCycles

};

const struct Element proxE = {

		// In the application, Port 1.4 is used for an electrode
              .inputPxselRegister = (uint8_t *)&P1SEL,
              .inputPxsel2Register = (uint8_t *)&P1SEL2,
              .inputBits = BIT4,
//              .maxResponse = 40 + 655,

		// Threshold to detect proximity can be set as you want
              .threshold = 200
};

const struct Sensor proxS =
{
		// Only 1 element is used for proximity sensing
                  .halDefinition = fRO_PINOSC_TA0_TA1,
                  .numElements = 1,
                  .baseOffset = 0,
                  .arrayPtr[0] = &proxE,

		// Gate time measurement source is SMCLK
                  .measGateSource = TIMER_SMCLK,
//#ifdef BATT_POWER
//                  .accumulationCycles= 40000
//#else
                  .accumulationCycles= 25000
//#endif
};



//************************************************
// FUNCTIONS DECLARATION
//************************************************
void sleep(uint16_t);							// Sleep function enters low power mode(LPM3)
												//   for 'time' VLO clock cycles.

// Measuring function.
// In this application, measuring method is Fast RO
//  using TimerA0, TimerA1 and Pin Oscillator.
void TI_CAPT_Raw(const struct Sensor*, uint32_t*);
void TI_CTS_fRO_PINOSC_TA0_TA1_HAL(const struct Sensor*, uint32_t*);



//************************************************
// MAIN
//************************************************
void main(void)
{
	int j;
#ifdef TouchPro
	uint8_t FrameLength = (3*TOTAL_NUMBER_OF_ELEMENTS) + 1;
	uint8_t CheckSum;
	uint8_t HighByte;
	uint8_t LowByte;
#endif

	WDTCTL = WDTPW + WDTHOLD;					// Stop Watchdog timer

	// In PHASE II devices do not have additional timer and
	//   therefore fRO method is not a good option. Moving back
	//   to the RO method need to slow down SMCLK to 1Mhz
	BCSCTL1 = CALBC1_12MHZ;
	DCOCTL = CALDCO_12MHZ;
	BCSCTL2 |= DIVS_0;							// SMCLK = MCLK = 12Mhz
	BCSCTL3 |= LFXT1S_2;						// LFXT1 = VLO

	// Port initialize
	P1SEL = 0x00;
	P1SEL2 = 0x00;
	P1DIR = 0xFF;
	P1OUT = 0x00;
	P1OUT |= LED;

#ifdef TouchPro
	P1SEL = BIT1 + BIT2 ;						// P1.1 = RXD, P1.2=TXD
	P1SEL2 = BIT1 + BIT2 ;						// P1.1 = RXD, P1.2=TXD
	UCA0CTL1 |= UCSSEL_2;						// SMCLK
	UCA0BR0 = 226;								// 12 MHz 9600 (see User's Guide)
	UCA0BR1 = 4;								// 12 MHz 9600, 1250
	UCA0MCTL |= UCBRS_0 +UCBRF_0;				// Modulation UCBRSx=0, UCBRFx=0
	UCA0CTL1 &= ~UCSWRST;						// **Initialize USCI state machine**
//	IE2 |= UCA0RXIE;							// Enable USCI_A0 RX interrupt
#endif

	sleep(30000);								// Wait to be ready

    // Baseline initialize
	TI_CAPT_Raw(&proxS, &baseCnt[proxS.baseOffset]);

    // Baseline update
	for(j=4; j>0; j--)
	{
		TI_CAPT_Raw(&proxS, measCnt);
		baseCnt[0] = measCnt[0]/2 + baseCnt[0]/2;
	}


	//************************************************
	//	Main loop
	while (1)
	{
		// Turn off indicator before measurement
		sleep(200);
		P1OUT &= ~LED;

		// Collect currently measured counts from Electrode
		TI_CAPT_Raw(&proxS,&raw);

		// Determine proximity detection
		if(raw > (baseCnt[0] + proxE.threshold))	// Approach
		{
			P1OUT |= LED;
		}
		else										// No approach
		{
			// Baseline update
			if(raw < baseCnt[0])					// Change baseline to the average
				baseCnt[0] = (baseCnt[0] + raw)/2;	//   of last baseline and current value.
			else									// Increase baseline slowly when current value
				baseCnt[0]++;						//   is between baseline and baseline+threshold.
		}

#ifdef TouchPro										// Send UART Frame to TouchPro
		CheckSum = 0;
		LowByte = (uint8_t)temp;
		temp = temp >> 8;
		HighByte = (uint8_t)temp;
		while (!(IFG2&UCA0TXIFG));					// USCI_A0 TX buffer ready?
		UCA0TXBUF = 0x55;							// Send Frame Header for TouchPro
		while (!(IFG2&UCA0TXIFG));
		UCA0TXBUF = 0xAA;							// 	Data Header : | 0x55 | 0xAA |
		CheckSum += 0x55;
		CheckSum += 0xAA;

		while (!(IFG2&UCA0TXIFG));
		UCA0TXBUF = FrameLength;					// Send FrameLength
		CheckSum += FrameLength;

		while (!(IFG2&UCA0TXIFG));
		UCA0TXBUF = 0x00;							// Send Signal number
		CheckSum += 0x00;

		while (!(IFG2&UCA0TXIFG));
		UCA0TXBUF = HighByte;
		CheckSum += HighByte;

		while (!(IFG2&UCA0TXIFG));
		UCA0TXBUF = LowByte;
		CheckSum += LowByte;

		while (!(IFG2&UCA0TXIFG));
		UCA0TXBUF = CheckSum;
#endif

		// After sensing, make device go into LPM3
		sleep(ACTIVE_DELAY);

	}	// Main loop end ***************************//
}




//************************************************
// FUNCTIONS DEFINITION
//************************************************

// Sleep function enters low power mode(LPM3)
//  for 'time' VLO clock cycles.
void sleep(uint16_t time)
{
	TA0CCR0 = time;
	TA0CTL = MC_1+TACLR+TASSEL_1;
	TA0CCTL0 &= ~(CAP+CCIFG);
	TA0CCTL0 |= CCIE;
	__bis_SR_register(LPM3_bits+GIE);
	TA0CTL &= ~(MC_1);
	TA0CCTL0 &= ~(CCIE);
}

// Measuring function
// TI_CAPT_Raw calls actual measuring function according to HAL definition
// In this application, HAL definition is fixed as fRO_PINOSC_TA0_TA1
// TI_CAPT_Raw collects the number of counts from TI_CTS_fRO_PINOSC_TA0_TA1,
//   and first parameter for this function choose the sensor.
// Collected value is saved into the second parameter.
void TI_CAPT_Raw(const struct Sensor* groupOfElements, uint32_t * counts)
{
	if(groupOfElements->halDefinition == fRO_PINOSC_TA0_TA1)
	{
		TI_CTS_fRO_PINOSC_TA0_TA1_HAL(groupOfElements, counts);
	}
}


void TI_CTS_fRO_PINOSC_TA0_TA1_HAL(const struct Sensor *group, uint32_t *counts)
{
	uint8_t i;

//** Context Save
//  Status Register:
//  TIMERA0: TA0CTL, TA0CCTL0
//  TIMERA1: TA1CTL, TA1CCTL0
//  Ports: PxSEL, PxSEL2
	uint8_t contextSaveSR;
	uint16_t contextSaveTA0CTL,contextSaveTA0CCTL0,contextSaveTA0CCR0;
	uint16_t contextSaveTA1CTL,contextSaveTA1CCTL0,contextSaveTA1CCR0;
	uint8_t contextSaveSel,contextSaveSel2;

	contextSaveSR = __get_SR_register();

	contextSaveTA0CTL = TA0CTL;
	contextSaveTA0CCTL0 = TA0CCTL0;
	contextSaveTA0CCR0 = TA0CCR0;

	contextSaveTA1CTL = TA1CTL;
	contextSaveTA1CCTL0 = TA1CCTL0;
	contextSaveTA1CCR0 = TA1CCR0;

//** Setup Measurement timer***************************************************
// Choices are TA0,TA1,TB0,TB1,TD0,TD1 these choices are pushed up into the
// capacitive touch layer.

	count = 0;

	// Configure Measurement interval with TimerA0
	TA0CCR0 = (group->accumulationCycles);
	TA0CTL = TASSEL_3+MC_1;							// INCLK, up mode
	TA0CCTL0 = CCIE;

	// Configure and start measurment timerA1
	TA1CTL = group->measGateSource + MC_2 + TACLR + TAIE;	// cont
	TA1CCTL0 = CM_3+CCIS_2+CAP;						// Pos&Neg,GND,Cap

	for (i = 0; i<(group->numElements); i++)
	{
		// Context Save
		contextSaveSel = *((group->arrayPtr[i])->inputPxselRegister);
		contextSaveSel2 = *((group->arrayPtr[i])->inputPxsel2Register);

		// Configure Ports for relaxation oscillator
		*((group->arrayPtr[i])->inputPxselRegister) &= ~((group->arrayPtr[i])->inputBits);
		*((group->arrayPtr[i])->inputPxsel2Register) |= ((group->arrayPtr[i])->inputBits);

		//**  Setup Gate Timer ********************************************************
		TA1CTL |= TACLR;
		TA0CTL |= TACLR;							// Clear Timer_A TAR
		TA1CTL &= ~TAIFG;
		count = 0;//Clear roll-over counter
		if(group->measGateSource == TIMER_ACLK)
		{
			__bis_SR_register(LPM3_bits+GIE);		// Wait for WDT interrupt
		}
		else
		{
			__bis_SR_register(LPM0_bits+GIE);		// Wait for WDT interrupt
		}
		TA1CCTL0 ^= CCIS0;							// Create SW capture of CCR1
		counts[i] = TA1CCR0;						// Save result
		if(count > 0 )
		{
#ifdef TouchPro
			temp = counts[i];
#endif
			counts[i] = counts[i] + (count * 65535);
		}

		// Context Restore
		*((group->arrayPtr[i])->inputPxselRegister) = contextSaveSel;
		*((group->arrayPtr[i])->inputPxsel2Register) = contextSaveSel2;
	}
    // End Sequence

    // Context Restore
	__bis_SR_register(contextSaveSR);

	TA0CTL = contextSaveTA0CTL;
	TA0CCTL0 = contextSaveTA0CCTL0;
	TA0CCR0 = contextSaveTA0CCR0;

	TA1CTL = contextSaveTA1CTL;
	TA1CCTL0 = contextSaveTA1CCTL0;
	TA1CCR0 = contextSaveTA1CCR0;
}


//************************************************
// INTERRUPT - TIMER
//************************************************
#pragma vector=TIMER0_A0_VECTOR
__interrupt void TIMER0_A0_ISR(void)

{
	__bic_SR_register_on_exit(LPM3_bits);           // Exit LPM3 on reti
}


#ifdef TIMER1_OVERFLOW_COUNT
// Rollover on Timer1 for the fRO method
// TA1 CCR1, CCR2, Overflow
#pragma vector = TIMER1_A1_VECTOR
__interrupt void Timer1_A1 (void)
{
	__no_operation();
	if(TA1IV & TA1IV_TAIFG)
	{
		count++;
	}
}
#endif
