/*******************************************************************************
 *  CTS_SYSBIOS_Layer.c
 *
 *  Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************/
/***************************************************************************//**
 * @file   CTS_SYSBIOS_Layer.c
 *
 * @brief  Update baseline tracking in the event of no detection or interpret
 *         measurement information in the event of a detection.
 *
 * @par    Project:
 *              MSP430 SYSBIOS Capacitive Touch Interface
 *
 * @par    Developed using:
 *              CCS Version : 5.1.0.09000, w/support for GCC extensions (--gcc)
 *              RTSC/XDCtools: 3.22.04.46
 *              SYS/BIOS: 6.33.04.39
 *
 * @version     1.0.0 Initial Release
 *
 * @par    Function Call from SYSBIOS Interface:
 *
 * 				- TI_CTS_SYSBIOS_Post_Process
 *
 * @par    Other Functions
 * 
 *              - TI_CTS_SYSBIOS_Baseline_Update
 *              - TI_CTS_SYSBIOS_Button
 *              - TI_CTS_SYSBIOS_Slider
 *              - TI_CTS_SYSBIOS_Wheel
 *              - TI_CTS_SYSBIOS_Dominant_Element
 *
 ******************************************************************************/

#include "CTS_SYSBIOS_Layer.h"

// Global variables for sensing

uint16_t filter[TOTAL_NUMBER_OF_ELEMENTS][FILTER_SIZE];


/***************************************************************************//**
 * @addtogroup CTS_Layer
 * @{ 
 ******************************************************************************/

/***************************************************************************//**
 * @brief   Measure sensor and update all appropriate registers
 *          Assumption is that calling layer knows which sensors are valid
 *          and will only call valid sensors.
 * 
 * @param   sensor Pointer to Sensor structure to be measured
 * @return  none
 ******************************************************************************/
void TI_CTS_SYSBIOS_Post_Process(RamSensorConfig* sensor)
{
	volatile uint8_t i;
	uint16_t mask;

	mask = BIT0;

    // if no detection occurred then update baseline tracking
	if(!sensor->sRegister2.detect)
	{
	    for(i=0;i<TOTAL_NUMBER_OF_ELEMENTS;i++)
	    {
	    	if(sensor->sRegister0.elementLocation & mask)
	    	{
	            TI_CTS_SYSBIOS_Baseline_Update(&eConfig[i],i);
	    	}
	    	mask = mask << 1;
	    }
	}
	else
	{
		// Re-initialize base update after a detection
	    for(i=0;i<TOTAL_NUMBER_OF_ELEMENTS;i++)
	    {
	    	if(sensor->sRegister0.elementLocation & mask)
	    	{
	    		eConfig[i].eRegister0.baseUpdate = eConfigFlash[i].eRegister0.baseUpdate;
	    	}
	    	mask = mask << 1;
	    }
		// Perform additional measurements/processing
	    // Button/Buttons
		if(sensor->sRegister1.points == 0)
		{
			TI_CTS_SYSBIOS_Button(sensor);
		}
	}
}

/***************************************************************************//**
* @brief   Update Baseline Tracking
*
*          Update the baseline information for a given element.
*
* @param   Pointer to element configuration and the element index
* @return  none
******************************************************************************/
void TI_CTS_SYSBIOS_Baseline_Update(RamElementConfig* element, uint8_t index)
{
	  volatile uint8_t i;
	  // uint8_t peak;
	  uint16_t filteredResponse; // ,delta;
	  uint16_t remainder=0;

	  // does measurement pass through filter?
	  // 000 - No filter
	  // 001 - Average 2 measures
	  // 010 - Average 4 measures
	  // 011 - Limit measure, average 4

	  if(element->eRegister0.baseUpdate == 0)
	  {
		  element->eRegister0.baseUpdate = eConfigFlash[index].eRegister0.baseUpdate;
		  switch(element->eRegister0.filter)
		  {
			case E_FILTER0:
				filteredResponse = element->measure;
				break;
			case E_FILTER1:
				filteredResponse = element->measure/2 + filter[index][0]/2;
				filter[index][0] = element->measure;
				break;
			case E_FILTER2:
				remainder = (0x0003 & element->measure);
				filter[index][3] = element->measure;
				filteredResponse = element->measure/4;
				for(i=0;i<3;i++)
				{
					remainder += (0x0003 & filter[index][i]);
					filteredResponse += filter[index][i]/4;
					filter[index][i] = filter[index][i+1];		// update filter
				}
				filteredResponse += remainder/4;
				break;
			case E_FILTER3:
				// Peak rejection: looking for 'negative' touches caused by noise
				// measurement value is greater than baseline-threshold need to limit
				// filtered response to baseline-threshold/2 to baseline+threshold/2
				if((element->measure > (element->baseline + element->threshold/2))
						|| (element->measure < (element->baseline - element->threshold/2)))
				{
					//limit value
					if(element->measure > (element->baseline + element->threshold/2))
					{
						element->measure = element->baseline + element->threshold/2;
					}
					else if(element->measure < (element->baseline - element->threshold/2))
					{
						element->measure = element->baseline - element->threshold/2;
					}
				}
				remainder = (0x0003 & element->measure);
				filter[index][3] = element->measure;
				filteredResponse = element->measure/4;
				for(i=0;i<3;i++)
				{
					remainder += (0x0003 & filter[index][i]);
					filteredResponse += filter[index][i]/4;
					filter[index][i] = filter[index][i+1];		// update filter
				}
				filteredResponse += remainder/4;
				break;

			default:
				break;
		  }

		  // Drift compensation magnitude
		  // 000 - One for One, filtered response becomes the new baseline
		  // 001 - New baseline is an average of old and new
		  // 010 - New baseline is weighted average; new 2/8, old 6/8
		  // 011 - New baseline is weighted average; new 1/8, old 7/8

		  // 100 - New decreasing baseline is weighted average; new 1/8, old 7/8
		  //       New Increasing baseline is weighted average; new 7/8, old 1/8
		  // 101 - New decreasing baseline is weighted average; new 1/8, old 7/8
		  //       New Increasing baseline is weighted average; new 6/8, old 2/8

		  switch(element->eRegister0.driftComp)
		  {
			case DCOMP0:
				element->baseline = filteredResponse;
				break;
			case DCOMP1:
				element->baseline = 1 + (element->baseline/2) + (filteredResponse/2);
				break;
			case DCOMP2:
				remainder = 3*(0x0003 & element->baseline);
				remainder +=  (0x0003 & filteredResponse);
				remainder = remainder/4;
				element->baseline = 3*(element->baseline/4) + (filteredResponse/4) + remainder;
				break;
			case DCOMP3:
				remainder = 7*(0x0007 & element->baseline);
				remainder += (0x0007 & filteredResponse);
				remainder = remainder/8;
				element->baseline = 7*(element->baseline/8) + (filteredResponse/8) + remainder;
				break;
			case DCOMP4:
				if(filteredResponse < element->baseline)
				{
					remainder = 7*(0x0007 & element->baseline);
					remainder += (0x0007 & filteredResponse);
					remainder = remainder/8;
					element->baseline = 7*(element->baseline/8) + (filteredResponse/8) + remainder;
				}
				else
				{
					remainder = (0x0007 & element->baseline);
					remainder += 7*(0x0007 & filteredResponse);
					remainder = remainder/8;
					element->baseline = (element->baseline/8) + 7*(filteredResponse/8) + remainder;
				}
				break;
			case DCOMP5:
				if(filteredResponse < element->baseline)
				{
					remainder = 7*(0x0007 & element->baseline);
					remainder += (0x0007 & filteredResponse);
					remainder = remainder/8;
					element->baseline = 7*(element->baseline/8) + (filteredResponse/8) + remainder;
				}
				else
				{
					remainder = (0x0003 & element->baseline);
					remainder += 3*(0x0003 & filteredResponse);
					remainder = remainder/4;
					element->baseline = (element->baseline/4) + 3*(filteredResponse/4) + remainder;
				}
				break;
			default:
				break;
		  }
	  }
	  else
	  {
		 element->eRegister0.baseUpdate--;
	  }
	  return;
}

/***************************************************************************//**
 * @brief   Determine if a button is being pressed
 * @param   groupOfElements Pointer to button to be scanned
 * @return  result Indication if button is (1) or is not (0) being pressed
 ******************************************************************************/
void TI_CTS_SYSBIOS_Button(RamSensorConfig* sensor)
{
    // make measurement
    // determine threshold crossing
    // update
    volatile uint8_t i,j,k;
    uint8_t count=0;
    uint16_t mask = BIT0;
    
    // find out number of elements and measure
    for(i=0;i<TOTAL_NUMBER_OF_ELEMENTS;i++)
    {
    	if(sensor->sRegister0.elementLocation & mask)
    	{
    		count++;
    	}
    	mask = mask << 1;
    }
//*****************************************************************************
// Representation
// This only applies if there are multiple buttons
// 0 - Return all keys that are touched (threshold crossing)
// 1 - Return dominant key (threshold crossing by greatest margin)
	if(sensor->sRegister2.detect)
	{
	    switch(sensor->sRegister1.representation)
	    {
			case REPRESENTATION_0:
				// nothing to do, leave detection register as is
				break;
			case REPRESENTATION_1:
				if(count > 1)
				{
				    sensor->sRegister2.detail
				    = TI_CTS_SYSBIOS_Dominant_Element(sensor);
				}
				break;
			default:
				break;
	    }
	}
    return;
}

/***************************************************************************//**
 * @brief   Determine if a button is being pressed
 * @param   groupOfElements Pointer to button to be scanned
 * @return  result Indication if button is (1) or is not (0) being pressed
 ******************************************************************************/
uint16_t TI_CTS_SYSBIOS_Dominant_Element(RamSensorConfig * sensor)
{
	volatile uint8_t i;
	uint16_t currentPercent, dominantMask;
    uint16_t dominantPercent=0;
    uint16_t mask = BIT0;
    
	for(i=0;i<TOTAL_NUMBER_OF_ELEMENTS;i++)
    {
    	if(sensor->sRegister2.detail & mask)
    	{
			 currentPercent = eConfig[i].baseline - eConfig[i].measure;
			 if(currentPercent > eConfig[i].maxResponse)
			 {
			 	 currentPercent = 100;
			 }
			 else
			 {
			     currentPercent = currentPercent - eConfig[i].threshold;
			     currentPercent = currentPercent*100;
			     currentPercent = currentPercent/(eConfig[i].maxResponse - eConfig[i].threshold);
			 }
			 if(currentPercent >= dominantPercent)
			 {
			 	// set this bit and clear others
			 	dominantMask = mask;
			 	dominantPercent = currentPercent;
			 }
    	}
    	mask = mask << 1;
    }
    return dominantMask;
}

/***************************************************************************//**
 * @}
 ******************************************************************************/

/***************************************************************************//**
 * @defgroup CTS_support
 * @ingroup CTS_Layer
 ******************************************************************************/

