/*******************************************************************************
 *  main.c
 *
 *  Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************/

/*
 *  main.c
 *
 *  This project demonstrates capacitive touch functionality with the
 *  MSP430F5529 Experimenter's board.  The capacitive touch buttons are:
 *  CB0,CB1,CB2,CB3,CB4.  Discrete indicators are associated with LEDs :
 *  P1.1,P1.2,P1.3,P1.4,P1.5.  Additional information about each button is
 *  provided on the display.
 *
 *! Revision History
 *! ================
 *! 01-Aug-2012 CTS: Initial Release
 *
 */

#include <xdc/std.h>
#include <xdc/runtime/System.h>
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Clock.h>
#include <ti/sysbios/knl/Semaphore.h>
#include <ti/sysbios/family/msp430/Hwi.h>
#include <ti/sysbios/family/msp430/Timer.h>

#include <LED/led.h>

#include <CTS_SYSBIOS_Interface.h>
#include <structure.h>

#include "HAL_PMM.h"
#include "HAL_UCS.h"
#include "HAL_Board.h"
#include "HAL_Dogs102x6.h"

#define contrastSetpointAddress     0x1880
#define brightnessSetpointAddress   0x1882

void F5529_Brd_Init(void);

/*
 *  ======== main ========
 *  Initialize CTS RAM structures and the F5529 Experimenter's Board.
 */
int main(void)
{ 
	TI_CTS_Copy_Flash_to_RAM();
	F5529_Brd_Init();
    BIOS_start();     /* enable interrupts and start SYS/BIOS */
}

/*
 *  ======== ledFxn ========
 *  Swi function, associated with clock instance, for updating LED and Display.
 *
 *  This function updates the display on the F5529 Experimenter's Board and
 *  calls the function LED_Update which illuminates the appropriate LED
 *  corresponding to a touch.
 */
void F5529_LED_Display_Update(uintptr_t arg)
{
	volatile uint8_t i,j;
	uint16_t place[3];
	uint16_t rotate;
	int16_t temp;

	LED_Update();
	// write to dawgs to display baseline and delta for each pad
	for(i=0;i<5;i++)
	{
		for(j=0;j<3;j++)
		{
			temp = eConfig[i].baseline;
			rotate = 4*j;
			temp = temp >> rotate;
			temp = (0x000F & temp);
			if(temp > 0x09)
			{
				place[j] = 64 + temp - 9;
			}
			else
			{
				place[j] = 48 + temp;
			}
			rotate = 4;
		}

		Dogs102x6_charDraw(i+1, 54, place[0], DOGS102x6_DRAW_NORMAL);
		Dogs102x6_charDraw(i+1, 48, place[1], DOGS102x6_DRAW_NORMAL);
		Dogs102x6_charDraw(i+1, 42, place[2], DOGS102x6_DRAW_NORMAL);

		for(j=0;j<3;j++)
		{
			temp = eConfig[i].baseline - eConfig[i].measure;
			rotate = 4*j;
			temp = temp >> rotate;
			temp = (0x000F & temp);
			if(temp > 0x09)
			{
				place[j] = 64 + temp - 9;
			}
			else
			{
				place[j] = 48 + temp;
			}
			rotate = 4;
		}

		Dogs102x6_charDraw(i+1, 96, place[0], DOGS102x6_DRAW_NORMAL);
		Dogs102x6_charDraw(i+1, 90, place[1], DOGS102x6_DRAW_NORMAL);
		Dogs102x6_charDraw(i+1, 84, place[2], DOGS102x6_DRAW_NORMAL);
	}
}

/*
 *  ======== F5529_Brd_Init ========
 *  Initialize the MSP430F5529 Experimenter's Board
 *
 */
void F5529_Brd_Init(void)
{
    uint8_t contrast = *((uint8_t *)contrastSetpointAddress);
    uint8_t brightness = *((uint8_t *)brightnessSetpointAddress);

    // Basic GPIO initialization
    Board_init();

    // Set Vcore to accomodate for max. allowed system speed
    SetVCore(3);

    // Use 32.768kHz XTAL as reference
    LFXT_Start(XT1DRIVE_0);

    // Set system clock to max (25MHz)
    Init_FLL_Settle(25000, 762);

    // Set up LCD
    Dogs102x6_init();
    Dogs102x6_backlightInit();

    // Contrast not programed in Flash Yet
    if (contrast == 0xFF)
    {
        // Set Default Contrast
        contrast = 11;
    }

    // Brightness not programed in Flash Yet
    if (brightness == 0xFF)
    {
        // Set Default Brightness
        brightness = 11;
    }

    Dogs102x6_setBacklight(brightness);
    Dogs102x6_setContrast(contrast);
    Dogs102x6_clearScreen();

    // Display TI logo
    Dogs102x6_stringDraw(0, 36, "Base", DOGS102x6_DRAW_NORMAL);
    Dogs102x6_stringDraw(0, 72, "Delta", DOGS102x6_DRAW_NORMAL);
    Dogs102x6_stringDraw(1, 0, "PAD1", DOGS102x6_DRAW_NORMAL);
    Dogs102x6_stringDraw(2, 0, "PAD2", DOGS102x6_DRAW_NORMAL);
    Dogs102x6_stringDraw(3, 0, "PAD3", DOGS102x6_DRAW_NORMAL);
    Dogs102x6_stringDraw(4, 0, "PAD4", DOGS102x6_DRAW_NORMAL);
    Dogs102x6_stringDraw(5, 0, "PAD5", DOGS102x6_DRAW_NORMAL);

}
