/*******************************************************************************
 *  structure.c
 *
 *  Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************/
/*******************************************************************************
 * @file   structure.c
 *
 * @brief  Hardware configuration to perform capacitive touch measurements
 *         with the COMPB,TimerA1, and WDTA peripherals.  This is a
 *         configuration that is specific to the MSP430F5529 family where
 *         the CBOUT signal is tied internally to the TA1CLK input at
 *         P1.6.
 *
 * @par    Project:
 *              MSP430 SYSBIOS Capacitive Touch Interface
 *
 * @par    Developed using:
 *              CCS Version : 5.1.0.09000, w/support for GCC extensions (--gcc)
 *              RTSC/XDCtools: 3.22.04.46
 *              SYS/BIOS: 6.33.04.39
 *
 * @version     1.0.0 Initial Release
 *
 * @par    Supported Hardware Configurations:
 *              - TI_CTS_SYSBIOS_RO_COMPB_TA1_WDTA_HAL()
 *
 ******************************************************************************/

#include "structure.h"

PortInfo eport[TOTAL_NUMBER_OF_ELEMENTS] = {
		// P6.0, CB0
		{
              .inputBits = CBIMSEL_0,
		},
		// P6.1, CB1
		{
              .inputBits = CBIMSEL_1,
		},
		// P6.2, CB2
		{
              .inputBits = CBIMSEL_2,
		},
		// P6.3, CB3
		{
              .inputBits = CBIMSEL_3,
		},
		// P6.4, CB4
		{
              .inputBits = CBIMSEL_4,
		}
};

CompbTaxInfo compbTaxConfig =
{
		      // Disable the digital IO on the corresponding CBx bits
		      // BIT0->CB0-> P6.0 on the F5529
		      // BIT4->CB4-> P6.4 on the F5529
		      // ...
		      // BIT8->CB8-> P7.0 on the F5529
		      // Please note that CB8,9,10, and 11 are not available on all
		      // package types.
              .cbpdBits = (uint16_t)(BIT0+BIT1+BIT2+BIT3+BIT4),
              // CBOUT/TA1CLK is found on P1.6
              // CBOUT/TA1CLK can also be port mapped to port4 (P4.0-P4.7) but
              // this is beyond the scope of this application.
              .cboutTAxDirRegister = (uint8_t *)&P1DIR,  // PxDIR
              .cboutTAxSelRegister = (uint8_t *)&P1SEL,  // PxSEL
              .cboutTAxBits = BIT6, // P1.6
};

//*****************************************************************************
//*****************************************************************************
// Variable Parameters, placed in information memory
//
//
// P6.0, CB0
FlashElementConfig eConfigFlash[TOTAL_NUMBER_OF_ELEMENTS] = {
	{
			  // There are three fields in element register 0
			  // Filter -> EREGISTER0_FILTER_x where x is the desired filter
			  // 			setting.
			  // driftComp -> EREGISTER0_DRIFT_x where x is the desired drift
			  //			compensation.
			  // measureTime -> the measureTime field directly correlates to
			  // 			the WDTA register WDTCTL with the exception that
			  //            the measureTime field does not include the upper
			  //            byte for the WDTA password.  For the purposes of
			  //  			this application the two bit fields written to are
			  //			the WDTSSEL and WDTIS fields; see SLAU208.
			  .eRegister0 = (EREGISTER0_BASEUP_7+EREGISTER0_FILTER_3
					        +EREGISTER0_DRIFT_4+SOURCE_ACLK+SOURCE_DIVIDE_64),
			  // The threshold, baseline, and maxResponse values have the same
			  // meaning from the CTS library.  Please refer to SLAA490.
              .threshold = 100,
              .baseline = 350,
              // The maxResponse value is not used in this application.
              .maxResponse = 255
},

// P6.1, CB1
{
	    	  .eRegister0 = (EREGISTER0_BASEUP_7+EREGISTER0_FILTER_3
	    			        +EREGISTER0_DRIFT_4+SOURCE_ACLK+SOURCE_DIVIDE_64),
              .threshold = 100,
              .baseline = 395,
              .maxResponse = 290
},
// P6.2, CB2
{
		      .eRegister0 = (EREGISTER0_BASEUP_7+EREGISTER0_FILTER_3
		    		        +EREGISTER0_DRIFT_4+SOURCE_ACLK+SOURCE_DIVIDE_64),
              .threshold = 100,
              .baseline = 415,
              .maxResponse = 320
},
// P6.3, CB3
{
		      .eRegister0 = (EREGISTER0_BASEUP_7+EREGISTER0_FILTER_3
		    		        +EREGISTER0_DRIFT_4+SOURCE_ACLK+SOURCE_DIVIDE_64),
              .threshold = 100,
              .baseline = 405,
              .maxResponse = 315
},
// P6.4, CB4
{
		      .eRegister0 = (EREGISTER0_BASEUP_7+EREGISTER0_FILTER_3
		    		        +EREGISTER0_DRIFT_4+SOURCE_ACLK+SOURCE_DIVIDE_64),
              .threshold = 110,
              .baseline = 350,
              .maxResponse = 275
}
};

//*****************************************************************************
//*****************************************************************************
// Sensor Information in Flash

const FlashSensorConfig sConfigFlash[TOTAL_NUMBER_OF_SENSORS] = {
{
		      // Sensor0
		      // register0 contains the value of the number of elements and
		      // the bit location of those elements.  With the F5529
		      // experimenter's board there are 5 elements, bits0-4.
			  .sRegister0 = ((NUM_OF_ELEMENTS_5)
					  +(EPORT4+EPORT3+EPORT2+EPORT1+EPORT0)),
			  // register1 contains the following fields
			  // points -> the 8 LSBs represent the position or number of
			  //           points along a wheel or slider.  If points is '0'
			  //           this indicates that the sensor is a button(s).
			  // representation -> the representation applies to a group of
			  //                   buttons.  Please refer to the application
			  //                   note for a description.  This is also used
			  //                   to determine a wheel or slider
			  //                   implementation for a non-zero points value.
			  //                   REPRESENTATION_0 -> Wheel if points > 0
			  //                   REPRESENTATION_1 -> Slider if points > 0
			  // deglitch -> the deglitch or 'debounce' function sets the
			  //             number of confirmation measurements are needed
			  //             to declare a detection.
			  // scanRate -> this field is currently not used
              .sRegister1 = ((0)+REPRESENTATION_0+DEGLITCH_2)
}
//,{
//              // Sensor1
//			    .sRegister0 = ((NUM_OF_ELEMENTS_1)+(EPORT5)),
//              .sRegister1 = (REPRESENTATION_0+DEGLITCH_2)
//}
};

//*****************************************************************************
//*****************************************************************************
// Variable Paramters, placed in RAM

RamElementConfig eConfig[TOTAL_NUMBER_OF_ELEMENTS];

//*****************************************************************************
//*****************************************************************************
// Sensor Information in RAM

RamSensorConfig sConfig[TOTAL_NUMBER_OF_SENSORS];


