/*******************************************************************************
 *  structure.h
 *
 *  Copyright (C) 2012 Texas Instruments Incorporated - http://www.ti.com/
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************/
/***************************************************************************//**
 * @file   structure.h
 *
 * @brief  Definition of the number of sensors, elements, and filter size of
 *         each element.
 *
 * @par    Project:
 *              MSP430 SYSBIOS Capacitive Touch Interface
 *
 * @par    Developed using:
 *              CCS Version : 5.1.0.09000, w/support for GCC extensions (--gcc)
 *              RTSC/XDCtools: 3.22.04.46
 *              SYS/BIOS: 6.33.04.39
 *
 * @version     1.0.0 Initial Release
 *
 ******************************************************************************/
#ifndef CTS_STRUCTURE
#define CTS_STRUCTURE

#include "msp430.h"
#include <stdint.h>

//****** RAM ALLOCATION ********************************************************
#define TOTAL_NUMBER_OF_SENSORS		1
#define TOTAL_NUMBER_OF_ELEMENTS 	5
#define FILTER_SIZE					4

//******************************************************************************
// End of user configuration section.
//******************************************************************************
//******************************************************************************
//******************************************************************************

#define RO_COMPB_TA1_WDTA         69


// EPORT Definitions
#define EPORT0	BIT0
#define EPORT1	BIT1
#define EPORT2	BIT2
#define EPORT3	BIT3
#define EPORT4	BIT4
#define EPORT5	BIT5
#define EPORT6	BIT6
#define EPORT7	BIT7
#define EPORT8	BIT8
#define EPORT9	BIT9

// Element Register 0, Filter definitions
#define EREGISTER0_FILTER_0	0x0000
#define EREGISTER0_FILTER_1	0x0400
#define EREGISTER0_FILTER_2	0x0800
#define EREGISTER0_FILTER_3	0x0C00
#define EREGISTER0_FILTER_4	0x1000
#define EREGISTER0_FILTER_5	0x1400
#define EREGISTER0_FILTER_6	0x1800
#define EREGISTER0_FILTER_7	0x1C00

// Representation for decoding which filter to apply
#define E_FILTER0			0
#define E_FILTER1			1
#define E_FILTER2			2
#define E_FILTER3			3
#define E_FILTER4			4
#define E_FILTER5			5
#define E_FILTER6			6
#define E_FILTER7			7

#define EREGISTER0_DRIFT_0		0x0000
#define EREGISTER0_DRIFT_1		0x0080
#define EREGISTER0_DRIFT_2		0x0100
#define EREGISTER0_DRIFT_3		0x0180
#define EREGISTER0_DRIFT_4		0x0200
#define EREGISTER0_DRIFT_5		0x0280
#define EREGISTER0_DRIFT_6		0x0300
#define EREGISTER0_DRIFT_7		0x0380

#define EREGISTER0_BASEUP_0	0x0000           // update 1:1 with measurement rate
#define EREGISTER0_BASEUP_1	0x2000           //
#define EREGISTER0_BASEUP_2	0x4000           //
#define EREGISTER0_BASEUP_3	0x6000           //
#define EREGISTER0_BASEUP_7	0xE000           //

#define E_BASEUP0			0
#define E_BASEUP1			1
#define E_BASEUP2			2
#define E_BASEUP3			3
#define E_BASEUP4			4
#define E_BASEUP5			5
#define E_BASEUP6			6
#define E_BASEUP7			7

// Representation for decode
#define DCOMP0			0
#define DCOMP1			1
#define DCOMP2			2
#define DCOMP3			3
#define DCOMP4			4
#define DCOMP5			5
#define DCOMP6			6
#define DCOMP7			7

#define SOURCE_SMCLK		WDTSSEL_0
#define SOURCE_ACLK			WDTSSEL_1
#define SOURCE_VLO			WDTSSEL_2

#define SOURCE_DIVIDE_2G	WDTIS_0
#define SOURCE_DIVIDE_128M	WDTIS_1
#define SOURCE_DIVIDE_8192K	WDTIS_2
#define SOURCE_DIVIDE_512K	WDTIS_3
#define SOURCE_DIVIDE_32K	WDTIS_4
#define SOURCE_DIVIDE_8192	WDTIS_5
#define SOURCE_DIVIDE_512	WDTIS_6
#define SOURCE_DIVIDE_64	WDTIS_7

// Sensor Register 0
#define ELEMENT_0			BIT0
#define ELEMENT_1			BIT1
#define ELEMENT_2			BIT2
#define ELEMENT_3			BIT3
#define ELEMENT_4			BIT4
#define ELEMENT_5			BIT5
#define ELEMENT_6			BIT6
#define ELEMENT_7			BIT7
#define ELEMENT_8			BIT8
#define ELEMENT_9			BIT9

#define NUM_OF_ELEMENTS_1	0x0400
#define NUM_OF_ELEMENTS_2	0x0800
#define NUM_OF_ELEMENTS_3	0x0C00
#define NUM_OF_ELEMENTS_4	0x1000
#define NUM_OF_ELEMENTS_5	0x1400
#define NUM_OF_ELEMENTS_6	0x1800
#define NUM_OF_ELEMENTS_7	0x1C00
#define NUM_OF_ELEMENTS_8	0x2000
#define NUM_OF_ELEMENTS_9	0x2400
#define NUM_OF_ELEMENTS_A	0x2800

// Sensor Register 1
#define REPRESENTATION0		0x0100
#define REPRESENTATION1		0x0200

#define REPRESENTATION_0	0
#define REPRESENTATION_1	REPRESENTATION0
#define REPRESENTATION_2	REPRESENTATION1
#define REPRESENTATION_3	(REPRESENTATION0+REPRESENTATION1)

#define DEGLITCH0			0x0400
#define DEGLITCH1			0x0800

#define DEGLITCH_0			0
#define DEGLITCH_1			DEGLITCH0
#define DEGLITCH_2          DEGLITCH1
#define DEGLITCH_3          (DEGLITCH0+DEGLITCH1)

//******************************************************************************
// The sensor structure identifies port or comparator input definitions for each
// sensor.
//******************************************************************************

typedef const struct
{
    uint16_t inputBits; 	// CBCTL0
}PortInfo;

typedef const struct
{
    uint16_t cbpdBits;
    uint8_t *cboutTAxDirRegister;   // CBOUT_TA0CLK 
    uint8_t *cboutTAxSelRegister;   // CBOUT_TA0CLK
    uint8_t cboutTAxBits;           // Bit field for register  
}CompbTaxInfo;

union elementRegister0
{
	uint16_t eRegister0;
	struct
	{
	    uint16_t measureTime	:7;  // WDTSSEL, WDTTMSEL, WDTIS
		uint16_t driftComp		:3;
		uint16_t filter  		:3;
		uint16_t baseUpdate		:3;
	};
};

typedef const struct
{
	union elementRegister0 		eRegister0;
	uint16_t 					threshold;
	uint16_t 					maxResponse;
	uint16_t 					baseline;
}FlashElementConfig;

typedef struct
{
	PortInfo*					portInfo; 
	union elementRegister0 		eRegister0;
	uint16_t 					threshold;
	uint16_t 					maxResponse;
    uint16_t 					measure;
	uint16_t 					baseline;
}RamElementConfig;

union sensorRegister0
{
	uint16_t sRegister0;
	struct
	{
		uint16_t elementLocation	:10;  // LSB
		uint16_t numberOfElements	:4;
		uint16_t sensitivity		:2;   //
	};
};

union sensorRegister1
{
	uint16_t sRegister1;
	struct
	{
		uint16_t points  			:8;  // 0 represents a button
		uint16_t representation		:2;  // non-zero points, 0 - wheel, 1 - slider
		uint16_t deglitch 			:2;
		uint16_t scanRate			:4;  // TBD
	};
};

union sensorRegister2
{
	uint16_t sRegister2;
	struct
	{
		uint16_t detail  			:15;  // position or button
		uint16_t detect				:1;
	};
};

typedef struct
{
	union sensorRegister0 sRegister0;
	union sensorRegister1 sRegister1;
} FlashSensorConfig;

typedef struct
{
	union sensorRegister0 sRegister0;
	union sensorRegister1 sRegister1;
	union sensorRegister2 sRegister2;
} RamSensorConfig;

/* Public Globals */
extern PortInfo eport[TOTAL_NUMBER_OF_ELEMENTS];
extern CompbTaxInfo compbTaxConfig;
extern const FlashElementConfig eConfigFlash[TOTAL_NUMBER_OF_ELEMENTS];
extern const FlashSensorConfig sConfigFlash[TOTAL_NUMBER_OF_SENSORS];
extern RamElementConfig eConfig[TOTAL_NUMBER_OF_ELEMENTS];
extern RamSensorConfig sConfig[TOTAL_NUMBER_OF_SENSORS];

#endif


