/* --COPYRIGHT--,BSD
 * Copyright (c) 2014, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//****************************************************************************
//  MSP430FR59xx EnergyTrace Demo- Low Energy Consumption Code
//
//   Description: This code has been optimized with the help of EnergyTrace
//   for minimal energy consumption.
//
//   About every second, an ADC temperature sample is taken and the degrees
//   Celsius and Fahrenheit are calculated. The results are transmitted
//   through the UART.
//   The inefficient coding styles have been fixed in the following ways:
//      sprintf()----------------------> Removed. Replaced with manual calculation.
//      Floating point operations------> Truncate with integer calculations
//      Divide operations--------------> Only multiply, add, and shift if possible.
//      Flag polling-------------------> Use of interrupts wherever possible
//      Software delays----------------> Timer
//      Floating port pins-------------> Unused pins are output low
//      No use of low-power modes------> Uses LPM
//      Counting up in loops-----------> Counts down in loops
//
//   B. Finch
//   Texas Instruments Inc.
//   June 2013
//   Built with Code Composer Studio V5.5.0.00039
//****************************************************************************

#include <msp430.h>
#include <stdio.h>
#include <stdint.h>

#define CAL_ADC_12T30  (*((uint16_t *)0x1A1A)) // Temperature Sensor Calibration-30 C 1.2V ref
#define CAL_ADC_12T85  (*((uint16_t *)0x1A1C)) // Temperature Sensor Calibration-85 C 1.2V ref

void UART_print(char *string, char type);

/* The function below assumes that the value to be converted is ten times the desired value. As a
 * result, the accuracy can be to the tenths place (despite the absence of floating point
 * variables). This implementation uses built-in functions, rather than "divide" and "modulo".
 */
char* rawToAsciiString(int16_t input);

char byte = 0;
int main(void) {

    volatile int16_t temp;
    volatile int32_t IntDegF;
    volatile int32_t IntDegC;

    WDTCTL = WDTPW | WDTHOLD;                 // Stop watchdog timer

    // Configure Port Pins as Output Low. Clear all port interrupt flags.
    PAOUT = 0; PBOUT = 0; PJOUT = 0;
    PADIR = 0xFFFF; PBDIR = 0xFFFF; PJDIR = 0xFF;
    PAIFG = 0; PBIFG = 0;

    // Configure used port pins
    P2SEL1 |= BIT1 | BIT0;                    // Configure UART pins
    P2SEL0 &= ~(BIT1 | BIT0);                 // Configure UART pins
    PJSEL0 |= BIT4 | BIT5;                    // XT1 Setup
    PM5CTL0 &= ~LOCKLPM5;                     // Disable GPIO power-on default high-impedance mode
                                              // to activate previously configured port settings
    // Configure Clock System
    CSCTL0_H = 0xA5;                          // CS password
    CSCTL2 = SELA__LFXTCLK;                   // ACLK sourced from LFXT
    CSCTL3 = DIVA__1;                         // No division
    CSCTL4 |= LFXTDRIVE_3 | SMCLKOFF | VLOOFF; // Highest crystal drive setting. MAY CHANGEEEEE SMCLK turned off.
    CSCTL4 &= ~LFXTOFF;                       // Turn on LFXT

    do
    {
      CSCTL5 &= ~(LFXTOFFG | HFXTOFFG);       // Clear XT1 fault flag
      SFRIFG1 &= ~OFIFG;
    }while (SFRIFG1 & OFIFG);                 // Test oscillator fault flag

    // Configure ADC12
    ADC12CTL0 = ADC12SHT0_2 | ADC12ON;        // 16 ADC12CLKs; ADC ON
    ADC12CTL1 = ADC12SHP | ADC12SSEL_1 | ADC12CONSEQ_0; // s/w trigger, ACLK, single ch/conv
    ADC12CTL2 = ADC12RES__12BIT;              // 12-bit conversion results.
    ADC12CTL3 = ADC12TCMAP;                   // temp sensor selected for ADC input channel A30
    ADC12MCTL0 = ADC12VRSEL_1 | ADC12INCH_30; // ADC input ch A30 => temp sense
    ADC12IER0 |= ADC12IE0;                    // Enable A30 interrupt

    // Configure internal reference
    while(REFCTL0 & REFGENBUSY);              // If ref generator busy, WAIT
    REFCTL0 |= REFVSEL_0 | REFGENOT | REFON;  // Select internal ref = 1.2V, reference on

    // Configure Timer
    TA0CTL = TASSEL__ACLK | MC__UP | TACLR;   // ACLK, up mode, clear timer.
    TA0CCR0 = 131;                            // ~0.4ms
    TA0CCTL0 |= CCIE;                         // Capture/compare interrupt enable.
    __bis_SR_register(LPM3_bits | GIE);       // Enter LPM3. Delay for Ref to settle.
    TA0CCR0 = 0x8000;                         // Change timer delay to  ~1 sec.

    // Configure UART
    UCA0CTLW0 |= UCSSEL__ACLK | UCSWRST;      // No parity, LSB first, 8-bit data, 1 stop
    UCA0BRW = 3;                              // Baud rate register prescale. Configure  9600 baud
    UCA0MCTLW = 0x9200;                       // UCBRS = 0x92, UCBRF = -- (don't care); UCOS16 = 0
    UCA0CTLW0 &= ~UCSWRST;                    // Enable eUSCI_A

    __no_operation();                         // SET BREAKPOINT HERE

    while(1)
    {
        __bis_SR_register(LPM3_bits | GIE);   // Enter LPM3, wait for ~1sec timer

        ADC12CTL0 |= ADC12ENC | ADC12SC;      // Sampling and conversion start
        __bis_SR_register(LPM3_bits | GIE);   // Wait for conversion to complete
        temp = ADC12MEM0;                     // 'temp' = the raw ADC temperature conversion result
        __bic_SR_register(GIE);

        // Temperature in Celsius, multiplied by 10:
        IntDegC = ((temp - CAL_ADC_12T30)*10*(85-30)*10/((CAL_ADC_12T85-CAL_ADC_12T30)*10) + 30*10);

        // Temperature in Fahrenheit, multiplied by 10:
        IntDegF = 9*IntDegC/5+320;

        UART_print(rawToAsciiString(IntDegC), 'C'); // Send temperature information through UART
        UART_print(rawToAsciiString(IntDegF), 'F');
    }
}

char* rawToAsciiString(int16_t input)         //conversion algorithm which used built-in functions
{                                             // (rather than "divide" and "modulo")
    uint16_t i;
    uint16_t bcd;
    static char result[6];

    if (input < 0) input = -input;

    for (i = 16, bcd = 0; i; i--)
    {
        bcd = __bcd_add_short(bcd, bcd);
        if (input & 0x8000)
            bcd = __bcd_add_short(bcd,1);
        input <<= 1;
    }

    for (i = 4; i > 0; i--)
    {
        result[i-1] = 0x30 | ((bcd>>((4-i)*4))&0xF);
    }
    result[4] = result[3];                    // Move 10ths place
    result[3] = '.';                          // Insert decimal point

    return result;
}

void UART_print(char *string, char type)      // Send a zero-terminated string through the UART
{
    byte = *string++;
    char count = 0;
    const char cel[] = " Degrees Celsius\r\n";
    const char fah[] = " Degrees Fahrenheit\r\n";

    while (byte == 0x30)                      // Don't print leading zeros.
    {
        byte = *string++;
        count++;
    }

    while(count != 5)
    {
        UCA0IE |= UCTXIE;                     // Enable UART TX interrupt
        __bis_SR_register(LPM3_bits | GIE);   // Wait until TX buffer ready
        byte = *string++;
        count++;
    }
    count = 0;
    if (type == 'C')
    {
        while(cel[count] != 0)
        {
            byte = cel[count++];              // Send the next byte of info
            UCA0IE |= UCTXIE;                 // Enable UART TX interrupt
            __bis_SR_register(LPM3_bits | GIE); // Wait until TX buffer ready
        }
    }
    else if (type == 'F')
    {
        while(fah[count] != 0)
        {
            byte = fah[count++];              // Send the next byte of info
            UCA0IE |= UCTXIE;                 // Enable UART TX interrupt
            __bis_SR_register(LPM3_bits | GIE); // Wait until TX buffer ready
        }
    }
    UCA0IE &= ~UCTXIE;
}

#pragma vector = ADC12_VECTOR
__interrupt void ADC12_ISR(void)
{
  switch(__even_in_range(ADC12IV,76))
  {
    case  0: break;                           // Vector  0:  No interrupt
    case  2: break;                           // Vector  2:  ADC12MEMx Overflow
    case  4: break;                           // Vector  4:  Conversion time overflow
    case  6: break;                           // Vector  6:  ADC12HI
    case  8: break;                           // Vector  8:  ADC12LO
    case 10: break;                           // Vector 10:  ADC12IN
    case 12:                                  // Vector 12:  ADC12MEM0
        ADC12IFGR0 &= ~ADC12IFG0;             // Clear interrupt flag
        __bic_SR_register_on_exit(LPM3_bits); // Exit active CPU
        break;
    case 14: break;                           // Vector 14:  ADC12MEM1
    case 16: break;                           // Vector 16:  ADC12MEM2
    case 18: break;                           // Vector 18:  ADC12MEM3
    case 20: break;                           // Vector 20:  ADC12MEM4
    case 22: break;                           // Vector 22:  ADC12MEM5
    case 24: break;                           // Vector 24:  ADC12MEM6
    case 26: break;                           // Vector 26:  ADC12MEM7
    case 28: break;                           // Vector 28:  ADC12MEM8
    case 30: break;                           // Vector 30:  ADC12MEM9
    case 32: break;                           // Vector 32:  ADC12MEM10
    case 34: break;                           // Vector 34:  ADC12MEM11
    case 36: break;                           // Vector 36:  ADC12MEM12
    case 38: break;                           // Vector 38:  ADC12MEM13
    case 40: break;                           // Vector 40:  ADC12MEM14
    case 42: break;                           // Vector 42:  ADC12MEM15
    case 44: break;                           // Vector 44:  ADC12MEM16
    case 46: break;                           // Vector 46:  ADC12MEM17
    case 48: break;                           // Vector 48:  ADC12MEM18
    case 50: break;                           // Vector 50:  ADC12MEM19
    case 52: break;                           // Vector 52:  ADC12MEM20
    case 54: break;                           // Vector 54:  ADC12MEM21
    case 56: break;                           // Vector 56:  ADC12MEM22
    case 58: break;                           // Vector 58:  ADC12MEM23
    case 60: break;                           // Vector 60:  ADC12MEM24
    case 62: break;                           // Vector 62:  ADC12MEM25
    case 64: break;                           // Vector 64:  ADC12MEM26
    case 66: break;                           // Vector 66:  ADC12MEM27
    case 68: break;                           // Vector 68:  ADC12MEM28
    case 70: break;                           // Vector 70:  ADC12MEM29
    case 72: break;                           // Vector 72:  ADC12MEM30
    case 74: break;                           // Vector 74:  ADC12MEM31
    case 76: break;                           // Vector 76:  ADC12RDY
    default: break;
  }
}

// Timer0_A3 Interrupt Vector (TAIV) handler
#pragma vector=TIMER0_A0_VECTOR
__interrupt void TIMER0_A0_ISR(void)
{
    __bic_SR_register_on_exit(LPM3_bits);     // Exit active CPU
}

#pragma vector=USCI_A0_VECTOR
__interrupt void USCI_A0_ISR(void)
{
  switch(__even_in_range(UCA0IV,USCI_UART_UCTXCPTIFG))
  {
    case USCI_NONE: break;
    case USCI_UART_UCRXIFG: break;
    case USCI_UART_UCTXIFG:
        UCA0TXBUF = byte;                     // Send the next byte of info
        UCA0IE &= ~UCTXIE;                    // Disable UART TX interrupt
        __bic_SR_register_on_exit(LPM3_bits); // Exit active CPU
        break;
    case USCI_UART_UCSTTIFG: break;
    case USCI_UART_UCTXCPTIFG: break;
  }
}
