/* --COPYRIGHT--,BSD
 * Copyright (c) 2014, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//****************************************************************************
//  MSP430FR59xx EnergyTrace Demo- High Energy Consumption Code
//
//   Description: This code is intentionally written inefficiently in order
//   to use an unnecessary amount of energy. The ULP Advisor and EnergyTrace
//   are used to help identify the problem areas in the code to point out
//   where changes can be made to increase efficiency.
//
//   About every second, an ADC temperature sample is taken and the degrees
//   Celsius and Fahrenheit are found using floating point calculations.
//   The results are printed and transmitted through the UART.
//   Some examples of the inefficient coding styles used are:
//      sprintf()
//      Floating point operations
//      Divide operations
//      Flag polling
//      Software delays
//      Floating port pins
//      No use of low-power modes
//      Counting up in loops
//
//   B. Finch
//   Texas Instruments Inc.
//   June 2013
//   Built with Code Composer Studio V5.5.0.00039
//****************************************************************************

#include <stdio.h>
#include <msp430.h> 
#include <stdint.h>

#define CAL_ADC_12T30  (*((uint16_t *)0x1A1A)) // Temperature Sensor Calibration-30 C 1.2V ref
#define CAL_ADC_12T85  (*((uint16_t *)0x1A1C)) // Temperature Sensor Calibration-85 C 1.2V ref

void UART_print(char *string);

int main(void) {
    char resultC[50];
    char resultF[50];
    float temp;
    float IntDegF;
    float IntDegC;

    WDTCTL = WDTPW | WDTHOLD;                 // Stop watchdog timer

    // Configure used port pins
    P2SEL1 |= BIT1 | BIT0;                    // Configure UART pins
    P2SEL0 &= ~(BIT1 | BIT0);                 // Configure UART pins
    PM5CTL0 &= ~LOCKLPM5;                     // Disable GPIO power-on default high-impedance mode
                                              // to activate previously configured port settings
    // Configure ADC12
    ADC12CTL0 = ADC12SHT0_2 | ADC12ON;        // 16 ADC12CLKs, ADC ON
    ADC12CTL1 = ADC12SHP | ADC12SSEL_2 | ADC12CONSEQ_0; // s/w trigger, MCLK, single ch/conv
    ADC12CTL2 = ADC12RES__12BIT;              // 12-bit conversion results
    ADC12CTL3 = ADC12TCMAP;                   // temp sensor selected for ADC input channel A30
    ADC12MCTL0 = ADC12VRSEL_1 | ADC12INCH_30; // ADC input ch A30 => temp sense

    // Configure internal reference
    while(REFCTL0 & REFGENBUSY);              // If ref generator busy, WAIT
    REFCTL0 |= REFVSEL_0 | REFGENOT | REFON;  // Select internal ref = 1.2V, reference on
    __delay_cycles(400);                      // Delay for Ref to settle

    // Configure 1sec Timer
    TA0CTL = TASSEL__SMCLK | ID__8 | MC__UP | TACLR | TAIE; // SMCLK / 8, up mode, clear timer
    TA0EX0 = TAIDEX_7;                        // (SMCLK / 8) / 8 ~ 15.625 kHz. Default SMCLK: 1MHz
    TA0CCR0 = 0x3D09;                         // ~1 sec

    // Configure UART
    UCA0CTLW0 |= UCSSEL__SMCLK | UCSWRST;     // No parity, LSB first, 8-bit data, 1 stop
    UCA0BRW = 6;                              // Baud rate register prescale. Configure 9600 baud
    UCA0MCTLW |= 0x2081;                      // UCBRS = 0x20, UCBRF = 8; UCOS16 = 1
    UCA0CTLW0 &= ~UCSWRST;                    // Enable eUSCI_A

    __no_operation();                         // SET A BREAKPOINT HERE

    while(1)
    {
        if(TA0IV == TA0IV_TAIFG)              // Poll the timer overflow interrupt status
        {
            ADC12CTL0 |= ADC12ENC | ADC12SC;  // Sampling and conversion start
            while(ADC12IFGR1 & ~ADC12IFG30);  // Wait for the conversion to complete
            while(ADC12CTL1 & ADC12BUSY);
            temp = ADC12MEM0;

            // Temperature in Celsius:
            IntDegC = (temp - CAL_ADC_12T30) * (85.0-30.0)/(CAL_ADC_12T85-CAL_ADC_12T30) + 30.0;

            // Temperature in Fahrenheit:
            IntDegF = 9.0*IntDegC/5.0+32.0;

            sprintf(resultC, "%.1f Degrees Celsius\r\n", IntDegC);
            sprintf(resultF, "%.1f Degrees Fahrenheit\r\n", IntDegF);

            UART_print(resultC);              // Send the temperature information through the
            UART_print(resultF);              // backchannel UART

            while(UCA0STATW & UCBUSY);
            __no_operation();                 // For debugger
        }
    }
}

void UART_print(char *string)                 // Send a zero-terminated string through the UART
{
    char byte = *string++;
    while(byte != 0)
    {
        while(!(UCA0IFG & UCTXIFG));          // Wait until TX buffer ready
        UCA0TXBUF = byte;                     // Send the next byte of info
        byte = *string++;                     // Get the next character to send
    }
}
