/* --COPYRIGHT--,BSD
 * Copyright (c) 2014, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//****************************************************************************
//  MSP430FR59xx EnergyTrace Demo- Low Energy Consumption Code
//
//   Description: This code has been improved with the help of the ULP Advisor
//   and EnergyTrace.
//
//   About every second, an ADC temperature sample is taken and the degrees
//   Celsius and Fahrenheit are calculated. The results are transmitted
//   through the UART.
//   The inefficient coding styles have been improved in the following ways:
//      sprintf()----------------------> Removed. Replaced with manual calculation.
//      Floating point operations------> Truncate with integer calculations
//      Divide operations
//      Flag polling-------------------> Use of some interrupts
//      Software delays----------------> Timer
//      Floating port pins-------------> Unused pins are output low
//      No use of low-power modes------> Uses LPM
//      Counting up in loops
//
//   B. Finch
//   Texas Instruments Inc.
//   June 2013
//   Built with Code Composer Studio V5.5.0.00039
//****************************************************************************

#include <msp430.h>
#include <stdio.h>
#include <stdint.h>

#define CAL_ADC_12T30  (*((uint16_t *)0x1A1A)) // Temperature Sensor Calibration-30 C 1.2V ref
#define CAL_ADC_12T85  (*((uint16_t *)0x1A1C)) // Temperature Sensor Calibration-85 C 1.2V ref

void UART_print(char *string, char type);

/* The function below assumes that the value to be converted is ten times the desired value. As a
 * result, the accuracy can be to the tenths place (despite the absence of floating point
 * variables).
 */
char* rawToAsciiString(int32_t input);

int main(void) {

    volatile int16_t temp;
    volatile int32_t IntDegF;
    volatile int32_t IntDegC;

    WDTCTL = WDTPW | WDTHOLD;                 // Stop watchdog timer

    // Configure all port pins as output low
    PAOUT = 0; PBOUT = 0; PJOUT = 0;
    PADIR = 0xFFFF; PBDIR = 0xFFFF; PJDIR = 0xFF;
    PAIFG = 0; PBIFG = 0;

    // Configure used port pins
    P2SEL1 |= BIT1 | BIT0;                    // Configure UART pins
    P2SEL0 &= ~(BIT1 | BIT0);                 // Configure UART pins
    PM5CTL0 &= ~LOCKLPM5;                     // Disable GPIO power-on default high-impedance mode
                                              // to activate previously configured port settings
    // Configure ADC12
    ADC12CTL0 = ADC12SHT0_2 | ADC12ON;        // 16 ADC12CLKs; ADC ON
    ADC12CTL1 = ADC12SHP | ADC12SSEL_3 | ADC12CONSEQ_0; // s/w trigger, SMCLK, single ch/conv
    ADC12CTL2 = ADC12RES__12BIT;              // 12-bit conversion results.
    ADC12CTL3 = ADC12TCMAP;                   // temp sensor selected for ADC input channel A30
    ADC12MCTL0 = ADC12VRSEL_1 | ADC12INCH_30; // ADC input ch A30 => temp sense
    ADC12IER0 |= ADC12IE0;                    // Enable A30 interrupt

    // Configure internal reference
    while(REFCTL0 & REFGENBUSY);              // If ref generator busy, WAIT
    REFCTL0 |= REFVSEL_0 | REFGENOT | REFON;  // Select internal ref = 1.2V, reference on

    // Configure Timer
    TA0CTL = TASSEL__SMCLK | ID__8 | MC__UP | TACLR; // SMCLK / 8, up mode, clear timer.
    TA0EX0 = TAIDEX_7;                        // (SMCLK / 8) / 8 ~ 15.625 kHz.  Default SMCLK: 1MHz
    TA0CCR0 = 6;                              // ~0.4ms
    TA0CCTL0 |= CCIE;                         // Capture/compare interrupt enable.
    __bis_SR_register(LPM3_bits | GIE);       // Enter LPM3. Delay for Ref to settle.
    TA0CCR0 = 0x3D09;                         // Change timer delay to  ~1 sec.

    // Configure UART
    UCA0CTLW0 |= UCSSEL__SMCLK | UCSWRST;     // No parity, LSB first, 8-bit data, 1 stop
    UCA0BRW = 6;                              // Baud rate register prescale. Configure  9600 baud
    UCA0MCTLW |= 0x2081;                      // UCBRS = 0x20, UCBRF = 8; UCOS16 = 1
    UCA0CTLW0 &= ~UCSWRST;                    // Enable eUSCI_A

    __no_operation();                         // SET BREAKPOINT HERE

    while(1)
    {
        __bis_SR_register(LPM3_bits | GIE);   // Enter LPM3, wait for ~1sec timer

        ADC12CTL0 |= ADC12ENC | ADC12SC;      // Sampling and conversion start
        __bis_SR_register(LPM3_bits | GIE);   // Wait for conversion to complete
        temp = ADC12MEM0;                     // 'temp' = the raw ADC temperature conversion result
        __bic_SR_register(GIE);

        // Temperature in Celsius, multiplied by 10:
        IntDegC = ((temp - CAL_ADC_12T30)*10*(85-30)*10/((CAL_ADC_12T85-CAL_ADC_12T30)*10) + 30*10);

        // Temperature in Fahrenheit, multiplied by 10:
        IntDegF = 9*IntDegC/5+320;

        UART_print(rawToAsciiString(IntDegC), 'C'); // Send the temperature information through
        UART_print(rawToAsciiString(IntDegF), 'F'); // the backchannel UART

        while(UCA0STATW & UCBUSY);
        __no_operation();                     // Set a breakpoint here to verify UART transmission
    }
}

char* rawToAsciiString(int32_t input)
{
    int16_t place = 0, j = 0, mod = 10, div = 1;
    uint16_t i = 0;
    int32_t temp1 = input;
    static char result[6];

    while(temp1 != 0)
    {
        temp1 /= 10;
        place++;
    } // Now 'place' equals the number of digits to the left of the decimal point

    for (j = 0; j < (place-1); j++)
    {
        mod *= 10;
        div *= 10;
    } // mod = 10^place, div = 10^(place-1)

    for (i = 0; i < (place-1); i++)
    {

        result[i] = (input % mod) / div | 0x30;
        mod /= 10;
        div /= 10;
        __no_operation();
    }
    // Now all digits to the left of the decimal place are stored in ASCII in 'result'
    result[i++] = 0x2E;                       // Decimal point
    result[i] = (input % mod) / div | 0x30;   // Last digit, after decimal point

    return result;
}

void UART_print(char *string, char type)      // Send a zero-terminated string through the UART
{
    char byte = *string++;
    const char cel[] = " Degrees Celsius\r\n";
    const char fah[] = " Degrees Fahrenheit\r\n";
    uint16_t k = 0;
    while(byte != 0)
    {
        while(!(UCA0IFG & UCTXIFG));          // Wait until TX buffer ready
        UCA0TXBUF = byte;                     // Send the next byte of info
        byte = *string++;
    }
    if (type == 'C')
    {
        while(cel[k] != 0)
        {
            while(!(UCA0IFG & UCTXIFG));      // Wait until TX buffer ready
            UCA0TXBUF = cel[k++];             // Send the next byte of info
        }
    }
    else if (type == 'F')
    {
        while(fah[k] != 0)
        {
            while(!(UCA0IFG & UCTXIFG));      // Wait until TX buffer ready
            UCA0TXBUF = fah[k++];             // Send the next byte of info
        }
    }
}

#pragma vector = ADC12_VECTOR
__interrupt void ADC12_ISR(void)
{
  switch(__even_in_range(ADC12IV,76))
  {
    case  ADC12IV_NONE: break;                // Vector  0:  No interrupt
    case  ADC12IV_ADC12OVIFG: break;          // Vector  2:  ADC12MEMx Overflow
    case  ADC12IV_ADC12TOVIFG: break;         // Vector  4:  Conversion time overflow
    case  ADC12IV_ADC12HIIFG: break;          // Vector  6:  ADC12HI
    case  ADC12IV_ADC12LOIFG: break;          // Vector  8:  ADC12LO
    case ADC12IV_ADC12INIFG: break;           // Vector 10:  ADC12IN
    case ADC12IV_ADC12IFG0:                   // Vector 12:  ADC12MEM0
        ADC12IFGR0 &= ~ADC12IFG0;             // Clear interrupt flag
        __bic_SR_register_on_exit(LPM3_bits); // Exit active CPU
        break;
    case ADC12IV_ADC12IFG1: break;            // Vector 14:  ADC12MEM1
    case ADC12IV_ADC12IFG2: break;            // Vector 16:  ADC12MEM2
    case ADC12IV_ADC12IFG3: break;            // Vector 18:  ADC12MEM3
    case ADC12IV_ADC12IFG4: break;            // Vector 20:  ADC12MEM4
    case ADC12IV_ADC12IFG5: break;            // Vector 22:  ADC12MEM5
    case ADC12IV_ADC12IFG6: break;            // Vector 24:  ADC12MEM6
    case ADC12IV_ADC12IFG7: break;            // Vector 26:  ADC12MEM7
    case ADC12IV_ADC12IFG8: break;            // Vector 28:  ADC12MEM8
    case ADC12IV_ADC12IFG9: break;            // Vector 30:  ADC12MEM9
    case ADC12IV_ADC12IFG10: break;           // Vector 32:  ADC12MEM10
    case ADC12IV_ADC12IFG11: break;           // Vector 34:  ADC12MEM11
    case ADC12IV_ADC12IFG12: break;           // Vector 36:  ADC12MEM12
    case ADC12IV_ADC12IFG13: break;           // Vector 38:  ADC12MEM13
    case ADC12IV_ADC12IFG14: break;           // Vector 40:  ADC12MEM14
    case ADC12IV_ADC12IFG15: break;           // Vector 42:  ADC12MEM15
    case ADC12IV_ADC12IFG16: break;           // Vector 44:  ADC12MEM16
    case ADC12IV_ADC12IFG17: break;           // Vector 46:  ADC12MEM17
    case ADC12IV_ADC12IFG18: break;           // Vector 48:  ADC12MEM18
    case ADC12IV_ADC12IFG19: break;           // Vector 50:  ADC12MEM19
    case ADC12IV_ADC12IFG20: break;           // Vector 52:  ADC12MEM20
    case ADC12IV_ADC12IFG21: break;           // Vector 54:  ADC12MEM21
    case ADC12IV_ADC12IFG22: break;           // Vector 56:  ADC12MEM22
    case ADC12IV_ADC12IFG23: break;           // Vector 58:  ADC12MEM23
    case ADC12IV_ADC12IFG24: break;           // Vector 60:  ADC12MEM24
    case ADC12IV_ADC12IFG25: break;           // Vector 62:  ADC12MEM25
    case ADC12IV_ADC12IFG26: break;           // Vector 64:  ADC12MEM26
    case ADC12IV_ADC12IFG27: break;           // Vector 66:  ADC12MEM27
    case ADC12IV_ADC12IFG28: break;           // Vector 68:  ADC12MEM28
    case ADC12IV_ADC12IFG29: break;           // Vector 70:  ADC12MEM29
    case ADC12IV_ADC12IFG30: break;           // Vector 72:  ADC12MEM30
    case ADC12IV_ADC12IFG31: break;           // Vector 74:  ADC12MEM31
    case ADC12IV_ADC12RDYIFG: break;          // Vector 76:  ADC12RDY
    default: break;
  }
}

// Timer0_A3 Interrupt Vector (TAIV) handler
#pragma vector=TIMER0_A0_VECTOR
__interrupt void TIMER0_A0_ISR(void)
{
    __bic_SR_register_on_exit(LPM3_bits); // Exit active CPU
}
