/* --COPYRIGHT--,BSD
 * Copyright (c) 2014, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
// *******************************************************************************
//	Example code demonstrating oversampling to get higher resolution
//
//	DAC8560 EVM jumpers:
//	W1 position +3VA	W2 position 1
//	W3 position open	W4 position open
//	W5 position 3		W6 position 1
//	W7 position open	W8 position open
//	W15 position open	J6 position 5 tied to 6
//
//	MSP-EXP432P401R P5.7 to GND
//
//  MSP-EXP432P401R	DAC8560 EVM:
//	3V3				J6 1 (VCC)
//	2.6	PM_UCA1SIMO	J2 11 (SDI)
//	2.5 PM_UCA1CLK	J2 3 (SCLK)
//	2.4 PM_UCA1STE	J2 1 (CS)
//	5.6 VeREF+		W4 2 (VREF)
//	5.0	A5			J4 2 (VOUT)
//	GND				J2 18 (GND)
//	3V3				J6 8 (+3.3VA)
//
//  R Brown
//  Texas Instruments Inc.
//  December 2015
//  Built with CCS Version: 6.0.1
//******************************************************************************

#include "msp.h"
#include "math.h"

unsigned long result = 0;
unsigned long ADCresult;                    	// Result after oversampling & Averaging
                                            	// 16 samples for 16-bit ADC

const int extrabits = 0;					// Extra bits of resolution, controls the oversampling factor
const int period = 64;						// 256 [over]samples per second (ACLK/(2*period*oversampling factor))
unsigned int data = 0x00;					// Data to be sent out
unsigned int oversampling = 0;				// Oversampling factor, set at the start of main

unsigned long movingAVG = 0;
unsigned int AVGBuffer [1] = {0};				//Oversampling factor
unsigned int AVGpnt = 0;
unsigned int i = 0;

void sendline(unsigned int, unsigned long);
void Port_Mapping(void);

// Port2 Port Mapping definitions
const uint8_t P2Mapping[8] = {
                                      PM_NONE,
                                      PM_NONE,
                                      PM_NONE,
                                      PM_NONE,
									  PM_UCA1STE,
									  PM_UCA1CLK,
									  PM_UCA1SIMO,
									  PM_UCA1SOMI     };

int main(void)
{
    WDTCTL = WDTPW | WDTHOLD;                	// Stop WDT

    oversampling = pow(4, extrabits);			// Oversampling factor

    // Initialize P1.1 for push button
    P1REN |= BIT1;
    P1OUT |= BIT1;

    // Configure P5.0/A5, P5.6/VeREF+, P5.7/VeREF-
    P5SEL1 |= BIT0 | BIT6 | BIT7;
    P5SEL0 |= BIT0 | BIT6 | BIT7;

    // Configure P1.3/UCA0TXD, P1.2/UCA0RXD
    P1SEL0 |= BIT2 | BIT3;

    // Set LFXT pin as second function
    PJSEL0 |= BIT0 | BIT1;

    // Port Map P2.4/UCA1STE, P2.5/UCA1CLK, P2.6/UCA1SIMO, & P2.7/UCA1SOMI
    Port_Mapping();

    /* Setup UCA1 Port Pins */
    P2DIR |= 0xF0;                          // P2.4 - P2.7 output
    P2SEL0 |= 0xF0;                         // P2.4 - P2.7 Port Map functions
    P2SEL1 = 0;                             // P2.4 - P2.7 Port Map functions

    CS->KEY = CS_KEY_VAL;                      	// Unlock CS module for register access
    CS->CTL0 = 0;                            		// Reset tuning parameters
    CS->CTL0 = CS_CTL0_DCORSEL_3;                   		// Set DCO to 12MHz (nominal, center of 8-16MHz range)
    CS->CTL2 |= CS_CTL2_LFXT_EN;                    		// LFXT on
    // Loop until XT1, XT2 & DCO fault flag is cleared
    do
    {
    // Clear XT2,XT1,DCO fault flags
       CS->CLRIFG |= CS_CLRIFG_CLR_DCOR_OPNIFG | CS_CLRIFG_CLR_HFXTIFG | CS_CLRIFG_CLR_LFXTIFG;
       SYSCTL->NMI_CTLSTAT &= ~ SYSCTL_NMI_CTLSTAT_CS_SRC;
    } while (SYSCTL->NMI_CTLSTAT & SYSCTL_NMI_CTLSTAT_CS_FLG);// Test oscillator fault flag
    CS->CTL1 = CS->CTL1 = CS_CTL1_SELA_0 | CS_CTL1_SELS_3 | CS_CTL1_SELM_3;// Select ACLK as LFXTCLK
    CS->KEY = 0;                              	// Lock CS module from unintended accesses

    //Initialize SPI
    UCA1CTLW0 |= UCSWRST;                     	// **Put state machine in reset**
                                              	// 4-pin, 8-bit SPI master
    UCA1CTLW0 |= UCMST|UCSYNC|UCCKPL|UCMSB|UCMODE_2|UCSTEM;
                                              	// Clock polarity high, MSB
    UCA1CTLW0 |= UCCKPH | UCSSEL__ACLK;       	// ACLK
    UCA1BR0 = 0x0F;                           	// /15,fBitClock = fBRCLK/(UCBRx+1).
    UCA1BR1 = 0;                              	//
    UCA1CTLW0 &= ~UCSWRST;                    	// **Initialize USCI state machine**

    //Initialize UART
    UCA0CTLW0 |= UCSWRST;
    UCA0CTLW0 |= UCSSEL__SMCLK;             	// Put eUSCI in reset
    // Baud Rate calculation
    // 12000000/(16*9600) = 78.125
    // Fractional portion = 0.125
    // User's Guide Table 21-4: UCBRSx = 0x10
    // UCBRFx = int ( (78.125-78)*16) = 2
    UCA0BR0 = 78;                           	// 12000000/16/9600
    UCA0BR1 = 0x00;
    UCA0MCTLW = 0x1000 | UCOS16 | 0x0020;

    UCA0CTLW0 &= ~UCSWRST;                	  	// Initialize eUSCI

    // Configure ADC14
    ADC14->CTL0 = ADC14_CTL0_ON | ADC14_CTL0_SHP | ADC14_CTL0_SHT0_4// Turn on ADC14, set sampling time
    		| ADC14_CTL0_SHS_3 | ADC14_CTL0_CONSEQ_2;		// Rpt single ch; TA1.1 trigger
    ADC14->CTL1 = ADC14_CTL1_RES_3;                     // Use sampling timer, 14-bit conversion results
    ADC14->MCTL[0] = ADC14_MCTLN_VRSEL_14 | ADC14_MCTLN_INCH_5;   // Vr+ = VeREF+ (ext) and Vr-=AVss, A5
    ADC14->IER0 = 0x0001;                         // ADC_IFG upon conv result-ADCMEM0
    __enable_interrupt();
    NVIC->ISER[0] = 1 << (ADC14_IRQn & 31);	// Enable ADC interrupt in NVIC module

    // Configure ADC timer trigger TA1.1
    TA1CCR0 = period;                        	// PWM Period
    TA1CCR1 = period/2;                         // Duty cycle TA1.1
    TA1CCTL1 = OUTMOD_4;                   		// TA1CCR1 toggle
    TA1CTL = TASSEL_1 | MC_3 | TACLR;    		// ACLK, up mode
    ADC14->CTL0 |= ADC14_CTL0_ENC;               		// Enable conversion

    while(P1IN & BIT1);							// Wait for P1.2 button press

    while(1)
    {
    	SCB->SCR |= SCB_SCR_SLEEPONEXIT_Msk;        	// Do not wake up on exit from ISR
        __sleep();
        __no_operation();
        ADCresult = movingAVG >> extrabits;

        data += pow(2, 6-extrabits);

        while (!(UCA1IFG&UCTXIFG));
        UCA1TXBUF = 0x00;

        while (!(UCA1IFG&UCTXIFG));
        UCA1TXBUF = data>>8;

        while (!(UCA1IFG&UCTXIFG));
        UCA1TXBUF = data;

        sendline(data-pow(2, 6-extrabits), ADCresult);

        if (data >= 0xFFF0){data = 0; while(1) __no_operation();}

    }
}

// ADC14 interrupt service routine
void ADC14IsrHandler(void)
{
    if (ADC14->IFGR0 & ADC14_IFGR0_IFG0)
    {
		//Automatically clears ADC10IFG0 by reading memory buffer
		result = ADC14->MEM[0];
		movingAVG -= AVGBuffer[AVGpnt];
		movingAVG += result;
		AVGBuffer[AVGpnt++] = result;
		if (AVGpnt == oversampling)
		{
		  AVGpnt = 0;
		  // Wake up on exit from ISR
		  SCB->SCR &= ~SCB_SCR_SLEEPONEXIT_Msk;
		}
    }
}

void sendline(unsigned int step, unsigned long voltage)
{
	unsigned int i = 0;
	const char stepstring[] = {"Step: "};				//Strings being sent to the PC
	const int stepsize = 6;								//String size
	const char voltagestring[] = {"Voltage: "};
	const int voltagesize = 9;
	const char endlinestring[] = {"\n\r"};				//Command to end line
	const int endlinesize = 4;

	//Send the step string
	for(i=0; i<=stepsize; i++)
	{
		while(!(UCA0IFG&UCTXIFG));
		UCA0TXBUF = stepstring[i];
	}

	//Sends the step in hexadecimal
	while(!(UCA0IFG&UCTXIFG));
	if(((step>>12)&0xF)>=0x0A) UCA0TXBUF = ((step>>12)&0xF) + 0x37;
	else UCA0TXBUF = ((step>>12)&0xF) + 0x30;
	while(!(UCA0IFG&UCTXIFG));
	if(((step>>8)&0xF)>=0x0A) UCA0TXBUF = ((step>>8)&0xF) + 0x37;
	else UCA0TXBUF = ((step>>8)&0xF) + 0x30;
	while(!(UCA0IFG&UCTXIFG));
	if(((step>>4)&0xF)>=0x0A) UCA0TXBUF = ((step>>4)&0xF) + 0x37;
	else UCA0TXBUF = ((step>>4)&0xF) + 0x30;
	while(!(UCA0IFG&UCTXIFG));
	if((step&0xF)>=0x0A) UCA0TXBUF = (step&0xF) + 0x37;
	else UCA0TXBUF = (step&0xF) + 0x30;

	//Insert a tab to separate the step from the voltage
	while(!(UCA0IFG&UCTXIFG));
	UCA0TXBUF = 0x09;

	//Send the voltage string
	for(i=0; i<=voltagesize; i++)
	{
		while(!(UCA0IFG&UCTXIFG));
		UCA0TXBUF = voltagestring[i];
	}

	//Sends the voltage in hexadecimal
	while(!(UCA0IFG&UCTXIFG));
	if(((voltage>>12)&0xF)>=0x0A) UCA0TXBUF = ((voltage>>12)&0xF) + 0x37;
	else UCA0TXBUF = ((voltage>>12)&0xF) + 0x30;
	while(!(UCA0IFG&UCTXIFG));
	if(((voltage>>8)&0xF)>=0x0A) UCA0TXBUF = ((voltage>>8)&0xF) + 0x37;
	else UCA0TXBUF = ((voltage>>8)&0xF) + 0x30;
	while(!(UCA0IFG&UCTXIFG));
	if(((voltage>>4)&0xF)>=0x0A) UCA0TXBUF = ((voltage>>4)&0xF) + 0x37;
	else UCA0TXBUF = ((voltage>>4)&0xF) + 0x30;
	while(!(UCA0IFG&UCTXIFG));
	if((voltage&0xF)>=0x0A) UCA0TXBUF = (voltage&0xF) + 0x37;
	else UCA0TXBUF = (voltage&0xF) + 0x30;

	//Start a new line for the next string
	for(i=0; i<=endlinesize; i++)
	{
		while(!(UCA0IFG&UCTXIFG));
		UCA0TXBUF = endlinestring[i];
	}
}

void Port_Mapping()
{
    uint8_t i;
    volatile uint8_t *ptr;

    PMAPKEYID = PMAP_KEYID_VAL;                      // Enable Write-access to modify port mapping registers

    #ifdef PORT_MAP_RECFG
    PMAPCTL = PMAPRECFG;                    // Allow reconfiguration during runtime
    #endif

    ptr = (volatile uint8_t *) (&P2MAP01);
    for (i = 0; i < 8; i++)
    {
        *ptr = P2Mapping[i];
        ptr++;
    }

    PMAPKEYID = 0;                            // Disable Write-Access to modify port mapping registers
    #ifdef PORT_MAP_EINT
    __enable_interrupt();                   // Re-enable all interrupts
    #endif

}

