/* --COPYRIGHT--,BSD
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//******************************************************************************
//  msp_fir_q15 - Real FIR filter of a 50 tap real FIR filter with vector length
//  of 200.
//
//  This example demonstrates simple benchmarking of the msp_fir_q15 MSP DSP Lib
//  API. There are a few options to get started. First select the clock system
//  setting most suitable for the operating case desired. Also select whether
//  or not the test should count cycles per function call, or measure power
//  consumption. There are separate test cases for these as the timer used for
//  measuring cycles also consumes a little power.
//
//  This is the example program used for the official LEA Benchmark's.
//
//  Evan Wakefield
//  Texas Instruments Inc.
//  Septembers 2016
//******************************************************************************
#include "msp430.h"

#include <math.h>
#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include "DSPLib.h"

//if both of the above are 0, defaults to 1MHz operation
#define CS_8MHZ             0
#define CS_16MHZ            1

// Decide on whether or not to take power or cycle measurement.
#define MEASURE_POWER       1
#define MEASURE_CYCLE_COUNT 0

/* Define example coefficients if no DSPLib GUI header is provided. */
#ifndef FILTER_COEFFS_EX1
/* 15th-order low pass filter coefficients (Fs=8192, Fc=1200) */
DSPLIB_DATA(FILTER_COEFFS_EX1,4)            // 50 total coeffs
_q15 FILTER_COEFFS_EX1[] = {
    12, 6554, 38, 6821, 66, 7089, 67, 7356, 12,
    7624, -82, 7891, -146, 8159, -97, 8426, 75, 8694, 257, 8961, 268, 9229,
    16, 9496, -362, 9764, -542, 10031, -260, 10299, 400, 10566, 940, 10834,
    772, 11101, -260, 11368, -1537, 11636, -1902, 11903, -404, 12171, 2916,
    12438, 6807, 12706, 9425, 12973, 9425, 13241, 6807, 13508, 2916, 13776,
    -404, 14043, -1902, 14311, -1537, 14578, -260, 14846, 772, 15113, 940,
    15381, 400, 15648, -260, 15916, -542, 16183, -362, 16451, 16, 16718, 268,
    16986, 257, 17253, 75, 17521, -97, 17788, -146, 18056, -82, 18323, 12, 18591,
    67, 18858, 66, 19126, 38, 19393, 12, 19661
};
#endif

/* Input signal parameters */
#define FIR_LENGTH          200
#define FIR_DATA_LENGTH     256
#define COEFF_LENGTH        50

/* Allocate data arrays */

/*
 * Circular buffer of size 2*FIR_LENGTH, aligned to 4*FIR_LENGTH in order to
 * use the circular buffer feature.
 */
/* Allocate data arrays */
#if defined(__MSP430_HAS_LEA__)
DSPLIB_DATA(input,MSP_ALIGN_FIR_Q15(FIR_DATA_LENGTH))
#endif
_q15 input[FIR_DATA_LENGTH*2];

/* Filter result */
DSPLIB_DATA(result,4)
_q15 result[FIR_DATA_LENGTH+COEFF_LENGTH];

/* Golden Input vector for static calculation base */
#pragma PERSISTENT(inputVector)
int32_t inputVector[FIR_LENGTH] = {
0x08f11159, 0x121ce61b, 0x173ff715, 0x163be513, 0x1f3307ca, 0xf5eb162e, 0x07be107c, 0xf852ea79,
0x18b0001f, 0x09fe0215, 0xfa7a08c6, 0x10e3f37a, 0x095603dd, 0x1c98fe4c, 0xf5e8f08d, 0x01e6f133,
0x174cfeda, 0xf5ee1403, 0xf916ee8d, 0xf6570c28, 0xe2581849, 0xe8c6ec42, 0x16baea16, 0xea35e118,
0xe32def6b, 0xf832f434, 0xe7ad1da6, 0xe1850915, 0x0760f104, 0xe191fe1d, 0xef50fd49, 0xfc22e377,
0xf77b069c, 0xf529f210, 0xfe39f091, 0x08d9eb7f, 0x17c0e090, 0x0aa5f1a0, 0x12b7ee7b, 0x1de5f4db,
0xe90ef5e4, 0x138a1617, 0x1f2c10bb, 0x01bef50f, 0xf32ce68d, 0xe3d5e34f, 0x06c6e27b, 0xe9171ff6,
0x1206fe41, 0xeed11e3f, 0x09be103e, 0xf0cff7ab, 0xe94be17f, 0x0ff9fb75, 0xf04ffc02, 0xf1e5eddf,
0x03f6195d, 0xea19056f, 0xfa7fe323, 0x09aeebd7, 0x0f26edaa, 0x1025ed84, 0x0d7a15ec, 0x142cf93d,
0x177b1f81, 0xefc002fd, 0xfaa9e13b, 0x02b9e08f, 0x1c02e3f3, 0x1ff412b2, 0x00910c52, 0xe5b011d9,
0x17480487, 0xe7aa0fc7, 0xfb9f11c7, 0xff72f159, 0x1edc0ac5, 0xe0b20f96, 0x08d2ec58, 0x0bd8f4df,
0xf7dce04f, 0x0189fb99, 0x1c27f285, 0x16770442, 0x16f5f82b, 0x047cf66b, 0x08441786, 0x1c0eea2c,
0xf9f41f8c, 0xf15403b8, 0xf512f592, 0x0057f0c6, 0x005b13ee, 0x0046e109, 0x15e6092e, 0xe97e0c1f,
0x07b80dc2, 0x0047eb07, 0xef49e3e0, 0x1c0a16be, 0x0d29063e, 0x1dc4e087, 0x0ab2156d, 0xf4faf9d4,
0xfd8d04a6, 0xfa38e64e, 0x1713f29f, 0x068dfa90, 0x1b97176f, 0xe09e06d3, 0x12f2f17f, 0xff420a1b,
0xf5221aab, 0xfe8bff89, 0x1646e46b, 0xeaa91a95, 0xfb1ce370, 0xf8dd086d, 0x024105ce, 0x0a74edd7,
0x1425ffce, 0xf26c04ac, 0xff3ceb39, 0x02a7f8f9, 0xffcd1ad3, 0x0c52e346, 0xed6112bf, 0xed6b0b93,
0x08f11159, 0x121ce61b, 0x173ff715, 0x163be513, 0x1f3307ca, 0xf5eb162e, 0x07be107c, 0xf852ea79,
0x18b0001f, 0x09fe0215, 0xfa7a08c6, 0x10e3f37a, 0x095603dd, 0x1c98fe4c, 0xf5e8f08d, 0x01e6f133,
0x174cfeda, 0xf5ee1403, 0xf916ee8d, 0xf6570c28, 0xe2581849, 0xe8c6ec42, 0x16baea16, 0xea35e118,
0xe32def6b, 0xf832f434, 0xe7ad1da6, 0xe1850915, 0x0760f104, 0xe191fe1d, 0xef50fd49, 0xfc22e377,
0xf77b069c, 0xf529f210, 0xfe39f091, 0x08d9eb7f, 0x17c0e090, 0x0aa5f1a0, 0x12b7ee7b, 0x1de5f4db,
0xe90ef5e4, 0x138a1617, 0x1f2c10bb, 0x01bef50f, 0xf32ce68d, 0xe3d5e34f, 0x06c6e27b, 0xe9171ff6,
0x1206fe41, 0xeed11e3f, 0x09be103e, 0xf0cff7ab, 0xe94be17f, 0x0ff9fb75, 0xf04ffc02, 0xf1e5eddf,
0x03f6195d, 0xea19056f, 0xfa7fe323, 0x09aeebd7, 0x0f26edaa, 0x1025ed84, 0x0d7a15ec, 0x142cf93d,
0x177b1f81, 0xefc002fd, 0xfaa9e13b, 0x02b9e08f, 0x1c02e3f3, 0x1ff412b2, 0x00910c52, 0xe5b011d9,
};

/* Allocate DSPLib parameter structures. */
msp_fir_q15_params firParams;

uint32_t cycleCount;

//Function declarations
void initSignal(void);
void configClocks(void);
void configGPIO(void);
void isolateLEA(void);

void main(void)
{
    /* Disable WDT. */
    WDTCTL = WDTPW + WDTHOLD;

    configGPIO();

    // Disable the GPIO power-on default high-impedance mode to activate
    // previously configured port settings
    PM5CTL0 &= ~LOCKLPM5;

    configClocks();

    initSignal();

    /* Initialize the FIR parameter structure. */
    firParams.length = FIR_LENGTH;
    firParams.tapLength = COEFF_LENGTH;
    firParams.coeffs = FILTER_COEFFS_EX1;
    firParams.enableCircularBuffer = false;

    // Isolate LEA call with LPM1 routine
    isolateLEA();

    // Wake up and compute
    /* Call fixed scaling FFT function. */
#if MEASURE_CYCLE_COUNT
    msp_benchmarkStart(TIMER_A0_BASE, 16);
    msp_fir_q15(&firParams, input, result);
    cycleCount = msp_benchmarkStop(TIMER_A0_BASE);
#endif

#if MEASURE_POWER
    msp_fir_q15(&firParams, input, result);
#endif

    // Isolate LEA call with LPM1 routine
    isolateLEA();

    /* End of program. */
    __bis_SR_register(LPM4_bits);

    __no_operation();
}

// Timer A1 interrupt service routine for isolate LEA call
#if defined(__TI_COMPILER_VERSION__) || defined(__IAR_SYSTEMS_ICC__)
#pragma vector = TIMER1_A0_VECTOR
__interrupt void Timer1_A0_ISR(void)
#elif defined(__GNUC__)
void __attribute__ ((interrupt(TIMER1_A0_VECTOR))) Timer1_A0_ISR (void)
#else
#error Compiler not supported!
#endif
{
    __bic_SR_register_on_exit(LPM1_bits);
}

void isolateLEA() {
    // Use Timer A for a short delay to isolate the FFT for power analysis
    TA1CCTL0 = CCIE;                        // TACCR0 interrupt enabled
    TA1CCR0 = 1000;
    TA1CTL |= TACLR;                        // Clear Timer A
    TA1CTL = TASSEL__ACLK | MC__UP;         // ACLK, UP mode
    __bis_SR_register(LPM1_bits | GIE);
    TA1CTL &= ~(MC__UP);                    // Stop Timer
    TA1CCTL0 &= ~(CCIE);                        // TACCR0 interrupt disable
}

void initSignal(){
    // Copy golden vector to input vector
    memcpy(input, inputVector, sizeof(inputVector));
}

void configGPIO(void)
{
    // Configure ports for low power
    PADIR = 0xFFFF;  PAOUT = 0;
    PBDIR = 0xFFFF;  PBOUT = 0;
    PCDIR = 0xFFFF;  PCOUT = 0;
    PDDIR = 0xFFFF;  PDOUT = 0;
    PJDIR = 0xFFFF;  PJOUT = 0;
    P1DIR = 0xFF;  P1OUT = 0;
    P2DIR = 0xFF;  P2OUT = 0;
    P3DIR = 0xFF;  P3OUT = 0;
    P4DIR = 0xFF;  P4OUT = 0;
    P5DIR = 0xFF;  P5OUT = 0;
    P6DIR = 0xFF;  P6OUT = 0;
    P7DIR = 0xFF;  P7OUT = 0;
    P8DIR = 0xFF;  P8OUT = 0;
}

// Configure clocks for either 8 or 16MHz
void configClocks(void)
{
#if CS_8MHZ
    //Ensure that there are 0 wait states enabled
    FRCTL0 = FRCTLPW | NWAITS_0;


    // Clock System Setup
    CSCTL0_H = CSKEY_H;                     // Unlock CS registers
    CSCTL1 = DCOFSEL_0;                     // Set DCO to 1MHz

    // Set SMCLK = MCLK = DCO, ACLK = VLOCLK
    CSCTL2 = SELA__VLOCLK | SELS__DCOCLK | SELM__DCOCLK;

    // Per Errata CS12 set divider to 4 before changing frequency to
    // prevent out of spec operation from overshoot transient
    CSCTL3 = DIVA__4 | DIVS__4 | DIVM__4;   // Set all corresponding clk sources to divide by 4 for errata
    CSCTL1 = DCOFSEL_6;                     // Set DCO to 8MHz

    // Delay by ~10us to let DCO settle. 60 cycles = 20 cycles buffer + (10us / (1/4MHz))
    __delay_cycles(60);

    CSCTL3 = DIVA__1 | DIVS__1 | DIVM__1;   // Set MCLK  to 8MHz, SMCLK to 1MHz

    CSCTL4 = HFXTOFF + LFXTOFF;             // Cancel external clocks
    CSCTL5 &= ~(HFXTOFFG + LFXTOFFG);

    CSCTL0_H = 0;                           // Lock CS Registers

#endif

#if CS_16MHZ
    // Configure one FRAM waitstate as required by the device datasheet for MCLK
    // operation beyond 8MHz _before_ configuring the clock system.
    FRCTL0 = FRCTLPW | NWAITS_1;

    // Clock System Setup
    CSCTL0_H = CSKEY_H;                     // Unlock CS registers
    CSCTL1 = DCOFSEL_0;                     // Set DCO to 1MHz

    // Set SMCLK = MCLK = DCO, ACLK = VLOCLK
    CSCTL2 = SELA__VLOCLK | SELS__DCOCLK | SELM__DCOCLK;

    // Per Errata CS12 set divider to 4 before changing frequency to
    // prevent out of spec operation from overshoot transient
    CSCTL3 = DIVA__4 | DIVS__4 | DIVM__4;   // Set all corresponding clk sources to divide by 4 for errata
    CSCTL1 = DCOFSEL_4 | DCORSEL;           // Set DCO to 16MHz

    // Delay by ~10us to let DCO settle. 60 cycles = 20 cycles buffer + (10us / (1/4MHz))
    __delay_cycles(60);

    CSCTL3 = DIVA__1 | DIVS__1 | DIVM__1;   // Set MCLK to 16MHz, SMCLK to 1MHz

    CSCTL4 = HFXTOFF + LFXTOFF;             // Cancel external clocks
    CSCTL5 &= ~(HFXTOFFG + LFXTOFFG);

    CSCTL0_H = 0;                           // Lock CS Registers
#endif
}

