/*
 * Copyright (c) 2021, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== enet_httpServer_tirtos.c ========
 */
/* XDCtools Header files */
#include <xdc/std.h>
#include <xdc/runtime/System.h>

/* BIOS Header files */
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/knl/Clock.h>

/* TI-RTOS Header files */
#include <ti/drivers/GPIO.h>

/* Board Header file */
#include "Board.h"

#include <ti/ndk/inc/netmain.h>

/* file system header file */
#include "fs/index.h"
#include "fs/about.h"
#include "fs/overview.h"
#include "fs/block.h"
#include "fs/greetings.h"
#include "fs/chip.h"
#include "fs/tm4c1294xl.h"
#include "fs/error404.h"
#include "fs/blockdiagram.h"
#include "fs/ti.h"
#include "fs/tm4c.h"
#include "fs/styles.h"
#include "fs/javascript.h"

#define TASKSTACKSIZE   512

Task_Struct task0Struct;
Char task0Stack[TASKSTACKSIZE];

/*
 * An entry for the file getTime.cgi is translated into the C function getTime().
 * Whenever a HTTP POST is made to the file getTime.cgi, the getTime() function
 * is called.
 */
Int getTime(SOCKET s, int length)
{
    Char buf[200];
    static UInt scalar = 0;

    if (scalar == 0) {
        scalar = 1000000u / Clock_tickPeriod;
    }
    /*
     * Send the status of this request to the client.
     * Ex:
     *      HTTP/1.1 200 OK
     *      Content-Type: text/html
     */
    httpSendStatusLine(s, HTTP_OK, CONTENT_TYPE_HTML);

    httpSendClientStr(s, CRLF);
    /*
     * Below lines of code construct the body of the HTML to
     * send to the browser. When opened on the browser it will
     * display the Elapsed Time.
     */
    httpSendClientStr(s,
        "<html><head><title>SYS/BIOS Clock "\
	"Time</title></head><body><h1>Elapsed Time</h1>\n");
    System_sprintf(buf, "<p>Up for %d seconds</p>\n",
        ((unsigned long)Clock_getTicks() / scalar));
    httpSendClientStr(s, buf);
    httpSendClientStr(s, "</table></body></html>");
    return (1);
}

/*
 * AddWebFiles() is the user defined hook function that is called when the stack
 * is configured. See httpServer.cfg for details.
 *
 * var Global = xdc.useModule('ti.ndk.config.Global');
 * var Http = xdc.useModule('ti.ndk.config.Http');
 * Global.IPv6 = false;
 * var http0Params = new Http.Params();
 * var http0 = Http.create(http0Params);
 * Global.stackInitHook = "&AddWebFiles";
 * Global.stackDeleteHook = "&RemoveWebFiles";
 */
Void AddWebFiles(Void)
{
    /*
     * efs_createfile creates RAM based file. This function creates an internal
     * record of the RAM based file with the indicated filename, file length, and
     * data pointer. Refer to TI Network Developer's Kit (NDK) API Reference Guide
     * (SPRU524K) for details.
     */
    efs_createfile("index.html", INDEX_SIZE, (UINT8 *)INDEX);
    efs_createfile("overview.htm", OVERVIEW_SIZE, (UINT8 *)OVERVIEW);
    efs_createfile("about.htm", ABOUT_SIZE, (UINT8 *)ABOUT);
    efs_createfile("block.htm", BLOCK_SIZE, (UINT8 *)BLOCK);
    efs_createfile("404.htm", ERROR404_SIZE, (UINT8 *)ERROR404);
    efs_createfile("ek-tm4c1294xl.jpg", TM4C1294XL_SIZE, (UINT8 *)TM4C1294XL);
    efs_createfile("ti.jpg", TI_SIZE, (UINT8 *)TI);
    efs_createfile("block.jpg", BLOCKDIAGRAM_SIZE, (UINT8 *)BLOCKDIAGRAM);
    efs_createfile("tm4c.jpg", TM4C_SIZE, (UINT8 *)TM4C);
    efs_createfile("getTime.cgi", 0, (UINT8 *)&getTime);
    efs_createfile("greetings.html", GREETINGS_SIZE, (UINT8 *)GREETINGS);
    efs_createfile("javascript.js", JAVA_SIZE, (UINT8 *)JAVA);
    efs_createfile("styles.css", STYLES_SIZE, (UINT8 *)STYLES);
}

Void RemoveWebFiles(Void)
{
    /*
     * Destroy a RAM Based file.
     */
    efs_destroyfile("index.html");
    efs_destroyfile("overview.htm");
    efs_destroyfile("about.htm");
    efs_destroyfile("block.htm");
    efs_destroyfile("404.htm");
    efs_destroyfile("ek-tm4c1294xl.jpg");
    efs_destroyfile("ti.jpg");
    efs_destroyfile("block.jpg");
    efs_destroyfile("tm4c.jpg");
    efs_destroyfile("getTime.cgi");
    efs_destroyfile("greetings.html");
    efs_destroyfile("javascript.js");
    efs_destroyfile("styles.css");

}

/*
 *  ======== heartBeatFxn ========
 *  Toggle the Board_LED0. The Task_sleep is determined by arg0 which
 *  is configured for the heartBeat Task instance.
 */
Void heartBeatFxn(UArg arg0, UArg arg1)
{
    while (1) {
        Task_sleep((unsigned int)arg0);
        GPIO_toggle(Board_LED0);
    }
}

/*
 *  ======== main ========
 */
int main(void)
{
    Task_Params taskParams;
    /* Call board init functions */
    Board_initGeneral();
    Board_initEMAC();
    Board_initGPIO();

    /* Construct heartBeat Task  thread */
    Task_Params_init(&taskParams);
    taskParams.arg0 = 1000;
    taskParams.stackSize = TASKSTACKSIZE;
    taskParams.stack = &task0Stack;
    Task_construct(&task0Struct, (Task_FuncPtr)heartBeatFxn, &taskParams, NULL);

     /* Turn on user LED */
    GPIO_write(Board_LED0, Board_LED_ON);

    System_printf("Starting the HTTP Webserver example\nSystem provider is set to SysMin. "
                  "Halt the target to view any SysMin contents in ROV.\n");
    /* SysMin will only print to the console when you call flush or exit */
    System_flush();

    /* Start BIOS */
    BIOS_start();

    return (0);
}
