//--------------------------------------------------------------------------
// IP Stack Multicast host demo
//--------------------------------------------------------------------------
#include <stdio.h>
#include <netmain.h>
#include <_stack.h>

//---------------------------------------------------------------------------
// Title String
//
char *VerStr = "\nTCP/IP Multicast Example Host\n";

// Define the Interface number
#define IFIDX 1

// Our NETCTRL callback functions
static void   NetworkOpen();
static void   NetworkClose();
static void   NetworkIPAddr( IPN IPAddr, uint IfIdx, uint fAdd );

// Fun reporting function
static void   ServiceReport( uint Item, uint Status, uint Report, HANDLE hCfgEntry );

static volatile int IPAddrCnt = 0;
//---------------------------------------------------------------------------
// Multicast UDP handle, server task prototype and buffers
//
static HANDLE hUDPServer=0;
static void udp_test();
char sendBuffer[1500];
char recvBuffer[1500];

//---------------------------------------------------------------------------
// Configuration
//
char *HostName    = "HostEVMDM642";
char *LocalIPAddr = "0.0.0.0";          // Set to "0.0.0.0" for DHCP
char *LocalIPMask = "255.255.254.0";    // Not used when using DHCP
char *GatewayIP   = "0.0.0.0";                  // Not used when using DHCP
char *DomainName  = "demo.net";         // Not used when using DHCP
char *DNSServer   = "0.0.0.0";          // Used when set to anything but zero

char *MulticastAddr = "224.1.2.3";  // Multicast group address

UINT8 DHCP_OPTIONS[] = { DHCPOPT_SERVER_IDENTIFIER, DHCPOPT_ROUTER };

//---------------------------------------------------------------------
// Main Entry Point
//---------------------------------------------------------------------
int main()
{

}

//
// Main Thread
//
int StackTest()
{
    int               rc;
    HANDLE            hCfg;

    //
    // THIS MUST BE THE ABSOLUTE FIRST THING DONE IN AN APPLICATION!!
    //
    rc = NC_SystemOpen( NC_PRIORITY_LOW, NC_OPMODE_INTERRUPT );
    if( rc )
    {
        printf("NC_SystemOpen Failed (%d)\n",rc);
        for(;;);
    }

    // Print out our banner
    printf(VerStr);

    //
    // Create and build the system configuration from scratch.
    //

    // Create a new configuration
    hCfg = CfgNew();
    if( !hCfg )
    {
        printf("Unable to create configuration\n");
        goto main_exit;
    }

    // We better validate the length of the supplied names
    if( strlen( DomainName ) >= CFG_DOMAIN_MAX ||
        strlen( HostName ) >= CFG_HOSTNAME_MAX )
    {
        printf("Names too long\n");
        goto main_exit;
    }

    // Add our global hostname to hCfg (to be claimed in all connected domains)
    CfgAddEntry( hCfg, CFGTAG_SYSINFO, CFGITEM_DHCP_HOSTNAME, 0,
                 strlen(HostName), (UINT8 *)HostName, 0 );

    // If the IP address is specified, manually configure IP and Gateway
    if( inet_addr(LocalIPAddr) )
    {
        CI_IPNET NA;
        CI_ROUTE RT;
        IPN      IPTmp;

        // Setup manual IP address
        bzero( &NA, sizeof(NA) );
        NA.IPAddr  = inet_addr(LocalIPAddr);
        NA.IPMask  = inet_addr(LocalIPMask);
        strcpy( NA.Domain, DomainName );
        NA.NetType = 0;

        // Add the address to interface 1
        CfgAddEntry( hCfg, CFGTAG_IPNET, IFIDX, 0,
                           sizeof(CI_IPNET), (UINT8 *)&NA, 0 );

        // Add the default gateway. Since it is the default, the
        // destination address and mask are both zero (we go ahead
        // and show the assignment for clarity).
        bzero( &RT, sizeof(RT) );
        RT.IPDestAddr = 0;
        RT.IPDestMask = 0;
        RT.IPGateAddr = inet_addr(GatewayIP);

        // Add the route
        CfgAddEntry( hCfg, CFGTAG_ROUTE, 0, 0,
                           sizeof(CI_ROUTE), (UINT8 *)&RT, 0 );

        // Manually add the DNS server when specified
        IPTmp = inet_addr(DNSServer);
        if( IPTmp )
            CfgAddEntry( hCfg, CFGTAG_SYSINFO, CFGITEM_DHCP_DOMAINNAMESERVER,
                         0, sizeof(IPTmp), (UINT8 *)&IPTmp, 0 );
    }
    // Else we specify DHCP
    else
    {
        CI_SERVICE_DHCPC dhcpc;

        // Specify DHCP Service on IF-1
        bzero( &dhcpc, sizeof(dhcpc) );
        dhcpc.cisargs.Mode   = CIS_FLG_IFIDXVALID;
        dhcpc.cisargs.IfIdx  = IFIDX;
        dhcpc.cisargs.pCbSrv = &ServiceReport;
        dhcpc.param.pOptions = DHCP_OPTIONS;
        dhcpc.param.len = 2;
        CfgAddEntry( hCfg, CFGTAG_SERVICE, CFGITEM_SERVICE_DHCPCLIENT, 0,
                     sizeof(dhcpc), (UINT8 *)&dhcpc, 0 );
    }

    //
    // Configure IPStack/OS Options
    //

    // We don't want to see debug messages less than WARNINGS
    rc = DBG_WARN;
    CfgAddEntry( hCfg, CFGTAG_OS, CFGITEM_OS_DBGPRINTLEVEL,
                 CFG_ADDMODE_UNIQUE, sizeof(uint), (UINT8 *)&rc, 0 );

    //
    // This code sets up the TCP and UDP buffer sizes
    // (Note 8192 is actually the default. This code is here to
    // illustrate how the buffer and limit sizes are configured.)
    //

    // TCP Transmit buffer size
    rc = 8192;
    CfgAddEntry( hCfg, CFGTAG_IP, CFGITEM_IP_SOCKTCPTXBUF,
                 CFG_ADDMODE_UNIQUE, sizeof(uint), (UINT8 *)&rc, 0 );

    // TCP Receive buffer size (copy mode)
    rc = 8192;
    CfgAddEntry( hCfg, CFGTAG_IP, CFGITEM_IP_SOCKTCPRXBUF,
                 CFG_ADDMODE_UNIQUE, sizeof(uint), (UINT8 *)&rc, 0 );

    // TCP Receive limit (non-copy mode)
    rc = 8192;
    CfgAddEntry( hCfg, CFGTAG_IP, CFGITEM_IP_SOCKTCPRXLIMIT,
                 CFG_ADDMODE_UNIQUE, sizeof(uint), (UINT8 *)&rc, 0 );

    // UDP Receive limit
    rc = 8192;
    CfgAddEntry( hCfg, CFGTAG_IP, CFGITEM_IP_SOCKUDPRXLIMIT,
                 CFG_ADDMODE_UNIQUE, sizeof(uint), (UINT8 *)&rc, 0 );

    //
    // Boot the system using this configuration
    //
    // We keep booting until the function returns 0. This allows
    // us to have a "reboot" command.
    //
    do
    {
        rc = NC_NetStart( hCfg, NetworkOpen, NetworkClose, NetworkIPAddr );
    } while( rc > 0 );

    // Leave the Multicast group we joined
    IGMPLeaveHostGroup( inet_addr(MulticastAddr), IFIDX );

    // Delete Configuration
    CfgFree( hCfg );

    // Close the OS
main_exit:
    NC_SystemClose();
    return(0);
}

//
// NetworkOpen
//
// This function is called after the configuration has booted
//
static void NetworkOpen()
{
    // Create our local servers
    hUDPServer = TaskCreate ( udp_test, "UDPTEST", OS_TASKPRINORM, 
                               OS_TASKSTKNORM, 0, 0, 0 ); 
}

//
// NetworkClose
//
// This function is called when the network is shutting down,
// or when it no longer has any IP addresses assigned to it.
//
static void NetworkClose()
{
    TaskDestroy( hUDPServer );
}


//
// NetworkIPAddr
//
// This function is called whenever an IP address binding is
// added or removed from the system.
//
static void NetworkIPAddr( IPN IPAddr, uint IfIdx, uint fAdd )
{
    IPN IPTmp;

    if( fAdd ) {
        printf("Network Added: ");
        IPAddrCnt++;
    }
    else {
        printf("Network Removed: ");
        IPAddrCnt--;
    }
    // Print a message
    IPTmp = ntohl( IPAddr );
    printf("If-%d:%d.%d.%d.%d\n", IfIdx,
            (UINT8)(IPTmp>>24)&0xFF, (UINT8)(IPTmp>>16)&0xFF,
            (UINT8)(IPTmp>>8)&0xFF, (UINT8)IPTmp&0xFF );

}

//
// DHCP_reset()
//
// Code to reset DHCP client by removing it from the active config,
// and then reinstalling it.
//
// Called with:
// IfIdx    set to the interface (1-n) that is using DHCP.
// fOwnTask set when called on a new task thread (via TaskCreate()).
//
void DHCP_reset( uint IfIdx, uint fOwnTask )
{
    CI_SERVICE_DHCPC dhcpc;
    HANDLE h;
    int    rc,tmp;
    uint   idx;

    // If we were called from a newly created task thread, allow
    // the entity that created us to complete
    if( fOwnTask )
        TaskSleep(500);

    // Find DHCP on the supplied interface
    for(idx=1; ; idx++)
    {
        // Find a DHCP entry
        rc = CfgGetEntry( 0, CFGTAG_SERVICE, CFGITEM_SERVICE_DHCPCLIENT,
                          idx, &h );
        if( rc != 1 )
            goto RESET_EXIT;

        // Get DHCP entry data
        tmp = sizeof(dhcpc);
        rc = CfgEntryGetData( h, &tmp, (UINT8 *)&dhcpc );

        // If not the right entry, continue
        if( (rc<=0) || dhcpc.cisargs.IfIdx != IfIdx )
        {
            CfgEntryDeRef(h);
            h = 0;
            continue;
        }

        // This is the entry we want!

        // Remove the current DHCP service
        CfgRemoveEntry( 0, h );

        // Specify DHCP Service on specified IF
        bzero( &dhcpc, sizeof(dhcpc) );
        dhcpc.cisargs.Mode   = CIS_FLG_IFIDXVALID;
        dhcpc.cisargs.IfIdx  = IfIdx;
        dhcpc.cisargs.pCbSrv = &ServiceReport;
        CfgAddEntry( 0, CFGTAG_SERVICE, CFGITEM_SERVICE_DHCPCLIENT, 0,
                     sizeof(dhcpc), (UINT8 *)&dhcpc, 0 );
        break;
    }

RESET_EXIT:
    // If we are a function, return, otherwise, call TaskExit()
    if( fOwnTask )
        TaskExit();
}


void CheckDHCPOptions();

//
// Service Status Reports
//
// Here's a quick example of using service status updates
//
static char *TaskName[]  = { "Telnet","HTTP","NAT","DHCPS","DHCPC","DNS" };
static char *ReportStr[] = { "","Running","Updated","Complete","Fault" };
static char *StatusStr[] = { "Disabled","Waiting","IPTerm","Failed","Enabled" };
static void ServiceReport( uint Item, uint Status, uint Report, HANDLE h )
{
    printf( "Service Status: %-9s: %-9s: %-9s: %03d\n",
            TaskName[Item-1], StatusStr[Status],
            ReportStr[Report/256], Report&0xFF );

    //
    // Example of adding to the DHCP configuration space
    //
    // When using the DHCP client, the client has full control over access
    // to the first 256 entries in the CFGTAG_SYSINFO space.
    //
    // Note that the DHCP client will erase all CFGTAG_SYSINFO tags except
    // CFGITEM_DHCP_HOSTNAME. If the application needs to keep manual
    // entries in the DHCP tag range, then the code to maintain them should
    // be placed here.
    //
    // Here, we want to manually add a DNS server to the configuration, but
    // we can only do it once DHCP has finished its programming.
    //
    if( Item == CFGITEM_SERVICE_DHCPCLIENT &&
        Status == CIS_SRV_STATUS_ENABLED &&
        (Report == (NETTOOLS_STAT_RUNNING|DHCPCODE_IPADD) ||
         Report == (NETTOOLS_STAT_RUNNING|DHCPCODE_IPRENEW)) )
    {
        IPN IPTmp;

        // Manually add the DNS server when specified
        IPTmp = inet_addr(DNSServer);
        if( IPTmp )
            CfgAddEntry( 0, CFGTAG_SYSINFO, CFGITEM_DHCP_DOMAINNAMESERVER,
                         0, sizeof(IPTmp), (UINT8 *)&IPTmp, 0 );
                     
    }

    // Reset DHCP client service on failure
    if( Item==CFGITEM_SERVICE_DHCPCLIENT && (Report&~0xFF)==NETTOOLS_STAT_FAULT )
    {
        CI_SERVICE_DHCPC dhcpc;
        int tmp;

        // Get DHCP entry data (for index to pass to DHCP_reset).
        tmp = sizeof(dhcpc);
        CfgEntryGetData( h, &tmp, (UINT8 *)&dhcpc );

        // Create the task to reset DHCP on its designated IF
        // We must use TaskCreate instead of just calling the function as
        // we are in a callback function.
        TaskCreate( DHCP_reset, "DHCPreset", OS_TASKPRINORM, 0x1000,
                    dhcpc.cisargs.IfIdx, 1, 0 );
    }
}

void CheckDHCPOptions()
{
    char IPString[16];
    IPN  IPAddr;
    int  i, rc;

    // Now scan for DHCPOPT_SERVER_IDENTIFIER via configuration
    printf("\nDHCP Server ID:\n");
    for(i=1;;i++)
    {
        // Try and get a DNS server
        rc = CfgGetImmediate( 0, CFGTAG_SYSINFO, DHCPOPT_SERVER_IDENTIFIER,
                              i, 4, (UINT8 *)&IPAddr );
        if( rc != 4 )
            break;

        // Convert IP to a string:
        NtIPN2Str( IPAddr, IPString );
        printf("DHCP Server %d = '%s'\n", i, IPString);
    }
    if( i==1 )
        printf("None\n\n");
    else
        printf("\n");

    // Now scan for DHCPOPT_ROUTER via the configuration
    printf("Router Information:\n");
    for(i=1;;i++)
    {
        // Try and get a DNS server
        rc = CfgGetImmediate( 0, CFGTAG_SYSINFO, DHCPOPT_ROUTER,
                              i, 4, (UINT8 *)&IPAddr );
        if( rc != 4 )
            break;
             
       // We got something

        // Convert IP to a string:
        NtIPN2Str( IPAddr, IPString );
        printf("Router %d = '%s'\n", i, IPString);
    }
    if( i==1 )
        printf("None\n\n");
    else
        printf("\n");
}


//
//  UDP Server task
//

static void udp_test()
{
    SOCKET   recv = INVALID_SOCKET;     // Receiver socket
    SOCKET   send = INVALID_SOCKET;     // Sender socket. 
    struct   sockaddr_in sin1;          // Receiver socket address structure
    struct   sockaddr_in sout1;         // Sender socket address structure
    struct   timeval timeout;           // Timeout struct for select
    int      tmp,cnt;
    int sentCnt;
    int reply = 0;
    fd_set   ibits;

    while(!IPAddrCnt)
        TSK_sleep(100);

    // Join the Mulicast Group for MulticastAddr on interface "1"
        if (IGMPJoinHostGroup( inet_addr(MulticastAddr), 1 ))
            printf("Successfully joined the multicast group %s\n", MulticastAddr);

    // Allocate the file environment for this task
    fdOpenSession( TaskSelf() );

    // Create the multicast receive socket
    recv = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if( recv == INVALID_SOCKET )
        goto leave;

    // Create the multicast receive socket
    send = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if( recv == INVALID_SOCKET )
        goto leave;

    // Receive socket: set Port = 5001, IP address = ANY
    bzero( &sin1, sizeof(struct sockaddr_in) );
    sin1.sin_family = AF_INET;
    sin1.sin_len    = sizeof( sin1 );
    sin1.sin_port   = htons(5001);

    // Send socket: set Port = 5000, IP address = ANY
    bzero( &sout1, sizeof(struct sockaddr_in) );
    sout1.sin_family = AF_INET;
    sout1.sin_len    = sizeof( sout1 );
    sout1.sin_port   = htons(5000);

    // Bind receive socket
    if ( bind( recv, (PSA) &sin1, sizeof(sin1) ) < 0 )
        {
        printf("Receive: %d",fdError());
        goto leave;
        }

    // Bind send socket
    if ( bind( send, (PSA) &sout1, sizeof(sout1) ) < 0 )
        {
        printf("Send: %d",fdError());
        goto leave;
        }

    // Set timeout to 1 seconds
    timeout.tv_sec  = 1;
    timeout.tv_usec = 0;

    strcpy( sendBuffer, "Are you there?" );
        
    for(;;) {
         sout1.sin_addr.s_addr = inet_addr(MulticastAddr);
         sout1.sin_port = htons(5000);

         // Send to multicast group/port every 5 sec until successful
         do {
             TSK_sleep(5000);
             sentCnt = sendto( send, sendBuffer, 15, 0, &sout1, sizeof(sout1) );
         } while (sentCnt < 0);

        printf (" Message sent to multicast group \n");           
        
        // Check for network data on port 5001
        FD_ZERO(&ibits);
        FD_SET(recv, &ibits);

        tmp = fdSelect( 1, &ibits, 0, 0, &timeout );

        // See if there is data on the UDP socket...
        if( FD_ISSET(recv, &ibits) ) {
            // Read the data
            tmp = sizeof( sin1 );
            cnt = (int)recvfrom( recv, recvBuffer, 1500, 0, &sin1, &tmp );

            // See if the message was sent back
            if( cnt==11 && !strcmp( (char *)recvBuffer, "I am here!" ) ) {
                printf( "Reply #%d from: %d.%d.%d.%d\n", ++reply,
                     sin1.sin_addr.s_addr & 0xFF, 
                     (sin1.sin_addr.s_addr >> 8) & 0xFF,
                     (sin1.sin_addr.s_addr >> 16) & 0xFF, 
                     sin1.sin_addr.s_addr >> 24 );         
            }
        }

    }

leave:
    // We only get here on a fatal error - close the sockets
    if( recv != INVALID_SOCKET )
        fdClose( recv );

    if( send != INVALID_SOCKET )
        fdClose( send );
    
    printf("Fatal Error\n");

    // This task is killed by the system - here, we block
    TaskBlock( TaskSelf() );
}
