/**************************************************************************************************
    Filename:       hal_board.c
    Revised:        $Date: 2009-02-20 12:38:44 -0800 (Fri, 20 Feb 2009) $
    Revision:       $Revision: 19218 $

    Description: Board specific implementations for the MSP430FG4618 Experimenter's Board.

    Copyright 2006-2008 Texas Instruments Incorporated. All rights reserved.

    IMPORTANT: Your use of this Software is limited to those specific rights
    granted under the terms of a software license agreement between the user
    who downloaded the software, his/her employer (which must be your employer)
    and Texas Instruments Incorporated (the "License").  You may not use this
    Software unless you agree to abide by the terms of the License. The License
    limits your use, and you acknowledge, that the Software may not be modified,
    copied or distributed unless embedded on a Texas Instruments microcontroller
    or used solely and exclusively in conjunction with a Texas Instruments radio
    frequency transceiver, which is integrated into your product.  Other than for
    the foregoing purpose, you may not use, reproduce, copy, prepare derivative
    works of, modify, distribute, perform, display or sell this Software and/or
    its documentation for any purpose.

    YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
    PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
    INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
    NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
    TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
    NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
    LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
    INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
    OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
    OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
    (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

    Should you have any questions regarding your right to use this Software,
    contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/

/* ------------------------------------------------------------------------------------------------
 *                                          Includes
 * ------------------------------------------------------------------------------------------------
 */

#include "hal_board.h"
#include "hal_lcd.h"

/* ------------------------------------------------------------------------------------------------
 *                                           Constants
 * ------------------------------------------------------------------------------------------------
 */

// Count for ~20msec - ADC needs 17msec for refV to settle.
#define ADC_WAIT_MSECS         20

// Debounce for ~15msec.
#define DEBOUNCE_MSECS         15

// Timer comparator value for roughly 10 ms.
// Calculation based on ACLK = 32.768kHz
// Using divider value 8 -> 32.768 / 8 kHz * 0.01 sec = 40.96
#define TACCR0_INIT            41

// UART buffer polling period in miliseconds
#define AP_DELAY_SINCE_LAST_RX  1

// Max Rx count to trigger event - set to AP_MAX_BUF_LEN to disable.
#define AP_RX_FULL (AP_MAX_BUF_LEN-1)

/* ------------------------------------------------------------------------------------------------
 *                                           Macros
 * ------------------------------------------------------------------------------------------------
 */

#define RST_Clr()                (HAL_RESETCC_PORT |= HAL_RESETCC_BV)
#define RST_Set()                (HAL_RESETCC_PORT &= ~HAL_RESETCC_BV)

/* ------------------------------------------------------------------------------------------------
 *                                           Global Variables
 * ------------------------------------------------------------------------------------------------
 */

// Flags set by ISR's to indicate events without blocking or invoking background functions.
volatile uint16 halEventFlags;
uint8 halResetKeyState;

/* ------------------------------------------------------------------------------------------------
 *                                           Local Variables
 * ------------------------------------------------------------------------------------------------
 */

// Software timers w/ +/- 1 sec resolution and 65535 msec max.
static volatile uint16 tmrTicks[HAL_TIMER_CNT];
static          uint16 tmrPeriod[HAL_TIMER_CNT];
static volatile uint8 halDelayIsDone; // HAL delay done flag

/* ------------------------------------------------------------------------------------------------
 *                                           Local Functions
 * ------------------------------------------------------------------------------------------------
 */

       void halResetSlave( void );
static void halDioInit( void );
static void halMcuInit( void );
static void halTimerInit( void );

#if ((defined HAL_UART) && (HAL_UART == TRUE))

  void  halUARTInit( void );
  uint8 halUARTRead( uint8 port, uint8 *pBuf, uint8 len );
  void  halUARTWrite( uint8 port, uint8 *pBuf, uint8 len );
  bool  halUartTxEmpty( void );

  static volatile uint8 rx0Head;
  static uint8          rx0Tail;
  static uint8          rx0Buf[AP_MAX_BUF_LEN];
  static volatile uint8 rx0Time;

  static uint8          tx0Head;
  static volatile uint8 tx0Tail;
  static uint8          tx0Buf[AP_MAX_BUF_LEN];

  #define INTERRUPT_RX0() HAL_ISR_FUNCTION(isrRx0, USCIAB0RX_VECTOR)
  #define INTERRUPT_TX0() HAL_ISR_FUNCTION(isrTx0, USCIAB0TX_VECTOR)

#endif

#define INTERRUPT_PORT1()  HAL_ISR_FUNCTION(isrPort1, PORT1_VECTOR)
#define INTERRUPT_PORT2()  HAL_ISR_FUNCTION(isrPort2, PORT2_VECTOR)
#define INTERRUPT_TIMERA() HAL_ISR_FUNCTION(isrTimerA, TIMER1_A0_VECTOR)

/**************************************************************************************************
 * @fn          halBoardInit
 *
 * @brief       This is the HAL board main initialization function for ZASA.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void halBoardInit(void)
{
  // initialize some digital IO
  halDioInit();

  // disable watchdog and setup system clock to 8MHz
  halMcuInit();

  // setup TimerA for 1ms software interrupt; enable interrupts
  halTimerInit();

  // setup the LCD and clear it
  HalLcdInit();
  
  // Initial display
  HalLcdWriteString("RemoTI", HAL_LCD_LINE_1);
}

/**************************************************************************************************
 * @fn          halDioInit
 *
 * @brief       This function initializes the DIO.
 *              MSP430x2xx: 6.2.7 Configuring Unused Port Pins
 *              Unused I/O pins should be configured as I/O function, output direction,
 *              and left unconnected on the PC board, to prevent a floating input and reduce power
 *              consumption. The value of the PxOUT bit is a dont care.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void halDioInit(void)
{
  // assert RESETn
  HAL_RESETCC_PORT &= ~HAL_RESETCC_BV;
  HAL_RESETCC_DDR |= HAL_RESETCC_BV;

  // deassert MRDY
  SPI_MRDY_PORT |= SPI_MRDY_BV;
  SPI_MRDY_DDR |= SPI_MRDY_BV;

  // configure ISR on SRDY (falling edge)
  SPI_SRDY_IES |= SPI_SRDY_BV;

  // clear interrupt flags
  SPI_SRDY_IFG &= ~SPI_SRDY_BV;
  
  // enable port bits as interrupts
  SPI_SRDY_IE |= SPI_SRDY_BV;
  
  HAL_KEYS_INIT();

  // Read SW1 to determine whether user wants to reset persistent data such
  // as pairing entries
  if (HAL_KEYS_PORT & HAL_S1_BV)
  {
    halResetKeyState = 0;
  }
  else
  {
    halResetKeyState = 1;
  }

  // Set LED as outputs
  HAL_LED1_DDR |= HAL_LED1_BV;
  HAL_LED2_DDR |= HAL_LED2_BV;
}


/**************************************************************************************************
 * @fn          halResetSlave
 *
 * @brief       This function asserts the CC2430 RESETn for 10ms, deasserts, and
 *              waits 255ms for the slave to boot.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void halResetSlave( void )
{
  // reset the CC2430EM
  RST_Set();
  halDelay(10, TRUE);

  // release the CC2430EM from reset
  RST_Clr();
  halDelay(255, TRUE);
}  


/**************************************************************************************************
 * @fn          halMcuInit
 *
 * @brief       Turn off watchdog and set up system clock. Set system clock to 8 MHz.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void halMcuInit(void)
{
  volatile uint16 tmp = 0;

  WDTCTL = WDTPW | WDTHOLD;          // Disable the watchdog.

  //   ACLK = REFO = 32kHz, MCLK = SMCLK = 8MHz
  
  UCSCTL0 = 0x00;                           // Set lowest possible DCOx, MODx
  UCSCTL1 = DCORSEL_6;                      // Select range for 12MHz operation
  UCSCTL2 = 249;                            // Set DCO Multiplier for 8MHz
                                            // (N + 1) * FLLRef = Fdco
                                            // (249 + 1) * 32768 = 8MHz
  UCSCTL3 = SELREF_2;                       // Set DCO FLL reference = REFO
  UCSCTL4 = SELS_3 + SELM_3;                // Set MCLK = DCOCLK

  /* CPU15 Errata workaround */
  __bic_SR_register(SCG0);
}

/**************************************************************************************************
 * @fn          halTimerInit
 *
 * @brief       Configure timer hardware.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void halTimerInit(void)
{
  // halDelay is not requested yet.
  halDelayIsDone = TRUE;
  
  // Configure TimerA for 10 mili-second ISR for driving the HAL S/W timers.
  TA1CCR0 = TACCR0_INIT;      // Comparator value for 10ms
  TA1CTL = TASSEL_1 + ID_3 + MC_1;  // ACLK, divide by 8, up count mode.
  TA1CCTL0 = CCIE;                  // TACCR0 interrupt enabled
}


/**************************************************************************************************
 * @fn          halTimerSet
 *
 * @brief       This function sets the timer period and control flags as specified.
 *
 * input parameters
 *
 * @param       tIdx - HAL Timer index.
 * @param       period - Timer period in seconds (zero to turn off.)
 * @param       ctlFlags - Bit mask of timer control flags.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void halTimerSet(uint8 tIdx, uint16 period, uint8 ctlFlags)
{
  halIntState_t s;

  if (ctlFlags & HAL_TIMER_AUTO)
  {
    tmrPeriod[tIdx] = period * 1000;
  }
  else
  {
    tmrPeriod[tIdx] = 0;
  }

  HAL_ENTER_CRITICAL_SECTION(s);
  tmrTicks[tIdx] = period * 1000;
  halEventFlags &= ~BV(tIdx); // Clear the event flag
  HAL_EXIT_CRITICAL_SECTION(s);
}

/**************************************************************************************************
 * @fn          halDelay
 *
 * @brief       Delay for milliseconds.
 *              Do not invoke with zero.
 *              Do not invoke with greater than 255 msecs.
 *              Invoking with very high frequency and/or with long delays will start to
 *              significantly impact the real time performance of TimerA tasks because this will
 *              invisibly overrun the period when the TimerA count remaining, when this function
 *              is invoked, is less than the delay requested.
 *
 * input parameters
 *
 * @param       msecs - Milliseconds to delay in low power mode.
 * @param       sleep - Enforces blocking delay in low power mode if set.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void halDelay(uint8 msecs, uint8 sleep)
{
  // Expected scaling for msecs is 3/2, but depends on result of TimerA calibration.
  uint16 stop = msecs * TACCR0_INIT / 10;

  HAL_DISABLE_INTERRUPTS();
  TA1CTL &= ~MC_1;  // Stop the timer.
  halDelayIsDone = FALSE;

  if (TA1R != TACCR0_INIT)
  {
    stop += TA1R;
    if (stop == TACCR0_INIT)
    {
      stop++;
    }
  }
  TA1CCR0 = stop;

  TA1CTL |= MC_1;  // Re-start the timer.
  HAL_ENABLE_INTERRUPTS();

  if (sleep)
  {
    do {
#ifdef POWER_SAVING
      HAL_LOW_POWER_MODE();
#endif
    } while (!halDelayIsDone);
  }
}

/**************************************************************************************************
 * @fn          halDelayDone
 *
 * @brief       Check to determine if a requested HAL delay is done.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
uint8 halDelayDone(void)
{
  return halDelayIsDone;
}

/**************************************************************************************************
 * @fn          isrPort1
 *
 * @brief       This function services the SRDY interrupt.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
INTERRUPT_PORT1()
{
  // SRDY ISR:
  if (SPI_SRDY_IFG & SPI_SRDY_BV)
  {
    SPI_SRDY_IFG &= ~SPI_SRDY_BV;
    // ISR will exit from low power (at bottom of this routine) which will automatically poll slave.
  }
#ifdef POWER_SAVING
  __low_power_mode_off_on_exit();
#endif
}

/**************************************************************************************************
 * @fn          isrPort2
 *
 * @brief       This function services the Push Button interrupt.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
INTERRUPT_PORT2()
{
  uint8 ifg = HAL_KEYS_IFG & HAL_KEYS_BV;
  
  if (ifg)
  {
    HAL_KEYS_IE &= (uint8) ~(int)ifg; // Disable key press ISR while debouncing
    halDelay(DEBOUNCE_MSECS, FALSE);
    while (!halDelayDone());
    HAL_KEYS_IFG &= (uint8) ~(int)ifg;
    HAL_KEYS_IE |= ifg;
    halEventFlags |= ((uint16) ifg) << 8;
  }

#ifdef POWER_SAVING
  __low_power_mode_off_on_exit();
#endif
}

/**************************************************************************************************
 * @fn          isrTimerA
 *
 * @brief       This function services the Timer-A interrupt.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
INTERRUPT_TIMERA()
{
  uint8 idx;
  uint16 tickCount = 10;

  if (TA1CCR0 != TACCR0_INIT)
  {
#ifdef POWER_SAVING
    __low_power_mode_off_on_exit();
#endif
    if (TA1CCR0 < TACCR0_INIT)
    {
      TA1CCR0 = TACCR0_INIT;
      return;
    }
    TA1R = 0;
    tickCount = TA1CCR0 * 10 / TACCR0_INIT; // adjust tick count
    TA1CCR0 = TACCR0_INIT;
  }

  // Update all active S/W timers.
  for (idx = 0; idx < HAL_TIMER_CNT; idx++)
  {
    if (tmrTicks[idx] > tickCount)
    {
      // timer hasn't expired yet.
      // keep decrementing remaining tick count
      tmrTicks[idx] -= tickCount;
    }
    else if (tmrTicks[idx] != 0)
    {
      // timer has expired and it is still an active timer
      // (if tmrTicks[idx] == 0, it is inactive timer)
      
      if (tmrPeriod[idx] > 0)
      {
        if (tmrPeriod[idx] < tickCount)
        {
          // Not allowed to program halDelay in such a way as to
          // have tickCount exceed one second.
          volatile uint16 i;
          
          while (TRUE)
          {
            HAL_TURN_ON_RED();
            HAL_TURN_OFF_GRN();
            i = 1000;
            while (--i > 0);
            HAL_TURN_OFF_RED();
            HAL_TURN_ON_GRN();
            while (--i > 0);
          }
        }
        // Automatic restart for periodic timer
        // Timer should wrap around
        tmrTicks[idx] += tmrPeriod[idx] - tickCount;
      }
      else
      {
        // The timer was one-time timer.
        // Inactivate the timer.
        tmrTicks[idx] = 0;
      }
      // Trigger HAL event for the timer
      halEventFlags |= BV(idx);
    }
  }

  if (halDelayIsDone == FALSE)
  {
    halDelayIsDone = TRUE;
#ifdef POWER_SAVING
    __low_power_mode_off_on_exit();
#endif
  }
  else if (halEventFlags != 0)
  {
#ifdef POWER_SAVING
    __low_power_mode_off_on_exit();
#endif
  }
}
