/**************************************************************************************************
  Filename:       rcns_ap.c
  Revised:        $Date: 2008-11-04 17:27:21 -0800 (Tue, 04 Nov 2008) $
  Revision:       $Revision: 18382 $

  Description:    This file contains RF4CE network layer surrogate implementation
                  for application processor

  Copyright 2008 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/

/**************************************************************************************************
 *                                           Includes
 **************************************************************************************************/

/* RCN surrogate framework */
#include "rcns_af.h"

/* RTI includes */
#include "rcn_nwk.h"
#include "rcn_marshal.h"

/* NPI includes */
#include "npi.h"

/**************************************************************************************************
 *                                        Externals
 **************************************************************************************************/

/**************************************************************************************************
 *                                           Constant
 **************************************************************************************************/

/**************************************************************************************************
 *                                        Type definitions
 **************************************************************************************************/

/**************************************************************************************************
 *                                        Global Variables
 **************************************************************************************************/

/**************************************************************************************************
 *                                        Local Variables
 **************************************************************************************************/

/**************************************************************************************************
 *                                     Local Function Prototypes
 **************************************************************************************************/

/**************************************************************************************************
 * @fn          RCN_NldeDataAlloc
 *
 * @brief       This function is a special function to allocate a memory block
 *              where NSDU should be copied into before passing NLDE-DATA.Request primitive
 *              to network layer.
 *
 * input parameters
 *
 * @param       pPrimitive - a pointer to a C structure describing NLDE-Data.Request.
 *                          nsdu field of the C structure will be filled in before this function
 *                          returns if the function successfully allocated memory block.
 *
 * output parameters
 *
 * None.
 *
 * @return      either RCN_SUCCESS or RCN_ERROR_OUT_OF_MEMORY
 **************************************************************************************************
 */
RCNLIB_API uint8 RCN_NldeDataAlloc( rcnNldeDataReq_t *pPrimitive )
{
  rcnNldeDataReqStream_t *pSerialized = (rcnNldeDataReqStream_t *) afMalloc(
    sizeof(rcnNldeDataReqStream_t) + pPrimitive->nsduLength);

  if (!pSerialized)
  {
    return RCN_ERROR_OUT_OF_MEMORY;
  }

  afMemcpy(pSerialized, pPrimitive, sizeof(rcnNldeDataReqStream_t));
  pPrimitive->internal = (uint8 *) pSerialized;
  pPrimitive->nsdu = (uint8 *) (pSerialized + 1);

  return RCN_SUCCESS;
}

/**************************************************************************************************
 * @fn          RCN_NldeDataReq
 *
 * @brief       This function requests transfer of a data NSDU.
 *
 * input parameters
 *
 * @param       pPrimitive - a pointer to a C structure containing NLDE-Data.Request.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
RCNLIB_API void RCN_NldeDataReq( rcnNldeDataReq_t *pPrimitive )
{
  npiMsgData_t msgData;

  msgData.len = sizeof(rcnNldeDataReqStream_t) + pPrimitive->nsduLength;
  msgData.subSys = RPC_SYS_RCN;
  msgData.cmdId = RCN_NLDE_DATA_REQ;
  afMemcpy(msgData.pData, pPrimitive->internal,
    sizeof(rcnNldeDataReqStream_t) + pPrimitive->nsduLength);

  afFree(pPrimitive->internal);

  NPI_SendAsynchData(&msgData);
}

/**************************************************************************************************
 * @fn          RCN_NlmeDiscoveryReq
 *
 * @brief       This function requests discovery of other devices of interest operating in the
 *              personal operating space of the device.
 *
 * input parameters
 *
 * @param       pPrimitive - a pointer to a C structure containing NLME-Discovery.Request.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
RCNLIB_API void RCN_NlmeDiscoveryReq( rcnNlmeDiscoveryReq_t *pPrimitive )
{
  npiMsgData_t msgData;

  msgData.len = sizeof(*pPrimitive);
  msgData.subSys = RPC_SYS_RCN;
  msgData.cmdId = RCN_NLME_DISCOVERY_REQ;
  afMemcpy(msgData.pData, pPrimitive, sizeof(*pPrimitive));

  NPI_SendAsynchData(&msgData);
}

/**************************************************************************************************
 * @fn          RCN_NlmeDiscoveryRsp
 *
 * @brief       This function requests the local NLME to respond to a discovery request command.
 *
 * input parameters
 *
 * @param       pPrimitive - a pointer to a C structure containing NLME-Discovery.Response.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
RCNLIB_API void RCN_NlmeDiscoveryRsp( rcnNlmeDiscoveryRsp_t *pPrimitive )
{
  npiMsgData_t msgData;

  msgData.len = sizeof(*pPrimitive);
  msgData.subSys = RPC_SYS_RCN;
  msgData.cmdId = RCN_NLME_DISCOVERY_RSP;
  afMemcpy(msgData.pData, pPrimitive, sizeof(*pPrimitive));

  NPI_SendAsynchData(&msgData);
}

/**************************************************************************************************
 * @fn          RCN_NlmeGetReq
 *
 * @brief       This function retrieves an Network Information Base (NIB) attribute value.
 *
 * input parameters
 *
 * @param       attribute - NIB attribute enumeration
 * @param       attributeIndex - NIB attribute index in case NIB attribute is a table or an array
 *                               type
 *
 * output parameters
 *
 * @param       pValue     - pointer to a data block where to copy the attribute value into.
 *
 * @return      RCN_SUCCESS or RCN_ERROR_UNSUPPORTED_ATTRIBUTE
 **************************************************************************************************
 */
RCNLIB_API uint8 RCN_NlmeGetReq( uint8 attribute, uint8 attributeIndex, uint8 *pValue )
{
  npiMsgData_t msgData;

  // build message
  msgData.len = sizeof(rcnNlmeGetReq_t);
  msgData.subSys = RPC_SYS_RCN;
  msgData.cmdId = RCN_NLME_GET_REQ;
  {
    rcnNlmeGetReq_t *pGetReq = (rcnNlmeGetReq_t *) msgData.pData;

    pGetReq->attribute = attribute;
    pGetReq->attributeIndex = attributeIndex;
  }

  // Send the synchronous message and receive a response
  NPI_SendSynchData(&msgData);

  // check received message
  if (msgData.subSys != RPC_SYS_RCN_CLIENT ||
    msgData.cmdId != RCN_NLME_GET_CNF)
  {
    // error case
    return RCN_ERROR_COMMUNICATION;
  }

  {
    // parse received message
    rcnNlmeGetCnf_t *pGetCnf = (rcnNlmeGetCnf_t *) msgData.pData;

    if (pGetCnf->attribute != attribute)
    {
      return RCN_ERROR_COMMUNICATION;
    }
    if (pGetCnf->status == RCN_SUCCESS)
    {
      afMemcpy(pValue, pGetCnf + 1, pGetCnf->length);
    }
    return pGetCnf->status;
  }
}

/**************************************************************************************************
 * @fn          RCN_NlmeGetSizeReq
 *
 * @brief       This function retrieves an Network Information Base (NIB) attribute value size.
 *              In case the attribute is a table or an array, the returned size is that of an
 *              element.
 *
 * input parameters
 *
 * @param       attribute - NIB attribute enumeration
 *
 * output parameters
 *
 * NOne.
 *
 * @return      Attribute value size or 0 if attribute is not found
 **************************************************************************************************
 */
RCNLIB_API uint8 RCN_NlmeGetSizeReq( uint8 attribute )
{
  switch (attribute)
  {
  case RCN_NIB_NWK_ACTIVE_PERIOD:
  case RCN_NIB_NWK_DISCOVERY_REPETITION_INTERVAL:
  case RCN_NIB_NWK_DUTY_CYCLE:
  case RCN_NIB_NWK_RESPONSE_WAIT_TIME:
  case RCN_NIB_NWK_VENDOR_IDENTIFIER:
  case RCN_NIB_PAN_ID:
  case RCN_NIB_SHORT_ADDRESS:
    return 2;
  case RCN_NIB_NWK_FRAME_COUNTER: 
  case RCN_NIB_TRANSMIT_LATENCY:
    return 4;
  case RCN_NIB_IEEE_ADDRESS:
    return 8;
  case RCN_NIB_NWK_USER_STRING:
    return RCN_USER_STRING_LENGTH;
  case RCN_NIB_NWK_VENDOR_STRING:
    return RCN_VENDOR_STRING_LENGTH;
  case RCN_NIB_NWK_PAIRING_TABLE:
    return sizeof(rcnNwkPairingEntry_t);
  case RCN_NIB_ENERGY_SAMPLE:
    return 20;
  }
  return 1;
}

/**************************************************************************************************
 * @fn          RCN_NlmePairReq
 *
 * @brief       This function requests the local NLME to pair with another device.
 *
 * input parameters
 *
 * @param       pPrimitive - a pointer to a C structure containing NLME-PAIR.Request.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
RCNLIB_API void RCN_NlmePairReq( rcnNlmePairReq_t *pPrimitive )
{
  npiMsgData_t msgData;

  msgData.len = sizeof(*pPrimitive);
  msgData.subSys = RPC_SYS_RCN;
  msgData.cmdId = RCN_NLME_PAIR_REQ;
  afMemcpy(msgData.pData, pPrimitive, sizeof(*pPrimitive));

  NPI_SendAsynchData(&msgData);
}

/**************************************************************************************************
 * @fn          RCN_NlmePairRsp
 *
 * @brief       This function requests the local NLME to respond to a pairing request command.
 *
 * input parameters
 *
 * @param       pPrimitive - a pointer to a C structure containing NLME-PAIR.Response.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
RCNLIB_API void RCN_NlmePairRsp( rcnNlmePairRsp_t *pPrimitive )
{
  npiMsgData_t msgData;

  msgData.len = sizeof(*pPrimitive);
  msgData.subSys = RPC_SYS_RCN;
  msgData.cmdId = RCN_NLME_PAIR_RSP;
  afMemcpy(msgData.pData, pPrimitive, sizeof(*pPrimitive));

  NPI_SendAsynchData(&msgData);
}

/**************************************************************************************************
 * @fn          RCN_NlmeResetReq
 *
 * @brief       This function resets network layer.
 *
 * input parameters
 *
 * @param       setDefaultNib - If TRUE, the NWK layer is reset and all NIB attributes are set
 *                              to their default values.
 *                              If FALSE, the NWK layer is reset but all NIB attributes retain
 *                              their values prior to the generation of the NLME-RESET.request
 *                              primitive.
 *
 * output parameters
 *
 * @return      None.
 **************************************************************************************************
 */
RCNLIB_API void RCN_NlmeResetReq( uint8 setDefaultNib )
{
  npiMsgData_t msgData;

  msgData.len = sizeof(rcnNlmeResetReq_t);
  msgData.subSys = RPC_SYS_RCN;
  msgData.cmdId = RCN_NLME_RESET_REQ;
  {
    rcnNlmeResetReq_t *pResetReq = (rcnNlmeResetReq_t *)msgData.pData;

    pResetReq->setDefaultNib = setDefaultNib;
  }

  NPI_SendSynchData(&msgData);

  // received data must be reset confirm.
  // There is no way to return failure and hence no verification is performed
  // here.
}

/**************************************************************************************************
 * @fn          RCN_NlmeRxEnableReq
 *
 * @brief       This function requests that the receiver is either enabled (for a finite period
 *              or until further notice) or disabled..
 *
 * input parameters
 *
 * @param       rxOnDurationInMs - Duration in ms for which the receiver is to be enabled.
 *                                 If this parameter is equal to 0x0000,
 *                                 the receiver is to be disabled.
 *                                 If this parameter is equal to 0xffff,
 *                                 the receiver is to be enabled until further notice.
 *
 * output parameters
 *
 * @return      RCN_SUCCESS, MAC_PAST_TIME, MAC_ON_TIME_TOO_LONG or MAC_INVALID_PARAMETER
 **************************************************************************************************
 */
RCNLIB_API uint8 RCN_NlmeRxEnableReq( uint16 rxOnDurationInMs )
{
  npiMsgData_t msgData;

  // build primitive
  msgData.len = sizeof(rcnNlmeRxEnableReq_t);
  msgData.subSys = RPC_SYS_RCN;
  msgData.cmdId = RCN_NLME_RX_ENABLE_REQ;
  {
    rcnNlmeRxEnableReq_t *pRxEnableReq = (rcnNlmeRxEnableReq_t *)msgData.pData;

    pRxEnableReq->rxOnDurationInMs = rxOnDurationInMs;
  }

  // send the request and get confirmation
  NPI_SendSynchData(&msgData);

  // check received message
  if (msgData.subSys != RPC_SYS_RCN_CLIENT ||
    msgData.cmdId != RCN_NLME_RX_ENABLE_CNF)
  {
    // error case
    return RCN_ERROR_COMMUNICATION;
  }

  {
    // parse received message
    rcnNlmeRxEnableCnf_t *pRxEnableCnf = (rcnNlmeRxEnableCnf_t *) msgData.pData;

    return pRxEnableCnf->status;
  }
}

/**************************************************************************************************
 * @fn          RCN_NlmeSetReq
 *
 * @brief       This function sets an NIB attribute value.
 *
 * input parameters
 *
 * @param       attribute - The identifier of the NIB attribute to write.
 * @param       attributeIndex - index within a table or an array if the NIB attibute is a table
 *                               or an array
 *
 * @param       pValue - pointer to attribute value
 *
 * output parameters
 *
 * @return      RCN_SUCCESS, RCN_ERROR_UNSUPPORTED_ATTRIBUTE or RCN_ERROR_INVALID_INDEX
 **************************************************************************************************
 */
RCNLIB_API uint8 RCN_NlmeSetReq( uint8 attribute, uint8 attributeIndex, uint8 *pValue )
{
  npiMsgData_t msgData;
  uint8 valueLength;

  // build primitive
  valueLength = RCN_NlmeGetSizeReq(attribute);
  msgData.len = sizeof(rcnNlmeSetReq_t) + valueLength;
  msgData.subSys = RPC_SYS_RCN;
  msgData.cmdId = RCN_NLME_SET_REQ;
  {
    rcnNlmeSetReq_t *pSetReq = (rcnNlmeSetReq_t *)msgData.pData;

    pSetReq->nibAttribute = attribute;
    pSetReq->nibAttributeIndex = attributeIndex;
    pSetReq->length = valueLength;

    afMemcpy(pSetReq + 1, pValue, valueLength);
  }

  // send the request and get confirmation
  NPI_SendSynchData(&msgData);

  // check received message
  if (msgData.subSys != RPC_SYS_RCN_CLIENT ||
    msgData.cmdId != RCN_NLME_SET_CNF)
  {
    // error case
    return RCN_ERROR_COMMUNICATION;
  }

  {
    // parse received message
    rcnNlmeSetCnf_t *pSetCnf = (rcnNlmeSetCnf_t *) msgData.pData;

    return pSetCnf->status;
  }
}

/**************************************************************************************************
 * @fn          RCN_NlmeStartReq
 *
 * @brief       This function requests NLME to start network.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * @return      None.
 **************************************************************************************************
 */
RCNLIB_API void RCN_NlmeStartReq( void )
{
  npiMsgData_t msgData;

  msgData.len = 0;
  msgData.subSys = RPC_SYS_RCN;
  msgData.cmdId = RCN_NLME_START_REQ;

  NPI_SendAsynchData(&msgData);
}

/**************************************************************************************************
 * @fn          RCN_NlmeUnpairReq
 *
 * @brief       This function removes an entry from the pairing table.
 *              This function will trigger a callback of RCN_NLME_UNPAIR_CNF with
 *              the following status codes:
 *              RCN_SUCCESS, RCN_ERROR_NO_PAIRING, MAC_TRANSACTION_OVERFLOW,
 *              MAC_TRANSACTION_EXPIRED, MAC_CHANNEL_ACCESS_FAILURE, MAC_INVALID_ADDRESS,
 *              MAC_NO_ACK, MAC_COUNTER_ERROR, MAC_FRAME_TOO_LONG, MAC_UNAVAILABLE_KEY,
 *              MAC_UNSUPPORTED_SECURITY or MAC_INVALID_PARAMETER
 *
 * input parameters
 *
 * @param       pairingRef - The reference into the local pairing table of the entry
 *                           that is to be removed.
 *
 * output parameters
 *
 * None
 *
 * @return      None.
 **************************************************************************************************
 */
RCNLIB_API void RCN_NlmeUnpairReq( uint8 pairingRef )
{
  npiMsgData_t msgData;

  msgData.len = sizeof(rcnNlmeUnpairReq_t);
  msgData.subSys = RPC_SYS_RCN;
  msgData.cmdId = RCN_NLME_UNPAIR_REQ;
  {
    rcnNlmeUnpairReq_t *pUnpairReq = (rcnNlmeUnpairReq_t *) msgData.pData;

    pUnpairReq->pairingRef = pairingRef;
  }

  NPI_SendAsynchData(&msgData);
}


/**************************************************************************************************
 * @fn          RCN_NlmeUnpairRsp
 *
 * @brief       This function removes an entry from the pairing table in response to
 *              NLME-UNPAIR.indication.
 *
 * input parameters
 *
 * @param       pairingRef - The reference into the local pairing table of the entry
 *                           that is to be removed.
 *
 * output parameters
 *
 * @return      None
 **************************************************************************************************
 */
RCNLIB_API void RCN_NlmeUnpairRsp( uint8 pairingRef )
{
  npiMsgData_t msgData;

  msgData.len = sizeof(rcnNlmeUnpairRsp_t);
  msgData.subSys = RPC_SYS_RCN;
  msgData.cmdId = RCN_NLME_UNPAIR_RSP;
  {
    rcnNlmeUnpairRsp_t *pUnpairRsp = (rcnNlmeUnpairRsp_t *) msgData.pData;

    pUnpairRsp->pairingRef = pairingRef;
  }

  NPI_SendAsynchData(&msgData);
}

/**************************************************************************************************
 * @fn          RCN_NlmeDiscoveryAcceptReq
 *
 * @brief       This function configures the local NLME to respond to a discovery request command.
 *
 * input parameters
 *
 * @param       pPrimitive - a pointer to a C structure containing NLME-DISCOVERY-ACCEPT.request.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
RCNLIB_API void RCN_NlmeAutoDiscoveryReq( rcnNlmeAutoDiscoveryReq_t *pPrimitive )
{
  npiMsgData_t msgData;

  msgData.len = sizeof(*pPrimitive);
  msgData.subSys = RPC_SYS_RCN;
  msgData.cmdId = RCN_NLME_AUTO_DISCOVERY_REQ;
  afMemcpy(msgData.pData, pPrimitive, sizeof(*pPrimitive));

  NPI_SendAsynchData(&msgData);
}

/**************************************************************************************************
 * @fn          RCNS_AsynchMsgCback
 *
 * @brief       This function is a NPI callback to the client that inidcates an
 *              asynchronous message has been received. The client software is
 *              expected to complete this call.
 *
 *              Note: The client must copy this message if it requires it
 *                    beyond the context of this call.
 *
 * input parameters
 *
 * @param       *pMsg - A pointer to an asychronously received message.
 *              It is assumed that pMsg->subSys is already checked out to
 *              be RPC_SYS_RCN_CLIENT
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void RCNS_AsynchMsgCback( npiMsgData_t *pMsg )
{
  if ( pMsg->cmdId == RCN_NLDE_DATA_IND )
  {
    struct
    {
      uint8 eventId;
      rcnNldeDataInd_t dataInd;
    } cbackData;

    afMemcpy(&cbackData, &pMsg->cmdId, sizeof(cbackData));
    cbackData.dataInd.nsdu = &pMsg->pData[sizeof(rcnNldeDataIndStream_t)];

    RCN_CbackEvent( (rcnCbackEvent_t *) &cbackData );
  }
  else
  {
    RCN_CbackEvent( (rcnCbackEvent_t *) &pMsg->cmdId );
  }
}


/**************************************************************************************************
 **************************************************************************************************/

