/**************************************************************************************************
  Filename:       hal_board_cfg.h
  Revised:        $Date: 2009-02-20 12:38:44 -0800 (Fri, 20 Feb 2009) $
  Revision:       $Revision: 19218 $

  Description:    Describe the purpose and contents of the file.


  Copyright 2006-2008 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/

#ifndef HAL_BOARD_CFG_H
#define HAL_BOARD_CFG_H

/*
 *     =============================================================
 *     |                  Chipcon MSP-EXP430F5438                  |
 *     | --------------------------------------------------------- |
 *     |  mcu   : Texas Instruments MSP430F5438                    |
 *     |  clock : 12 MHz (Maximum 18MHz)                           |
 *     =============================================================
 */


/* ------------------------------------------------------------------------------------------------
 *                                           Includes
 * ------------------------------------------------------------------------------------------------
 */

#include <msp430.h>
#include "hal_types.h"
#include "hal_defs.h"

/* ------------------------------------------------------------------------------------------------
 *                             Interrupt Service Routine declaration macros
 * ------------------------------------------------------------------------------------------------
 */
#define _PRAGMA(x) _Pragma(#x)
#define HAL_ISR_FUNC_DECLARATION(f,v) _PRAGMA(vector=v) __interrupt void f(void)
#define HAL_ISR_FUNC_PROTOTYPE(f,v)   _PRAGMA(vector=v) __interrupt void f(void)
#define HAL_ISR_FUNCTION(f,v)         HAL_ISR_FUNC_PROTOTYPE(f,v); HAL_ISR_FUNC_DECLARATION(f,v)

/* ------------------------------------------------------------------------------------------------
 *                                       Pin Configuration
 * ------------------------------------------------------------------------------------------------
 */
// RESETCC (RESETn of network processor)
#define HAL_RESETCC_BV    BV(2)
#define HAL_RESETCC_PORT  P1OUT
#define HAL_RESETCC_DDR   P1DIR
#define HAL_RESETCC_SEL   P1SEL

// S1 Button
#define HAL_S1_BV         BV(6)

// S2 Button
#define HAL_S2_BV         BV(7)

// Joystick LEFT
#define HAL_JOYLEFT_BV    BV(1)

// Joystick RIGHT
#define HAL_JOYRIGHT_BV   BV(2)

// Joystick SELECT
#define HAL_JOYSEL_BV     BV(3)

// Joystick UP
#define HAL_JOYUP_BV      BV(4)

// Joystick DOWN
#define HAL_JOYDOWN_BV    BV(5)

// All keys
#define HAL_KEYS_PORT     P2IN
#define HAL_KEYS_IE       P2IE
#define HAL_KEYS_IFG      P2IFG
#define HAL_KEYS_IES      P2IES
#define HAL_KEYS_REN      P2REN
#define HAL_KEYS_OUT      P2OUT
#define HAL_KEYS_BV  \
  (HAL_S1_BV | HAL_S2_BV | HAL_JOYLEFT_BV | HAL_JOYRIGHT_BV | \
    HAL_JOYSEL_BV | HAL_JOYUP_BV | HAL_JOYDOWN_BV)

// SW key configuration
// Push button
#define HAL_KEYS_INIT() \
  do { \
    HAL_KEYS_REN |= HAL_KEYS_BV; /* pull up/down enable */ \
    HAL_KEYS_OUT |= HAL_KEYS_BV; /* pull down */ \
    HAL_KEYS_IES |= HAL_KEYS_BV; /* falling edge */ \
    HAL_KEYS_IFG &= ~HAL_KEYS_BV; /* clear interrupt flags */ \
    HAL_KEYS_IE |= HAL_KEYS_BV; /* enable interrupts */ \
  } while (FALSE)

// LED1
#define HAL_LED1_BV       BV(0)
#define HAL_LED1_PORT     P1OUT
#define HAL_LED1_DDR      P1DIR

// LED2
#define HAL_LED2_BV       BV(1)
#define HAL_LED2_PORT     P1OUT
#define HAL_LED2_DDR      P1DIR

// MRDY
#define SPI_MRDY_BV       BV(6)
#define SPI_MRDY_PORT     P1OUT
#define SPI_MRDY_DDR      P1DIR

// SRDY
#define SPI_SRDY_BV       BV(4)
#define SPI_SRDY_PORT     P1IN
#define SPI_SRDY_DDR      P1DIR
#define SPI_SRDY_IES      P1IES   // interrupt edge select
#define SPI_SRDY_IE       P1IE    // interrupt enable
#define SPI_SRDY_IFG      P1IFG   // interrupt flag

// SS
#define SPI_SS_BV         BV(0)
#define SPI_SS_PORT       P3OUT
#define SPI_SS_DDR        P3DIR
#define SPI_SS_SEL        P3SEL

// MOSI
#define SPI_MOSI_BV       BV(1)
#define SPI_MOSI_SEL      P3SEL
#define SPI_MOSI_DDR      P3DIR

// MISO
#define SPI_MISO_BV       BV(2)
#define SPI_MISO_SEL      P3SEL
#define SPI_MISO_DDR      P3DIR

// CLK
#define SPI_CLK_BV        BV(3)
#define SPI_CLK_SEL       P3SEL
#define SPI_CLK_DDR       P3DIR

// Transmit buffer
#define SPI_TXBUF         UCB0TXBUF

// Receive buffer
#define SPI_RXBUF         UCB0RXBUF


/* ------------------------------------------------------------------------------------------------
 *                              Board specific feature configurations
 * ------------------------------------------------------------------------------------------------
 */
#define HAL_UART          FALSE

#ifndef HAL_LCD
#define HAL_LCD           TRUE
#endif

/* ------------------------------------------------------------------------------------------------
 *                                          Constants
 * ------------------------------------------------------------------------------------------------
 */

/* Changing this constant requires the corresponding changes to both the HAL_TIMER_X bit flags
 * in halEventFlag_t and  the HAL_TIMER_Y index values in halTimerIdx_t.
 */
#define HAL_TIMER_CNT       8

// Flag bits for halTimerSet() arg.
#define HAL_TIMER_AUTO     0x01  // Auto-reset timer period on every expiration (i.e. real-time.)

// Flag bits for halTimerGet() arg.
#define HAL_TIMER_INTERVAL 0x01  // The timer period (zero if not auto-resetting.)
#define HAL_TIMER_TICKS    0x02  // Ticks of msecs left on the timer.

// Changing the position of the 1st timer flag requires a change to the timer update logic.
#define HAL_EVT_NONE       0x0000
#define HAL_EVT_TIMER0     0x0001
#define HAL_EVT_TIMER1     0x0002
#define HAL_EVT_TIMER2     0x0004
#define HAL_EVT_TIMER3     0x0008
#define HAL_EVT_TIMER4     0x0010
#define HAL_EVT_TIMER5     0x0020
#define HAL_EVT_TIMER6     0x0040
#define HAL_EVT_TIMER7     0x0080

#define HAL_EVT_JOY_LEFT   0x0200 // The joystick and button sequence is intentionally
#define HAL_EVT_JOY_RIGHT  0x0400 // set in this order to ease conversion from
#define HAL_EVT_JOY_SELECT 0x0800 // interrupt flag register bits
#define HAL_EVT_JOY_UP     0x1000 // Hence, DO NOT CHANGE.
#define HAL_EVT_JOY_DOWN   0x2000
#define HAL_EVT_BTN1_PRESS 0x4000
#define HAL_EVT_BTN2_PRESS 0x8000

#define HAL_EVT_TIMER_LED  HAL_EVT_TIMER0
#define HAL_EVT_TIMER_APP  HAL_EVT_TIMER1
#define HAL_EVT_TIMER_BTN  HAL_EVT_TIMER2
// Timers 3-7 are available.

#define HAL_PORT_NUM       0

/* ------------------------------------------------------------------------------------------------
 *                                           Typedefs
 * ------------------------------------------------------------------------------------------------
 */

typedef istate_t halIntState_t;

typedef enum {
  HAL_IDX_TIMER_LED,
  HAL_IDX_TIMER_APP,
  HAL_IDX_TIMER_BTN,
  HAL_IDX_TIMER3,
  HAL_IDX_TIMER4,
  HAL_IDX_TIMER5,
  HAL_IDX_TIMER6,
  HAL_IDX_TIMER7
} halTimerIdx_t;

/* ------------------------------------------------------------------------------------------------
 *                                            Macros
 * ------------------------------------------------------------------------------------------------
 */

#define HAL_LOW_POWER_MODE()            __low_power_mode_1()

#define HAL_ENABLE_INTERRUPTS()         asm("eint")
#define HAL_DISABLE_INTERRUPTS()        st( asm("dint"); asm("nop"); )
#define HAL_INTERRUPTS_ARE_ENABLED()    (__get_SR_register() & GIE)

#define HAL_ENTER_CRITICAL_SECTION(x) \
  st( x = __get_interrupt_state();  HAL_DISABLE_INTERRUPTS(); )

#define HAL_EXIT_CRITICAL_SECTION(x) __set_interrupt_state((x))

#define HAL_CRITICAL_STATEMENT(x) \
  st( halIntState_t s; HAL_ENTER_CRITICAL_SECTION(s); x; HAL_EXIT_CRITICAL_SECTION(s); )

#define HAL_BTN_PRESSED          ((HAL_SW1_PORT & HAL_SW1_BV) == 0)

#define HAL_TURN_OFF_RED()       (HAL_LED1_PORT &= ~HAL_LED1_BV)
#define HAL_TURN_OFF_GRN()       (HAL_LED2_PORT &= ~HAL_LED2_BV)

#define HAL_TURN_ON_RED()        (HAL_LED1_PORT |= HAL_LED1_BV)
#define HAL_TURN_ON_GRN()        (HAL_LED2_PORT |= HAL_LED2_BV)

#define HAL_TOGGLE_RED()         (HAL_LED1_PORT ^= HAL_LED1_BV)
#define HAL_TOGGLE_GRN()         (HAL_LED2_PORT ^= HAL_LED2_BV)

#define HAL_STATE_RED()          (HAL_LED1_PORT & HAL_LED1_BV)
#define HAL_STATE_GRN()          (HAL_LED2_PORT & HAL_LED2_BV)


// -- UCSI or USART SPI programming macros --

// Reset and configure mode for either UCSI or USART
//
// Reset.
// Set mode: clock phase: capture and change, inactive low, MSB first, master mode,
//           3-pin, synchronous (SPI) mode
// clock source select to SMCLK (10)
// baud is 9600 when used with these settings and a 32KHz clock
//   TURNS OUT EVEN THOUGH SMCLK IS SELECTED AND EVEN THOUGH THE MSP DOC SAYS THIS IS
//   TYPICALLY 2^20MHZ, THE ACTUAL CLOCK IS THE SYSTEM CLOCK WHICH IS 8MHZ
//   SO 8MHZ / 9600 BAUD = 833.33 (0x0341+0.33), BUT USING MODULATION OF ZERO IS CLOSER
//   TWEAKED IT DOWN BY ONE USING USBee PC Oscope
//   MAX SPI SPEED IS HALF OF SYS CLOCK, SO 4MHz. For now, use 1MHz (8MHz/1Mhz=8).
#define SPI_RESET_MODE() \
  st \
  ( \
    UCB0CTL1 |= UCSWRST; \
    UCB0CTL0 = UCCKPH | UCMSB | UCMST | UCSYNC; \
    UCB0CTL1 &= ~UCSSEL0; \
    UCB0CTL1 |= UCSSEL1; \
    UCB0BR0  = 0x08; \
    UCB0BR1  = 0x00; \
  )

// Clear the reset bit
#define SPI_CLEAR_RESET() \
  st \
  ( \
    UCB0CTL1 &= ~UCSWRST; \
  )

// Block till RX data is available
#define SPI_WAIT_RX() \
  do \
  { \
  } while (!(UCRXIFG & UCB0IFG))


/* ------------------------------------------------------------------------------------------------
 *                                        UART driver
 * ------------------------------------------------------------------------------------------------
 */
/* Get 1 byte from UART */
#define HAL_UART_GETBYTE() UCA1RXBUF

/* Put 1 byte into the UART */
#define HAL_UART_PUTBYTE(x)            { UCA1TXBUF = (x); }

/* Set Baud rate */
#define HAL_UART_SETBAUDRATE(baudrate) { UCA1BR1 = (baudrate) >> 8;  UCA1BR0 = (baudrate); }

/* Set Source Clock */
#define HAL_UART_SET_SRC_CLK()         { UCA1CTL1 |= UCSSEL_3; } /* SMCLK */

/* Setup TXD and RXD Port */
#define HAL_UART_PORT_CONFIG()         { P5SEL |= BV(6) | BV(7); P5DIR |= BV(6); P5DIR &= ~BV(7); }  /* P5.6, P5.7 - UCA0TXD and RXD */

/* Setup format frame */
#define HAL_UART_FRAME_CONFIG()        { UCA1CTL0 = UCMODE_0;   /* UART Mode */        \
                                         UCA1CTL0 &= ~UCPEN;    /* Disable parity */   \
                                         UCA1CTL0 &= ~UCSPB;    /* 1 stop bit */       \
                                         UCA1CTL0 &= ~UC7BIT;   /* 8bit data*/         \
                                         UCA1CTL0 &= ~UCSYNC; } /* Asynchronous mode */

/* Enable/Disable TX INT */
#define HAL_UART_TX_INT_ENABLE()       { UCA1IE |= UCTXIE; }
#define HAL_UART_TX_INT_DISABLE()      { UCA1IE &= ~UCTXIE; }

/* Enable/Disable RX */
#define HAL_UART_RX_ENABLE()           /* N/A */
#define HAL_UART_RX_DISABLE()          /* N/A */

/* Enable/Disable TX INT */
#define HAL_UART_RX_INT_ENABLE()       { UCA1IE |= UCRXIE; }
#define HAL_UART_RX_INT_DISABLE()      { UCA1IE &= ~UCRXIE; }

/* Enable/Disable SWRST */
#define HAL_UART_SWRST_ENABLE()        { UCA1CTL1 |= UCSWRST; }
#define HAL_UART_SWRST_DISABLE()       { UCA1CTL1 &= ~UCSWRST; }

/* Get Rx/Tx status bit */
#define HAL_UART_GET_RXTX_STATUS()    (UCA1IFG & (UCRXIFG | UCRXIFG))
#define HAL_UART_GET_RX_STATUS()      (UCA1IFG & UCRXIFG)
#define HAL_UART_GET_TX_STATUS()      (UCA1IFG & UCTXIFG)
#define HAL_UART_CLR_TX_STATUS()      (UCA1IFG &= ~UCTXIFG)

// Get UART board rate in relation to CPU clock speed 8MHz.
#define HAL_GET_UBRR(BAUD_BPS)   ((uint32)(8000000) / (uint32)BAUD_BPS)

// Interrupt service routine declaration macro
#define INTERRUPT_UART(fname)   HAL_ISR_FUNCTION( fname, USCI_A1_VECTOR /*USCIA1_VECTOR*/ )

/* ------------------------------------------------------------------------------------------------
 *           Board specific delay routines for SPI Serial Boot loader interface
 * ------------------------------------------------------------------------------------------------
 */
// Delay macro to insert between SPI transmissions in order to allow slow slave serial boot loader
// software to process the data correctly.
#define HAL_SPI_SB_MODE_TX_DELAY() \
  { \
    volatile uint8 i = 6; \
    while (--i); \
  }

// Delay macro to insert before SPI transmission to feed clock to receive frame from slave
// processor.
#define HAL_SPI_SB_MODE_PAUSE_BEFORE_RX() \
  { \
    volatile uint16 j = 30000; \
    while (--j); \
  }


/* ------------------------------------------------------------------------------------------------
 *                                     Board REV identification
 * ------------------------------------------------------------------------------------------------
 */
#define BOARD_ID_PIN                          BV(7)
#define BOARD_ID_PORT                         P8IN
#define HAL_BOARD_ID_CONFIG()             st( P8DIR &= ~BV(7); P8REN |= BV(7); P8OUT |= BV(7); )
#define HAL_BOARD_ID_DISABLE_PULLUP()     st( P8REN &= ~BV(7); )
#define HAL_MSP_EXP430F5438_REV_02()        ((BOARD_ID_PORT & BOARD_ID_PIN) != 0)
#define HAL_MSP_EXP430F5438_REV_03()        ((BOARD_ID_PORT & BOARD_ID_PIN) == 0)


/* ------------------------------------------------------------------------------------------------
 *                                           Global Variables
 * ------------------------------------------------------------------------------------------------
 */

// Flags set by ISR's to indicate events without blocking or invoking background functions.
extern volatile uint16 halEventFlags;
extern uint8 halResetKeyState;

/* ------------------------------------------------------------------------------------------------
 *                                          Functions
 * ------------------------------------------------------------------------------------------------
 */

/**************************************************************************************************
 * @fn          halBoardInit
 *
 * @brief       This is the HAL board main initialization function.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void halBoardInit(void);

/**************************************************************************************************
 * @fn          halTimerSet
 *
 * @brief       This function sets the timer period and control flags as specified.
 *
 * input parameters
 *
 * @param       tIdx - HAL Timer index.
 * @param       period - Timer period in seconds (zero to turn off.)
 * @param       ctlFlags - Bit mask of timer control flags.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void halTimerSet(uint8 tIdx, uint16 period, uint8 ctlFlags);

/**************************************************************************************************
 * @fn          halDelay
 *
 * @brief       Delay for milliseconds.
 *              Do not invoke with zero.
 *              Do not invoke with greater than 500 msecs.
 *              Invoking with very high frequency and/or with long delays will start to
 *              significantly impact the real time performance of TimerA tasks because this will
 *              invisibly overrun the period when the TimerA count remaining, when this function
 *              is invoked, is less than the delay requested.
 *
 * input parameters
 *
 * @param       msecs - Milliseconds to delay in low power mode.
 * @param       sleep - Enforces blocking delay in low power mode if set.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
void halDelay(uint8 msecs, uint8 sleep);

/**************************************************************************************************
 * @fn          halDelayDone
 *
 * @brief       Check to determine if a requested HAL delay is done.
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
uint8 halDelayDone(void);


/**************************************************************************************************
*/

#endif /* HAL_BOARD_CFG_H */



