/**************************************************************************************************
  Filename:       sap_controller.c
  Revised:        $Date: 2010-09-10 16:14:00 -0700 (Fri, 10 Sep 2010) $
  Revision:       $Revision: 23752 $

  Description:    

  This file contains the remote controller node implementation for the SimpleApp
  running on top of the RTI interface.

  Copyright 2010 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/


/**************************************************************************************************
 *                                           Includes
 **************************************************************************************************/

/* Hardware Abstraction Layer, HAL */
#include "hal_types.h"
#include "hal_key.h"
#include "hal_timer.h"
#include "hal_drivers.h"
#include "hal_led.h"
#include "hal_adc.h"
#include "hal_lcd.h"
#include "hal_uart.h"
#include "hal_assert.h"

/* Operating System Abstraction Layer, OSAL*/
#include "OSAL.h"
#include "OSAL_Tasks.h"
#include "OSAL_PwrMgr.h"
#include "OnBoard.h"

/* RemoTI, RTI */
#include "rti_constants.h"
#include "rti.h"

/* SimpleApp, SAP */
#include "sap.h"
#include "sap_platform.h"

/**************************************************************************************************
 *                                           Macros
 **************************************************************************************************/

/**************************************************************************************************
 *                                           Constant
 **************************************************************************************************/

// SimpeleApp events
#define SAP_EVT_INIT  0x0001    // Event to initialize node, stack and application framework

// SimpleApp states
enum
{
  SAP_STATE_NULL = 0,
  SAP_STATE_INIT,               // Node is initializing
  SAP_STATE_READY,              // Node ready for key inputs
  SAP_STATE_NDATA,              // Node is sending data
  SAP_STATE_PAIR                // Node performing paring
};

/**************************************************************************************************
 *                                          Typedefs
 **************************************************************************************************/

/**************************************************************************************************
 *                                        Global Variables
 **************************************************************************************************/

/**************************************************************************************************
 *                                        Local Variables
 **************************************************************************************************/

// Current state
uint8 sapState;

// SimpleApp Task ID
uint8 SAP_TaskId;

// Maximum number of pairing in device capability
uint8 sapMaxPairs;

// Current target device pairing reference
uint8 sapDestIdx;


/**************************************************************************************************
 *                                     Local Function Prototypes
 **************************************************************************************************/

// OSAL Related
void SAP_Init(uint8 taskId);
uint16 SAP_ProcessEvent(uint8 taskId, uint16 events);

// HAL Related
void SAP_KeyCback(uint8 keys, uint8 state);

// SAP related
void sapStartup(void);
static void sapSelectDestIdx(void);

/**************************************************************************************************
 *
 * @fn          SAP_Init
 *
 * @brief       Initialize the application
 *
 * @param       taskId - taskId of the task after it was added in the OSAL task queue
 *
 * @return      none
 *
 **************************************************************************************************/
void SAP_Init(uint8 taskId)
{

  // Turn on LED2 (red) to indicated the controller is initializing
  HalLedSet(HAL_LED_2, HAL_LED_MODE_ON);
  
  // LCD information
  HalLcdWriteString ("RemoTI", HAL_LCD_LINE_1);
  HalLcdWriteString ("SimpleApp", HAL_LCD_LINE_2);
  HalLcdWriteString ("Controller Node", HAL_LCD_LINE_3);
 
  // Initialize the task id assigned by OSAL
  SAP_TaskId = taskId;
  
  // No valid paring yet
  sapDestIdx = RTI_INVALID_PAIRING_REF;
  
  // Set SimpleApp state to init until intialization is done
  sapState = SAP_STATE_INIT;
  
  // This event will trigger node, application framwork and stack intitalizaion
  osal_set_event(SAP_TaskId, SAP_EVT_INIT);
}


/**************************************************************************************************
 *
 * @fn          SAP_ProcessEvent
 *
 * @brief       This routine handles events
 *
 * @param       taskId - ID of the application task when it registered with the OSAL
 *              events - Events for this task
 *
 * @return      16bit - Unprocessed events
 *
 **************************************************************************************************/
uint16 SAP_ProcessEvent(uint8 taskId, uint16 events)
{
  uint8* pMsg;
  
  (void) taskId; // unused argument

  // System Events
  if (events & SYS_EVENT_MSG)
  {
    while ((pMsg = osal_msg_receive(SAP_TaskId)) != NULL)
    {
      /* Deallocate */
      osal_msg_deallocate((uint8 *) pMsg);
    }

    return events ^ SYS_EVENT_MSG;
  }

  // SimpleApp events
  if (events & SAP_EVT_INIT)
  {
    // This event should only happen in initialization state
    if (sapState == SAP_STATE_INIT)
    {
      // Configure node parameters
      sapStartup();

      // Start RemoTI application framework and stack
      RTI_InitReq();
    }
  }

  return 0;
}


/**************************************************************************************************
 *
 * @fn      RTI_InitCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_InitReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_InitReq has returned.
 *
 * @param   status - Result of RTI_InitReq API call.
 *
 * @return  void
 *
 **************************************************************************************************/
void RTI_InitCnf(rStatus_t status)
{
  if (status == RTI_SUCCESS)
  {
    uint8 value;

    // Now that RTI_InitReq() is done, change startup control settings so that
    // it will restore state from next time on.
    value = RESTORE_STATE;
    RTI_WriteItem(RTI_CP_ITEM_STARTUP_CTRL, 1, &value);

    // Set the first pairing entry index, and turn on LED3 (yellow) if one exists
    sapSelectDestIdx();

    // RTI has been successfully started, application is ready for key events
    sapState = SAP_STATE_READY;
    
    //Turn off LED2 (red) to indicate startup is done
    HalLedSet(HAL_LED_2, HAL_LED_MODE_OFF);
  }
  else
  {
    //We should not get here, but trap in case we do
    HAL_ASSERT_FORCED();
  }

}



/**************************************************************************************************
 *
 * @fn      RTI_PairCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_PairReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_PairReq has returned.
 *
 * @param   status - Result of RTI_PairReq API call.
 *
 * @return  void
 *
 **************************************************************************************************/
void RTI_PairCnf(rStatus_t status, uint8 dst, uint8 dev)
{
  (void) dev; // unused argument
  
  // Check if pairing was sucessful
  if (status == RTI_SUCCESS)
  {
    // Set simpleApp pairing reference to current pairing entry
    sapDestIdx = dst;

    // Turn on LED3 (yellow) to indicate successful pairing
    HalLedSet(HAL_LED_3, HAL_LED_MODE_ON);
  }
  else
  {
    // Turn off LED3 (yellow) to indicate unsuccessful pairing
    HalLedSet(HAL_LED_3, HAL_LED_MODE_OFF);
  }
  
  //  The application is now ready for key inputs
  sapState = SAP_STATE_READY;
}


/**************************************************************************************************
 *
 * @fn      RTI_AllowPairCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_AllowPairReq API
 *          call. The client is expected to complete this function.
 *
 * @param   status - Result of RTI_AllowPairReq API call.
 * @param   dstIndex - pairing table entry index for newly paired peer device
 * @param   devType - device type of the newly paired peer device
 *
 * @return  void
 *
 **************************************************************************************************/
void RTI_AllowPairCnf(rStatus_t status, uint8 dstIndex, uint8 devType)
{
  // This node is configured as controller and will not issue a RTI_AllowPairReq
  
  // unused arguments
  (void) status;
  (void) dstIndex;
  (void) devType;
}

/**************************************************************************************************
 *
 * @fn      RTI_PairAbortCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_PairAbortReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_PairAbortReq has returned.
 *
 * @param   status - Result of RTI_PairAbortReq API call.
 *
 * @return  void
 */
void RTI_PairAbortCnf( rStatus_t status )
{
  // unused arguments
  (void) status;
}


/**************************************************************************************************
 *
 * @fn      RTI_UnpairCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_UnpairReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_UnpairReq has returned.
 *
 * @param   status   - Result of RTI_PairReq API call.
 * @param   dstIndex - Pairing table index of paired device, or invalid.
 *
 * @return  void
 */
void RTI_UnpairCnf( rStatus_t status, uint8 dstIndex )
{
  // unused arguments
  (void) status;
  (void) dstIndex;
}


/**************************************************************************************************
 *
 * @fn      RTI_UnpairInd
 *
 * @brief   RTI indication callback initiated by receiving unpair request command.
 *
 * @param   dstIndex - Pairing table index of paired device.
 *
 * @return  void
 */
void RTI_UnpairInd( uint8 dstIndex )
{
  // unused arguments
  (void) dstIndex;
}


/**************************************************************************************************
 *
 * @fn      RTI_SendDataCnf
 *
 * @brief   RTI confirmation callback initiated by client's RTI_SendDataReq API
 *          call. The client is expected to complete this function.
 *
 *          NOTE: It is possible that this call can be made to the RTI client
 *                before the call to RTI_SendDataReq has returned.
 *
 * @param   status - Result of RTI_SendDataReq API call.
 *
 * @return  void
 *
 **************************************************************************************************/
void RTI_SendDataCnf(rStatus_t status)
{
  (void) status; // unused argument
  
  // Check that application is in data transfer state
  if (sapState == SAP_STATE_NDATA)
  {
    // Data transfer is complete so application is ready for key inputs again
    sapState = SAP_STATE_READY;
  }
  else
  {
    //We should not get here, but trap in case we do
    HAL_ASSERT_FORCED();
  }

  //Turn off LED1 (Green) to indicate data transfer was completed.
  HalLedSet(HAL_LED_1, HAL_LED_MODE_OFF);

}


/**************************************************************************************************
 *
 * @fn      RTI_ReceiveDataInd
 *
 * @brief   This function is used by the RTI stack to indicate to the Target application
 *          that data has been received from another RF4CE device.
 *          This function is to be completed by the application, and as such,
 *          constitutes a callback.
 *
 * @param   srcIndex - Specifies the index to the pairing table entry which contains
 *                     the information about the source node the data was received from.
 * @param   profileId - Specifies the identifier of the profile which indicates the format of
 *                      the received data.
 * @param   vendorId - Specifies the identifier of the vendor transmitting the data.
 * @param   rxLQI - Specifies the Link Quality Indication.
 * @param   rxFlags - Specifies the reception indication flags.
 *                    One or more of the following reception indication flags can be specified:
 *                      RTI_RX_FLAGS_BROADCAST
 *                      RTI_RX_FLAGS_SECURITY
 *                      RTI_RX_FLAGS_VENDOR_SPECIFIC
 * @param   len - Specifies the number of bytes of the received data.
 * @param   pData - Specifies a pointer to the received data.
 *
 * @return  void
 *
 **************************************************************************************************/
void RTI_ReceiveDataInd( uint8 srcIndex, uint8 profileId, uint16 vendorId, uint8 rxLQI, uint8 rxFlags, uint8 len, uint8 *pData )
{
  // This node is configured as controller node and is not expecting any data
  
  // unused arguments
  (void) srcIndex;
  (void) profileId;
  (void) vendorId;
  (void) rxLQI;
  (void) rxFlags;
  (void) len;
  (void) pData;
}

/**************************************************************************************************
 *
 * @fn      SAP_KeyCback
 *
 * @brief   Callback service for keys
 *
 * @param   keys  - key that was pressed (i.e. the scanned row/col index)
 *          state - shifted
 *
 * @return  void
 *
 **************************************************************************************************/
void SAP_KeyCback(uint8 keys, uint8 state)
{
  (void) state; // unused argument
  
  // Make sure we are ready to handle key commands
  if (sapState != SAP_STATE_READY)
  {
    return;
  }

  // Data Key
  if (keys & SAP_DATA_KEY)
  {
    // Only send data if we have a valid pairing
    if (sapDestIdx != RTI_INVALID_PAIRING_REF)
    {
      uint8 data[2];
      
      // Build CERC command message, using 'select' key
      data[0] = RTI_CERC_USER_CONTROL_PRESSED;
      data[1] = RTI_CERC_SELECT;
      
      // Set data transfer state
      sapState = SAP_STATE_NDATA;
      
      // Send data with multichannel, unicast, acknowledged transmit option
      RTI_SendDataReq(sapDestIdx,
                      RTI_PROFILE_CERC,
                      0, // vendor ID - only applicable for vendor specific data
                      RTI_TX_OPTION_ACKNOWLEDGED,
                      2,
                      data);

      //Turn on LED1 (green) to inciated data transfer in progress
      HalLedSet(HAL_LED_1, HAL_LED_MODE_ON);
    }
  }


  // Pair Key
  if (keys & SAP_PAIR_KEY)
  {
    // Request pairing, blink LED3 (yellow) as indication
    HalLedBlink (HAL_LED_3, 0, 50, 2000);
    sapState = SAP_STATE_PAIR;
    RTI_PairReq();
  }
}

/**************************************************************************************************
 *
 * @fn      sapStartup
 *
 * @brief   This function sets up the RTI configuration parameters and starts
 *          the RF4CE stack and network software.
 *
 * @param
 *
 * None.
 *
 * @return  void
 *
 **************************************************************************************************/
void sapStartup(void)
{
   uint8 pValue[MAX_AVAIL_DEVICE_TYPES]; // space for largest number of bytes, not counting strings
   
   // Configure node capabilities
   // (0 - Controller, 0 - battery powered, 1 - security capable, 0 - no channel normalizaton)
   pValue[0] = RTI_BUILD_NODE_CAPABILITIES(0,0,1,0);
   RTI_WriteItem(RTI_CP_ITEM_NODE_CAPABILITIES, 1, pValue);
   
   // Application Capability::Device Type List
   // Set application level device type as remote controller
   pValue[0] = RTI_DEVICE_REMOTE_CONTROL;
   RTI_WriteItem( RTI_CP_ITEM_APPL_DEV_TYPE_LIST, 1, pValue);
 
   // Node Capability::Supported Target Types
   // Configure controller capable of controlling TV, video player/rec and STB
   pValue[0] = RTI_DEVICE_TELEVISION;
   pValue[1] = RTI_DEVICE_VIDEO_PLAYER_RECORDER;
   pValue[2] = RTI_DEVICE_SET_TOP_BOX;
   RTI_WriteItem(RTI_CP_ITEM_NODE_SUPPORTED_TGT_TYPES, 3, pValue);
   
  // Reset state attributed if S1 is pushed down at power cycle
  if (HalKeyRead() & SAP_S1_KEY)
  {
    pValue[0] = CLEAR_STATE;
    RTI_WriteItem(RTI_CP_ITEM_STARTUP_CTRL, 1, pValue);
  }

  // Read size of pairing table for later use
  RTI_ReadItem(RTI_CONST_ITEM_MAX_PAIRING_TABLE_ENTRIES, 1, &sapMaxPairs);
  
}


/**************************************************************************************************
 * @fn          sapSelectDestIdx
 *
 * @brief       This function searched the pairing table and sets the sapDestIdx to last valid entry
 *
 * input parameters
 *
 * None.
 *
 * output parameters
 *
 * None.
 *
 * @return      None.
 **************************************************************************************************
 */
static void sapSelectDestIdx(void)
{
  uint8 i;
  rcnNwkPairingEntry_t sapPairingEntryCache;
  
  // Set default pairing referecnce to invalid before search
  sapDestIdx = RTI_INVALID_PAIRING_REF;
  
  // Search entire pairing table and initiate with last valid entry
  for (i = 0; i < sapMaxPairs; i++)
  {
    // Set pairing table index before calling RTI_ReadItem below
    if (RTI_WriteItem( RTI_SA_ITEM_PT_CURRENT_ENTRY_INDEX, 1, &i ) != RTI_SUCCESS)
    {
      // We should not get here, but trap in case we do.
      HAL_ASSERT_FORCED();
    }
 
    //This call will return RTI_SUCCESS if pairing table entry is valid
    if (RTI_ReadItem( RTI_SA_ITEM_PT_CURRENT_ENTRY, sizeof(sapPairingEntryCache),
                     (uint8 *) &sapPairingEntryCache ) == RTI_SUCCESS)
    {
      // Set SimpleApp destination index with valid pairing entry
      sapDestIdx = i;
      
      // Turn on LED3 (yellow) to indicate the controller has a valid pairing
      HalLedSet(HAL_LED_3, HAL_LED_MODE_ON);
    }
  }
}
