/**************************************************************************************************
Filename:       irg_target_RC5.c
Revised:        $Date: 2010-06-25 19:02:41 -0700 (Fri, 25 Jun 2010) $
Revision:       $Revision: 22827 $

Description:    This file contains the a sample application showcasing IR
signal generation of the RC5 IR format

Copyright 2010 Texas Instruments Incorporated. All rights reserved.

Copyright 2008-2009 Texas Instruments Incorporated. All rights reserved.

IMPORTANT: Your use of this Software is limited to those specific rights
granted under the terms of a software license agreement between the user
who downloaded the software, his/her employer (which must be your employer)
and Texas Instruments Incorporated (the "License").  You may not use this
Software unless you agree to abide by the terms of the License. The License
limits your use, and you acknowledge, that the Software may not be modified,
copied or distributed unless embedded on a Texas Instruments microcontroller
or used solely and exclusively in conjunction with a Texas Instruments radio
frequency transceiver, which is integrated into your product.  Other than for
the foregoing purpose, you may not use, reproduce, copy, prepare derivative
works of, modify, distribute, perform, display or sell this Software and/or
its documentation for any purpose.

YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
PROVIDED AS IS WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
(INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

Should you have any questions regarding your right to use this Software,
contact Texas Instruments Incorporated at www.TI.com.
**************************************************************************************************/

/**************************************************************************************************
*                                           Includes
**************************************************************************************************/

// Hardware Abstraction Layer, HAL
#include "hal_types.h"
#include "hal_key.h"
#include "hal_timer.h"
#include "hal_drivers.h"
#include "hal_led.h"
#include "hal_adc.h"
#include "hal_uart.h"
#include "hal_assert.h"

// IR singal generation for the RC5 format
#include "hal_irgen_RC5.h"

// Operating System Abstraction Layer, OSAL
#include "OSAL.h"
#include "OSAL_Tasks.h"
#include "OSAL_PwrMgr.h"
#include "OnBoard.h"

#include "osal_clock.h"

// RemoTI, RTI
#include "rti_constants.h"
#include "rti.h"

// IrGenApp, IRG
#include "irg.h"

/**************************************************************************************************
*                                           Macros
**************************************************************************************************/

/**************************************************************************************************
*                                           Constant
**************************************************************************************************/

// IrGenApp events
#define IRG_EVT_INIT                 0x0001  // Event to initialize node, stack and app framework
#define IRG_EVT_IR_START             0x0002  // Event to start IR generation
#define IRG_EVT_IR_REPEAT_TIMER      0x0004  // Event for IR repeat timer (114 ms)
#define IRG_EVT_IR_RELEASED_TIMER    0x0008  // Event for CERC button released detection timer
#define IRG_EVT_IR_COMPLETED         0x0010  // Event to indicates IR generation has completed

// Release detection timer in case release command is not received
#define IRG_RELEASE_DETECTION_TIME       100 // ms

// IrGenApp application states 
enum
{
  IRG_STATE_INIT = 0,           // Node is initializing
  IRG_STATE_READY,              // Node ready for key inputs
  IRG_STATE_PAIR                // Node performing paring
};

// IrGenApp IR generation states
enum
{
  IRG_STATE_IRGEN_READY = 0,  // Ready for immediate IR signal generation
  IRG_STATE_IRGEN_REPEAT      // Repeat the previous command
};

#define IRG_TOGGLE_BIT     1

/**************************************************************************************************
*                                          Typedefs
**************************************************************************************************/

/**************************************************************************************************
*                                        Global Variables
**************************************************************************************************/

/**************************************************************************************************
*                                        Local Variables
**************************************************************************************************/
// Command for IR signal generation
static uint16 irgCmd;

// Current IR signal generation state
static uint8 irgIrgenState;

// RC5 command toggle bit
static uint8 irgToggleBit;

// Current application state
static uint8 irgState;

// IrGenApp task ID
uint8 IRG_TaskId;

/**************************************************************************************************
*                                     Local Function Prototypes
**************************************************************************************************/

// OSAL Related
void   IRG_Init(uint8 taskId);
uint16 IRG_ProcessEvent(uint8 taskId, uint16 events);

// HAL Related
void IRG_KeyCback(uint8 keys, uint8 state);

// IRG related
void irgStartup(void);
static void irgCheckPairingTable(void);

/**************************************************************************************************
*
* @fn          IRG_Init
*
* @brief       Initialize the application
*
* @param       taskId - taskId of the task after it was added in the OSAL task queue
*
* @return      none
*
**************************************************************************************************/
void IRG_Init(uint8 taskId)
{
  // Turn on LED2 (red) to indicated target is initializing
  HalLedSet(HAL_LED_2, HAL_LED_MODE_ON);
  
  // Initialize the task id assigned by OSAL
  IRG_TaskId = taskId;
  
  // Initialize IR generation driver
  HalIrGenInitRc5();
  
  // Initialize IR toggle bit
  irgToggleBit = 0;
  
  // This event will trigger node, application framwork and stack intitalizaion
  osal_set_event(IRG_TaskId, IRG_EVT_INIT);
}

/**************************************************************************************************
*
* @fn          IRG_ProcessEvent
*
* @brief       This routine handles events
*
* @param       taskId - ID of the application task when it registered with the OSAL
*              events - Events for this task
*
* @return      16bit - Unprocessed events
*
**************************************************************************************************/
uint16 IRG_ProcessEvent(uint8 taskId, uint16 events)
{
  uint8* pMsg;
  
  (void) taskId; // unused argument
  
  // System Events
  if (events & SYS_EVENT_MSG)
  {
    while ((pMsg = osal_msg_receive(IRG_TaskId)) != NULL)
    {
      /* Deallocate */
      osal_msg_deallocate((uint8 *) pMsg);
    }
    
    return events ^ SYS_EVENT_MSG;
  }
  
  // IrGenApp events
  if (events & IRG_EVT_INIT)
  {
    // This event should only happen in initialization state
    if (irgState == IRG_STATE_INIT)
    {
      // Configure node parameters
      irgStartup();
      
      // Start RemoTI application framework and stack
      RTI_InitReq();
    }
  }
  
  // Start IR signal generation events
  if (events & IRG_EVT_IR_START)
  {
    // Start released detection timer
    osal_start_timerEx(IRG_TaskId, IRG_EVT_IR_RELEASED_TIMER, IRG_RELEASE_DETECTION_TIME);
    
    // Turn on LED2 (Red) when IR generation is in progress
    HalLedSet(HAL_LED_2, HAL_LED_MODE_ON);
    
    // Generate IR signal
    HalIrGenCommandRc5(irgCmd);
    
    // repeat the command till key release is detected or released timer expires
    irgIrgenState = IRG_STATE_IRGEN_REPEAT;
    
    // Start 114ms IR signal repeat timer    
    osal_start_timerEx(IRG_TaskId, IRG_EVT_IR_REPEAT_TIMER, HAL_IRGEN_RC5_REPEAT_INTERVAL);
  }
  
  
  // Repeat timer has expired, send repeat frame
  if (events & IRG_EVT_IR_REPEAT_TIMER)
  { 
    // Make sure we are in repeat state
    if (irgIrgenState == IRG_STATE_IRGEN_REPEAT)
    {
      // Turn on LED2 (Red) when IR generation is in progress
      HalLedSet(HAL_LED_2, HAL_LED_MODE_ON);
      
      // Generate repeat signa
      HalIrGenRepeatRc5();
      
      // restart repeat timer
      osal_start_timerEx(IRG_TaskId, IRG_EVT_IR_REPEAT_TIMER, HAL_IRGEN_RC5_REPEAT_INTERVAL);
    }
    else // This condition is not supposed to happen but added for completeness
    {
      irgIrgenState = IRG_STATE_IRGEN_READY;
    }
    
  }
  
  // Release detection timer has expired.  This means the ZRC release frame
  // was not received - we need to stop the IR repeat frame signal generation
  if (events & IRG_EVT_IR_RELEASED_TIMER) 
  {  
    // stop repetition
    if (irgIrgenState == IRG_STATE_IRGEN_REPEAT)
    {
      osal_stop_timerEx(IRG_TaskId, IRG_EVT_IR_REPEAT_TIMER);
      irgIrgenState = IRG_STATE_IRGEN_READY;
    }
  }
  
  // IR signal generation completed
  if (events & IRG_EVT_IR_COMPLETED) 
  {  
    // Turn off LED2 (Red)
    HalLedSet(HAL_LED_2, HAL_LED_MODE_OFF);
  }
  
  return 0;
}

/**************************************************************************************************
*
* @fn      RTI_InitCnf
*
* @brief   RTI confirmation callback initiated by client's RTI_InitReq API
*          call. The client is expected to complete this function.
*
*          NOTE: It is possible that this call can be made to the RTI client
*                before the call to RTI_InitReq has returned.
*
* @param   status - Result of RTI_InitReq API call.
*
* @return  void
*
**************************************************************************************************/
void RTI_InitCnf(rStatus_t status)
{
  if (status == RTI_SUCCESS)
  {
    uint8 value;
    
    // Now that RTI_InitReq() is done, change startup control settings so that
    // it will restore configured parameters after power cycle
    value = RESTORE_STATE;
    RTI_WriteItem(RTI_CP_ITEM_STARTUP_CTRL, 1, &value);
    
    // Turn on LED1 (green) according to pairing table content
    irgCheckPairingTable();
    
    // RTI has been successfully started, application is ready for key events
    irgState = IRG_STATE_READY;
    
    //Turn off LED2 (red) to inicated target has initialized
    HalLedSet(HAL_LED_2, HAL_LED_MODE_OFF);
    
  }
  else
  {
    //We should not get here, but trap in case we do
    HAL_ASSERT(0);
  }
}

/**************************************************************************************************
*
* @fn      RTI_PairCnf
*
* @brief   RTI confirmation callback initiated by client's RTI_PairReq API
*          call. The client is expected to complete this function.
*
*          NOTE: It is possible that this call can be made to the RTI client
*                before the call to RTI_PairReq has returned.
*
* @param   status - Result of RTI_PairReq API call.
*
* @return  void
*
**************************************************************************************************/
void RTI_PairCnf(rStatus_t status, uint8 dst, uint8 dev)
{
  // This node is configured as target and will not issue a RTI_PairReq. 
  // unused arguments
  (void) status;
  (void) dst;
  (void) dev;
}

/**************************************************************************************************
*
* @fn      RTI_AllowPairCnf
*
* @brief   RTI confirmation callback initiated by client's RTI_AllowPairReq API
*          call. The client is expected to complete this function.
*
* @param   status - Result of RTI_AllowPairReq API call.
* @param   dstIndex - pairing table entry index for newly paired peer device
* @param   devType - device type of the newly paired peer device
*
* @return  void
*
**************************************************************************************************/
void RTI_AllowPairCnf(rStatus_t status, uint8 dstIndex, uint8 devType)
{ 
  // unused arguments
  (void) status;
  (void) dstIndex;
  (void) devType;
  
  // Turn on LED1 (green) according to pairing table content
  irgCheckPairingTable();
  
  // The application is now ready for key inputs
  irgState = IRG_STATE_READY;
}

/**************************************************************************************************
*
* @fn      RTI_SendDataCnf
*
* @brief   RTI confirmation callback initiated by client's RTI_SendDataReq API
*          call. The client is expected to complete this function.
*
*          NOTE: It is possible that this call can be made to the RTI client
*                before the call to RTI_SendDataReq has returned.
*
* @param   status - Result of RTI_SendDataReq API call.
*
* @return  void
*
**************************************************************************************************/
void RTI_SendDataCnf(rStatus_t status)
{
  // This node is configured as target and will not issue a RTI_SendDataReq
  (void) status; // unused argument
}

/**************************************************************************************************
*
* @fn      RTI_PairAbortCnf
*
* @brief   RTI confirmation callback initiated by client's RTI_PairAbortReq API
*          call. The client is expected to complete this function.
*
*          NOTE: It is possible that this call can be made to the RTI client
*                before the call to RTI_PairAbortReq has returned.
*
* @param   status - Result of RTI_PairAbortReq API call.
*
* @return  void
*/
void RTI_PairAbortCnf(rStatus_t status)
{
  // unused arguments
  (void) status;
}

/**************************************************************************************************
*
* @fn      RTI_UnpairInd
*
* @brief   RTI indication callback initiated by receiving unpair request command.
*
* @param   dstIndex - Pairing table index of paired device.
*
* @return  void
*/
void RTI_UnpairInd(uint8 dstIndex)
{
  // unused arguments
  (void) dstIndex;
}

/**************************************************************************************************
*
* @fn      RTI_UnpairCnf
*
* @brief   RTI confirmation callback initiated by client's RTI_UnpairReq API
*          call. The client is expected to complete this function.
*
*          NOTE: It is possible that this call can be made to the RTI client
*                before the call to RTI_UnpairReq has returned.
*
* @param   status   - Result of RTI_PairReq API call.
* @param   dstIndex - Pairing table index of paired device, or invalid.
*
* @return  void
*/
void RTI_UnpairCnf( rStatus_t status, uint8 dstIndex )
{
  // unused arguments
  (void) status;
  (void) dstIndex;
}

/**************************************************************************************************
*
* @fn      RTI_ReceiveDataInd
*
* @brief   This function is used by the RTI stack to indicate to the Target application
*          that data has been received from another RF4CE device.
*          This function is to be completed by the application, and as such,
*          constitutes a callback.
*
* @param   srcIndex - Specifies the index to the pairing table entry which contains
*                     the information about the source node the data was received from.
* @param   profileId - Specifies the identifier of the profile which indicates the format of
*                      the received data.
* @param   vendorId - Specifies the identifier of the vendor transmitting the data.
* @param   rxLQI - Specifies the Link Quality Indication.
* @param   rxFlags - Specifies the reception indication flags.
*                    One or more of the following reception indication flags can be specified:
*                      RTI_RX_FLAGS_BROADCAST
*                      RTI_RX_FLAGS_SECURITY
*                      RTI_RX_FLAGS_VENDOR_SPECIFIC
* @param   len - Specifies the number of bytes of the received data.
* @param   pData - Specifies a pointer to the received data.
*
* @return  void
*
**************************************************************************************************/
void RTI_ReceiveDataInd(uint8 srcIndex, uint8 profileId, uint16 vendorId, uint8 rxLQI, uint8 rxFlags, uint8 len, uint8 *pData)
{
  // unused arguments
  (void) srcIndex;
  (void) vendorId;
  (void) rxLQI;
  (void) pData;
  
  // Check the data received is sent using CERC profile.  No vendor specific data is accepted
  if ((profileId == RTI_PROFILE_CERC) && (rxFlags & RTI_RX_FLAGS_VENDOR_SPECIFIC) == 0 && (len > 0))
  {
    // button pressed
    if (pData[0] == RTI_CERC_USER_CONTROL_PRESSED && irgIrgenState == IRG_STATE_IRGEN_READY)
    {
      
      switch (pData[1]) 
      {
      case RTI_CERC_PLAY: // Play button
        irgToggleBit ^= IRG_TOGGLE_BIT;  // 'Toggle' toggle bit on new command
        irgCmd = HAL_IRGEN_CMD_RC5(irgToggleBit, 0x13, 0x2C);
        break;
        
      default:
        return;
      }
      
      // Set event to start IR Signal generation
      osal_set_event(IRG_TaskId, IRG_EVT_IR_START);
           
    } // button repeated or released
    else if (pData[0] == RTI_CERC_USER_CONTROL_REPEATED && irgIrgenState == IRG_STATE_IRGEN_REPEAT)
    {
      // restart released detection timer
      osal_start_timerEx(IRG_TaskId, IRG_EVT_IR_RELEASED_TIMER, IRG_RELEASE_DETECTION_TIME);
    } 
    else if (pData[0] == RTI_CERC_USER_CONTROL_RELEASED)
    {
      // stop release detection and repeat timer
      osal_stop_timerEx(IRG_TaskId, IRG_EVT_IR_RELEASED_TIMER);
      osal_stop_timerEx(IRG_TaskId, IRG_EVT_IR_REPEAT_TIMER);
      
      // Ready for IR generation
      irgIrgenState = IRG_STATE_IRGEN_READY;
    } 
  }
}

/**************************************************************************************************
*
* @fn      RTI_DisableSleepCnf
*
* @brief   RTI confirmation callback initiated by client's RTI_DisableSleepReq API
*          call. The client is expected to complete this function.
*
* @param   status - Result of RTI_EnableSleepReq API call.
*
* @return  void
*
*/
void RTI_DisableSleepCnf(rStatus_t status)
{
  // Do nothing
  (void) status; // unused argument
}

/**************************************************************************************************
*
* @fn      RTI_EnableSleepCnf
*
* @brief   RTI confirmation callback initiated by client's RTI_EnableSleepReq API
*          call. The client is expected to complete this function.
*
* @param   status - Result of RTI_EnableSleepReq API call.
*
* @return  void
*
*/
void RTI_EnableSleepCnf(rStatus_t status)
{
  // Do nothing
  (void) status; // unused argument
}

/**************************************************************************************************
*
* @fn      IRG_KeyCback
*
* @brief   Callback service for keys
*
* @param   keys  - key that was pressed (i.e. the scanned row/col index)
*          state - shifted
*
* @return  void
*
**************************************************************************************************/
void IRG_KeyCback(uint8 keys, uint8 state)
{
  (void) state; // unused argument
  
  // S1 is pressed
  if (keys & HAL_KEY_SW_1)
  {
    if (irgState == IRG_STATE_PAIR)
    {  
      // Abort pairing if we are in pairing state
      RTI_AllowPairAbortReq();
           
      // The application is now ready for key inputs
      irgState = IRG_STATE_READY; 
      
      // Turn on LEDs according to pairing table content
      irgCheckPairingTable();
      
      return;
    }
    else if (irgState == IRG_STATE_READY)
    {
      // Blink LED1 (Green) when in pairing mode
      HalLedBlink (HAL_LED_1, 0, 50, 2000);
      
      // Start pairing
      irgState = IRG_STATE_PAIR;
      RTI_AllowPairReq();
    }
  }
}

/**************************************************************************************************
*
* @fn      irgStartup
*
* @brief   This function sets up the RTI configuration parameters
*
* @param   
*
* None.
*
* @return  void
*
**************************************************************************************************/
void irgStartup( void )
{
  uint8 pValue[MAX_AVAIL_DEVICE_TYPES]; // space for largest number of bytes, not counting strings
  
  // Configure node capabilities
  // (1 - Target, 1 - AC powered, 1 - Security capable, 0 - no channel normalizaton)
  pValue[0] = RTI_BUILD_NODE_CAPABILITIES(1,1,1,0); 
  RTI_WriteItem(RTI_CP_ITEM_NODE_CAPABILITIES, 1, pValue);
  
  // Application Capability::Device Type List
  // Set application level device type as TV
  pValue[0] = RTI_DEVICE_TELEVISION;
  RTI_WriteItem(RTI_CP_ITEM_APPL_DEV_TYPE_LIST, 1, pValue);
}

/**************************************************************************************************
*
* @fn      irgCheckPairingTable
*
* @brief   This function checks the if the pairing table has a valid entry and 
*          sets LED1 (green) accordingly
*
* @param   
*
* None.
*
* @return  void
*
**************************************************************************************************/
void irgCheckPairingTable(void)
{
  uint8 pairing;
  
  // Get the number of active pairing entries
  if (RTI_ReadItem(RTI_SA_ITEM_PT_NUMBER_OF_ACTIVE_ENTRIES, 1, &pairing)
      == RTI_SUCCESS && pairing)
  { 
    // Set LED1 (Green) to solid to indicate successful pairing
    HalLedSet(HAL_LED_1, HAL_LED_MODE_ON);
  }
  else
  {
    // Tune off LED1 (Green) to indicate no pairing entry
    HalLedSet(HAL_LED_1, HAL_LED_MODE_OFF);     
  }
}

/******************************************************************************
 * @fn      HalIrGenIsrCback
 *
 * @brief   This function is a callback function that a user application has to
 *          define. HAL IR signal generation driver will call this callback
 *          function upon completion of IR signal generation.
 *
 *          NOTE that this function is called in the ISR context and should as
 *          such do a minimum of processsing. A prudent design will set an
 *          application event to delay further processing
*
* input parameters
*
* None
*
* output parameters
*
* None.
*
* @return  None.
*
*/
void HalIrGenIsrCback(void)
{
  // Set event to avoid processing in isr
  osal_set_event(IRG_TaskId, IRG_EVT_IR_COMPLETED);
}
