% This Program is developed to simplify spurious emission measurement.
% The page references found here references to:
% TELEC-T245
% PARAMETER TEST METHOD OF 920MHz-BAND TELEMETER, TELECONTROL
% AND DATA TRANSMISSION RADIO EQUIPMENT FOR SPECIFIED LOW
% POWER RADIO STATION Rev4.0 2012

% Note that gpibio.m is needed. It can be found on the internet but small
% changes are made. For example, buffersize is set to 20000
% (enables max 1112 datapoints) instead of 10000 (max 556 datapoints).

clear all;
disp(' ');


%% gpib config.
BoardNumber = 0;        %The GPIB controller card number
DeviceNumber = 19;      %This is device settable
dev = gpibio(BoardNumber,DeviceNumber,0,13,0,0);

%Reset dev
dev.scmd(['*RST;*CLS';]);
pause(1);

disp(' ');
disp('###################################################################');
disp('Spurious emission measurement, close in');
disp('Carrier (915 to 930 MHz):');
disp('You can use Ctrl+C to abort anytime.');
Fc = input('Set center frequency in MHz: ');
ucBW = input('Set unit channel BW in MHz: ');
uc = input('Set number of used unit channels for carrier: ');
cLoss = input('Cable loss etc [dB]: ');
auto_center = input('Auto adjust center frequency? [y/n]: ', 's');
disp(' ');

% Max data points = 1112 when buffersize is set to 20000.
datap = 1001;


%% One single sweep for calculation
disp('Sweeping carrier band...');
% Sw: Sweep frequency band (MHz)
Sw = ucBW*uc;
Fmin = Fc - Sw/2;
Fmax = Fc + Sw/2;
% RBW: Resolution bandwidth (MHz)
RBW = 0.003;
%Spec_an_setup(gpib,Fmin,Fmax,RBW, VBW,datap,contsweep,detmode,mxhld, reflvl, atten)
Spec_an_setup(dev, Fmin, Fmax, RBW, RBW, datap, 'ON', 'POSP', 'ON', 30, 'AUTO');
% Read from dev
%             Read_data(gpib, Fmin, Fmax, wait_stable)
[Pc_ysingle, Pc_xsingle] = Read_data(dev, Fmin, Fmax, cLoss, 'OFF');
peak_lvl = round((max(Pc_ysingle)+10)/10)*10;
%peak_lvl = max(Pc_ysingle);

if(strcmp(auto_center, 'y') || strcmp(auto_center, 'Y'))
    Fc = Center_freq(Pc_xsingle, Pc_ysingle, max(Pc_ysingle));
    disp(['Center frequency Fc = ', num2str(Fc), ' MHz.']);
end
disp(' ');


%% Pc 4(8)- page 18, Observe that the measure order of Pc, Pb, Ps is changed
disp('Sweeping carrier band...');
% Sw: Sweep frequency band (MHz)
Sw = ucBW*uc;
Fmin = Fc - Sw/2;
Fmax = Fc + Sw/2;
% RBW: Resolution bandwidth (MHz)
RBW = 0.003;
% Set up according to 2(6) page 16
%Spec_an_setup(gpib,Fmin,Fmax,RBW, VBW,datap,contsweep,detmode,mxhld, reflvl, atten)
Spec_an_setup(dev, Fmin, Fmax, RBW, RBW, datap, 'ON', 'POSP', 'ON', peak_lvl, 'AUTO');
% Read from dev
%             Read_data(gpib, Fmin, Fmax, wait_stable)
[Pc_y, Pc_x] = Read_data(dev, Fmin, Fmax, cLoss, 'ON');
% Calculate average
%    Calc_avg(data,Sw,RBW, k)
Pc = Calc_avg(Pc_y, Sw, RBW, 33.1131);

disp(['Carrier Pc = ', num2str(Pc), ' mW.']);
disp(' ');
% disp('Please wait...');
% pause(2);


%% 4(4) page 18
% Guard calculation
if ((Fc > 915.9) & (Fc < 916.9))
    Fguard = 0.1 + 0.1*uc;
elseif ((Fc > 920.5) & (Fc < 922.3))
    Fguard = 0.1 + 0.1*uc;
elseif ucBW == 0.2
    Fguard = 0.2 + 0.1*uc;
elseif ucBW == 0.1
    Fguard = 0.1 + 0.05*uc;
end
% Sweep from 915 MHz to the first carrier unit channel (left).
disp('Sweeping from 915 MHz to the first carrier unit channel (left)...');
% Sw: Sweep frequency band (MHz)
Sw = ucBW*uc;
Sw_sp = 0.1; % Defined 100kHz for spurious measurements 
Fmin = 915;
Fmax = Fc - Fguard;
% RBW: Resolution bandwidth (MHz)
RBW = 0.003;

% Technical standard page 21
% -36dBm/100kHz = 2.5119e-04/100kHz = 2.5119e-03/MHz
% std_lvl = 2.5119e-03/MHz * RBW, [RBW=0.003] gives 7.5357e-06 = -51.2288 dBm
std_lvl = 10*log10(2.5119e-03 * RBW);

% Set up according to 2(5) page 15
% Attenuation = 'AUTO' from current reference level. Use same attenuation
% for the rest of the sweeps.
att = dev.ask(['INP:ATT?']);
%Spec_an_setup(gpib,Fmin,Fmax,RBW, VBW,datap,contsweep,detmode,mxhld, reflvl, atten)
Spec_an_setup(dev, Fmin, Fmax, RBW, RBW, datap, 'OFF', 'POSP', 'ON', peak_lvl, att);
% Read from dev
%             Read_data(gpib, Fmin, Fmax, wait_stable)
[P_y1, P_x1] = Read_data(dev, Fmin, Fmax, cLoss, 'OFF');
% Find crossover frequency where the standard value is crossed
%          Find_cross_freq(y_val, x_val, lvl, ascending);
F1_cross = Find_cross_freq(P_y1, P_x1, std_lvl, 1);
% Needed number of 100kHz cannels to the left.
sp_uc1 = ceil((Fmax - F1_cross)/Sw_sp);

% Sweep from the last carrier unit channel to 930 MHz (right).
disp('Sweeping from the last carrier unit channel to 930 MHz (right)...');
% Sw: Sweep frequency band (MHz)
Sw = ucBW*uc;
Fmin = Fc + Fguard;
Fmax = 930;

% Set up according to 2(5) page 15
%Spec_an_setup(gpib,Fmin,Fmax,RBW, VBW,datap,contsweep,detmode,mxhld, reflvl, atten)
Spec_an_setup(dev, Fmin, Fmax, RBW, RBW, datap, 'OFF', 'POSP', 'ON', peak_lvl, att);
% Read from dev
%             Read_data(gpib, Fmin, Fmax, wait_stable)
[P_y2, P_x2] = Read_data(dev, Fmin, Fmax, cLoss, 'OFF');
% Find crossover frequency where the standard value is crossed
%          Find_cross_freq(y_val, x_val, lvl, ascending);
F2_cross = Find_cross_freq(P_y2, P_x2, std_lvl, 0);
% Needed number of 100kHz cannels to the right.
sp_uc2 = ceil((F2_cross - Fmin)/Sw_sp);

% Compare needed unit channels (to gain symmetry)
sp_uc = max([sp_uc1 sp_uc2]);
disp([num2str(sp_uc), ' 100kHz channels for spurious emission needed per side.']);
disp(' ');
disp([num2str(F1_cross), ' First cross frequency']);
disp([num2str(F2_cross), ' Second cross frequency']);
disp(' ');


%% Pb 4(7) page 18
disp('Average power within burst Pb...');
% Sw: Sweep frequency band (MHz)
Sw = 0;
Fmin = Fc - Sw/2;
Fmax = Fc + Sw/2;
% RBW: Resolution bandwidth (MHz)
RBW = 1;
% Set up according to 2(4) page 15
%Spec_an_setup(gpib,Fmin,Fmax,RBW, VBW,datap,contsweep,detmode,mxhld, reflvl, atten)
Spec_an_setup(dev, Fmin, Fmax, RBW, RBW, datap, 'OFF', 'SAMPL', 'OFF', peak_lvl, att); %mxhold = off?
% Read from dev
%             Read_data(gpib, Fmin, Fmax, wait_stable)
[Pb_y, Pb_x] = Read_data(dev, Fmin, Fmax, cLoss, 'OFF');
% Calculate average
%    Calc_avg(data,Sw, RBW,k)
Pb = Calc_avg(Pb_y, Sw, RBW, 33.1131);

disp(['Average power Pb = ', num2str(Pb), ' mW.']);
disp(' ');
% disp('Please wait...');
% pause(2);


%% Ps 4(12)- page 19
disp('Sweeping spurious unit channels one by one (from left to right)...');
%uc = sp_uc;
% Sw: Sweep frequency band (MHz)
% Sw = ucBW*uc;
% Sweep frequency band is defined as 100kHz for spurious 2(6) page 16
% Note 14: Detune frequency max is (250+100xn)kHz. Given 100kHz Sw the
% closest to the Fc is unit channel Bw + 100kHz. The 100kHz 
Sw = 0.1;
Fmin = Fc - Fguard; 
Fmax = Fc + Fguard;
% sp_Sw: Spurious Sweep frequency band (MHz)
sp_Sw = Sw*sp_uc;
FminPs = Fmin - sp_Sw;
FmaxPs = FminPs + Sw;
% RBW: Resolution bandwidth (MHz)
RBW = 0.003;
% Set up according to 2(6) page 16
Ps_y = [];
Ps_x = [];
k = 1;
for i = 0:(sp_uc-1)
    disp(['Unit channel ', num2str(k), ' of ', num2str(2*sp_uc), '...']);
    k = k + 1;
    %Spec_an_setup(gpib, Fmin,                Fmax,     RBW, VBW, datap, contsweep, detmode, mxhld, reflvl, atten)
    Spec_an_setup(dev, FminPs + Sw*i, FmaxPs + Sw*i, RBW, RBW, datap, 'ON', 'POSP', 'ON', peak_lvl, att);
    % Read from dev
    %               Read_data(gpib, Fmin, Fmax, wait_stable)
    [Ps_y1, Ps_x1] = Read_data(dev, FminPs + Sw*i, FmaxPs + Sw*i, cLoss, 'ON');
    Ps_y = Arrconcat(Ps_y, Ps_y1);
    Ps_x = Arrconcat(Ps_x, Ps_x1);
end
FminPs = Fmax;
FmaxPs = FminPs + Sw;
for i = 0:(sp_uc-1)
    disp(['Unit channel ', num2str(k), ' of ', num2str(2*sp_uc), '...']);
    k = k + 1;
    %Spec_an_setup(gpib, Fmin,                Fmax,     RBW, VBW, datap, contsweep, detmode, mxhld, reflvl, atten)
    Spec_an_setup(dev, FminPs + Sw*i, FmaxPs + Sw*i, RBW, RBW, datap, 'ON', 'POSP', 'ON', peak_lvl, att);
    % Read from dev
    %               Read_data(gpib, Fmin, Fmax, wait_stable)
    [Ps_y1, Ps_x1] = Read_data(dev, FminPs + Sw*i, FmaxPs + Sw*i, cLoss, 'ON');
    Ps_y = Arrconcat(Ps_y, Ps_y1);
    Ps_x = Arrconcat(Ps_x, Ps_x1);
end
% Calculate average
%    Calc_avg(data,Sw,RBW, k)
if sp_uc == 0
    disp(['Ps is under the limit']);
    Ps = 0;
else
    Ps = Calc_avg(Ps_y, Sw, RBW, 33.1131);

    disp(['Unwanted emission Ps = ', num2str(Ps), ' mW.']);
    disp(' ');
    % disp('Please wait...');
    % pause(2);

    %% Equation in 4(16) page 20
    SE_mW = Ps/Pc*Pb;
    % Convert from milliwatts to dBm
    SE_dBm = 10*log10(SE_mW);
    disp(['Unwanted emission (Ps/Pc)*Pb = ', num2str(SE_dBm), ' dBm.']);
    disp(' ');
    
    disp('Generating plot...');
    Sp_plot( Fc, Pc_x, Pc_y, Ps_x, Ps_y, std_lvl, uc, 2*sp_uc, Pb, Pc, Ps, SE_mW );
    Sp_plot2( Pc_x, Pc_y, P_x1, P_y1, P_x2, P_y2, std_lvl );
    Sp_plot3( Fc, Pc_x, Pc_y, Ps_x, Ps_y, std_lvl, uc, 2*sp_uc, Pb, Pc, Ps, SE_mW, P_x1, P_y1, P_x2, P_y2 );
end
disp('Spurious emission measurement done.');

dev.close;