//*****************************************************************************
//  MSP430G2131 / DRV8837 Customer EVM Demo - Syncronous Rectification
//  
//  Description: This program operates MSP430 normally in LPM0 with WDT ISR 
//  used for system wake-up. Using DRV8837 Customer EVM PWM_SEL_IN open selects 
//  forward, closed selects reverse direction. Timer_A CCR1 drives in upmode, 
//  5-bit PWM. PWM DutyCycle is measured using poti on ADC10_A0.  
//  LED is toggled in while()speed relative counting up LEDcounter.
//  Forward Synchronous is defined as; IN1 = 1, IN2 modulated 
//  Reverse Synchronous is defined as; IN2 = 1, IN1 modulated  
//  ACLK = n/a, MCLK = TACLK = SMCLK = 1MHz calibrated DCO
//  PWM frequency = TACLK/32 = 31.250Khz  
//
//  M. Buccini
//  July 27, 2012
//******************************************************************************
#include "msp430g2131.h"

#define LED         BIT1                // P1.1
#define IN_1_PWM    BIT2                // P1.2
#define PWM_SEL_IN  BIT3                // P1.3
#define ADC_VREF    BIT4                // P1.4
#define IN_2_PWM    BIT6                // P2.6

unsigned int LEDcounter = 0;            // Countdown timer for LED toggle
unsigned int DutyCycle;                 // 6-bit PWM duty cycle

int main(void)
{
  WDTCTL = WDT_MDLY_8;                  // Set Watchdog Timer interval to ~30ms
  IE1 |= WDTIE;                         // Enable WDT interrupt
  BCSCTL1 = CALBC1_1MHZ;                // Set DCO to 1MHz
  DCOCTL = CALDCO_1MHZ;
  P1OUT = 0x00;                         // P1.x Reset
  P1DIR = 0xF6;                         // P1.3,0 inputs, else outputs
  P2OUT = 0x00;                         // P2.x reset
  P2SEL = 0x00;                         // P2.x no options
  P2DIR = 0xFF;                         // P2.x outputs
  CCTL1 = OUTMOD_7;                     // CCR1 in set/reset mode
  CCR1 = 0;				// CCR1 Duty cycle to 0%
  CCR0 = 64 - 1;                        // 6-bit PWM period
  TACTL = TASSEL_2 + MC_1;              // SCLK source, up mode  
  ADC10CTL0 = ADC10SHT_2 + ADC10ON;     // VCC ref
  ADC10AE0 |= INCH_0;                   // P1.0 ADC option select    

  while (1){
    _BIS_SR(LPM0_bits + GIE);           // Enter LPM0 with interrupts
    P1OUT |= ADC_VREF;                  // Enable poti divider    
    ADC10CTL0 |= ENC + ADC10SC;         // Sampling and conversion start
    while (ADC10IFG & ADC10CTL0);       // Wait for conversion complete 
    ADC10CTL0 &= ~ADC10IFG;             // Clear converstion flag
    P1OUT &= ~ADC_VREF;                 // Disable poti divider    
    DutyCycle = 63 - (ADC10MEM>>4);     // Mask upper 6-bits
    CCR1 = DutyCycle;                   // PWM duty cycle to DRV

    if (DutyCycle == 63) {              // Coast? 
      P1OUT &= ~IN_1_PWM;               // yes, IN_1 reset 
      P2OUT &= ~IN_2_PWM;               // IN_2 reset
      P1SEL = 0x00;                     // Disable IN_1 PWM
      P2SEL = 0x00;                     // Disable IN_2 PWM
    }
    else if (PWM_SEL_IN & P1IN) {       // Forward direction? 
      P1OUT |= IN_1_PWM;                // yes, IN_1 set 
      P1SEL = 0x00;                     // Disable IN_1 PWM
      P2SEL = IN_2_PWM;                 // Enable IN_2 PWM 
    }
    else {                              // Reverse 
      P2OUT |= IN_2_PWM;                // IN_2 set
      P2SEL = 0x00;                     // Disable IN_2 PWM
      P1SEL = IN_1_PWM;                 // Enable IN_1 PWM 
    }
    LEDcounter++;
    if(LEDcounter > DutyCycle){
      P1OUT ^= LED;                     // Toggle LED
      LEDcounter = 0;                   // Reset LED Counter
    }
  }
}
// Watchdog Timer interrupt service routine
#pragma vector=WDT_VECTOR
__interrupt void watchdog_timer(void)
{
  _BIC_SR_IRQ(LPM3_bits);                 // Clear LPM0 bits from 0(SR)
}
  
  